/**
 * Sistema Simplificado de Leituras
 * Funcionalidades básicas para registro de leituras
 */

class LeituraInteligente {
    constructor() {
        this.estacao = null;
        this.init();
    }

    init() {
        this.carregarDadosEstacao();
        this.configurarEventos();
    }

    // Carregar dados da estação
    carregarDadosEstacao() {
        this.estacao = {
            id: window.estacaoId || null,
            tipo: window.estacaoTipo || 'hidrometrica',
            codigo: window.estacaoCodigo || '',
            nome: window.estacaoNome || '',
            nivelAlerta: window.estacaoNivelAlerta || null
        };
    }

    // Configurar eventos do formulário
    configurarEventos() {
        // Verificação de duplicata
        document.getElementById('data_leitura')?.addEventListener('change', () => {
            this.verificarDuplicata();
        });

        // Configurar validações específicas
        this.configurarValidacoes();
    }

    // Configurar validações específicas
    configurarValidacoes() {
        // Validação de níveis hidrométricos
        ['nivel_6h', 'nivel_12h', 'nivel_18h'].forEach(campo => {
            document.getElementById(campo)?.addEventListener('input', () => {
                this.calcularEstatisticas();
            });
        });
    }

    // Verificar se já existe leitura para a data
    async verificarDuplicata() {
        const data = document.getElementById('data_leitura')?.value;
        if (!data || !this.estacao.id) return;

        try {
            const response = await fetch(`/api/estacoes/${this.estacao.id}/leituras/verificar/${data}`);
            const resultado = await response.json();

            const alerta = document.getElementById('verificacao-duplicata');
            if (alerta) {
                if (resultado.existe) {
                    alerta.classList.remove('hidden');
                } else {
                    alerta.classList.add('hidden');
                }
            }
        } catch (error) {
            console.error('Erro ao verificar duplicata:', error);
        }
    }

    // Calcular estatísticas hidrométricas
    calcularEstatisticas() {
        const nivel6h = parseFloat(document.getElementById('nivel_6h')?.value) || 0;
        const nivel12h = parseFloat(document.getElementById('nivel_12h')?.value) || 0;
        const nivel18h = parseFloat(document.getElementById('nivel_18h')?.value) || 0;

        const niveis = [nivel6h, nivel12h, nivel18h].filter(n => n > 0);

        if (niveis.length > 0) {
            // Nível médio
            const nivelMedio = niveis.reduce((a, b) => a + b, 0) / niveis.length;
            const nivelMedioEl = document.getElementById('nivel_medio');
            if (nivelMedioEl) nivelMedioEl.value = nivelMedio.toFixed(2);

            // Variação
            const nivelMax = Math.max(...niveis);
            const nivelMin = Math.min(...niveis);
            const variacao = nivelMax - nivelMin;
            const variacaoEl = document.getElementById('variacao_nivel');
            if (variacaoEl) variacaoEl.value = variacao.toFixed(2);

            // Caudal estimado (fórmula simplificada)
            const caudal = this.calcularCaudal(nivelMedio);
            const caudalEl = document.getElementById('caudal_estimado');
            if (caudalEl) caudalEl.value = caudal.toFixed(3);
        }
    }

    // Calcular caudal estimado
    calcularCaudal(nivel) {
        // Fórmula simplificada Q = a * h^b (onde h é o nível)
        const a = 2.5; // Coeficiente
        const b = 2.1; // Expoente
        return a * Math.pow(nivel, b);
    }
}

// Funções globais para compatibilidade
let leituraInteligente;

function calcularEstatisticas() {
    if (!leituraInteligente) {
        leituraInteligente = new LeituraInteligente();
    }
    leituraInteligente.calcularEstatisticas();
}

// Inicializar quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', function() {
    if (window.estacaoId) {
        leituraInteligente = new LeituraInteligente();
    }
});