@extends('layouts.app')

@section('title', 'Barragens')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-dam mr-3 text-blue-600"></i>
                    Gestão de Barragens
                </h1>
                <p class="text-gray-600 mt-2">Gerir todas as barragens do sistema hidrológico</p>
            </div>
            <div class="flex space-x-4">
                <a href="{{ route('barragens.create') }}" 
                   class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                    <i class="fas fa-plus mr-2"></i>
                    Nova Barragem
                </a>
                <a href="{{ route('importacao.index') }}" 
                   class="bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition-colors">
                    <i class="fas fa-upload mr-2"></i>
                    Importar Dados
                </a>
            </div>
        </div>
    </div>

    <!-- Filters com Ano Hidrológico -->
    <div class="bg-white rounded-lg shadow">
        <!-- Informações do Ano Hidrológico -->
        <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-blue-50 to-cyan-50">
            @php
                $anoHidrologico = \App\Helpers\HydrologicalYear::getCurrentHydrologicalYear();
                $estacaoAtual = \App\Helpers\HydrologicalYear::getCurrentSeason();
                $progresso = \App\Helpers\HydrologicalYear::getYearProgress();
            @endphp
            <div class="flex items-center justify-between">
                <div>
                    <h3 class="text-lg font-semibold text-gray-900 flex items-center">
                        <i class="fas fa-calendar-alt mr-2 text-blue-600"></i>
                        Ano Hidrológico {{ $anoHidrologico }}/{{ $anoHidrologico + 1 }}
                    </h3>
                    <p class="text-sm text-gray-600 mt-1">
                        {{ $estacaoAtual['label'] }} • {{ $progresso }}% completo
                    </p>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm bg-{{ $estacaoAtual['color'] }}-100 text-{{ $estacaoAtual['color'] }}-800 border border-{{ $estacaoAtual['color'] }}-200">
                        @if($estacaoAtual['season'] == 'chuvosa')
                            <i class="fas fa-cloud-rain mr-2"></i>
                        @else
                            <i class="fas fa-sun mr-2"></i>
                        @endif
                        {{ $estacaoAtual['months'] }}
                    </span>
                </div>
            </div>
        </div>
        
        <!-- Filtros -->
        <div class="p-6">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Buscar</label>
                    <input type="text" name="search" value="{{ request('search') }}" 
                           placeholder="Nome ou código da barragem"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Estado</label>
                    <select name="estado" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">Todos</option>
                        <option value="activa" {{ request('estado') == 'activa' ? 'selected' : '' }}>Ativa</option>
                        <option value="inactiva" {{ request('estado') == 'inactiva' ? 'selected' : '' }}>Inativa</option>
                        <option value="manutencao" {{ request('estado') == 'manutencao' ? 'selected' : '' }}>Manutenção</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Província</label>
                    <input type="text" name="provincia" value="{{ request('provincia') }}" 
                           placeholder="Província"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div class="flex items-end">
                    <button type="submit" class="w-full bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700 transition-colors">
                        <i class="fas fa-search mr-2"></i>
                        Filtrar
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Barragens Table -->
    <div class="bg-white rounded-lg shadow overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-medium text-gray-900">
                Lista de Barragens ({{ $barragens->total() }})
            </h3>
        </div>
        
        @if($barragens->count() > 0)
            <div class="overflow-x-auto">
                <table class="w-full table-fixed divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Barragem & Bacia</th>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Localização & Coordenadas</th>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Capacidade & Características</th>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Monitoramento Atual</th>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Estado & Alertas</th>
                            <th class="w-1/6 px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações & Relatórios</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @foreach($barragens as $barragem)
                        <tr class="hover:bg-gray-50">
                            <td class="w-1/6 px-6 py-4 break-words">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-dam text-blue-600"></i>
                                    </div>
                                    <div class="ml-4">
                                        <div class="text-sm font-medium text-gray-900">
                                            <a href="{{ route('barragens.show', $barragem) }}" 
                                               class="hover:text-blue-600">
                                                {{ $barragem->nome }}
                                            </a>
                                        </div>
                                        <div class="text-sm text-gray-500">{{ $barragem->codigo }}</div>
                                        @if($barragem->bacia_hidrografica)
                                            <div class="text-xs text-blue-600">{{ $barragem->bacia_hidrografica->nome }}</div>
                                        @endif
                                    </div>
                                </div>
                            </td>
                            
                            <td class="w-1/6 px-6 py-4 break-words">
                                <div class="text-sm text-gray-900">
                                    <i class="fas fa-map-marker-alt text-gray-400 mr-1"></i>
                                    {{ $barragem->provincia ?? 'N/A' }}
                                </div>
                                <div class="text-sm text-gray-500">{{ $barragem->distrito ?? 'N/A' }}</div>
                                @if($barragem->latitude && $barragem->longitude)
                                    <div class="text-xs text-gray-400 mt-1">
                                        <i class="fas fa-globe text-gray-400 mr-1"></i>
                                        {{ number_format($barragem->latitude, 4) }}°, {{ number_format($barragem->longitude, 4) }}°
                                    </div>
                                @endif
                                @if($barragem->ano_construcao)
                                    <div class="text-xs text-gray-400 mt-1">
                                        <i class="fas fa-calendar text-gray-400 mr-1"></i>
                                        {{ $barragem->ano_construcao }}
                                    </div>
                                @endif
                            </td>

                            <td class="w-1/6 px-6 py-4 break-words">
                                <div class="space-y-2">
                                    <!-- Volume e Capacidade -->
                                    @if($barragem->volume_maximo)
                                        <div class="flex items-center justify-between">
                                            <div class="text-sm text-gray-900">
                                                <i class="fas fa-tint text-blue-500 mr-1"></i>
                                                <span class="font-medium">{{ number_format($barragem->volume_maximo, 1) }}</span> Mm³
                                            </div>
                                            <div class="text-xs text-gray-500">Cap. Total</div>
                                        </div>
                                    @endif
                                    
                                    <!-- Volume Morto -->
                                    @if($barragem->volume_morto)
                                        <div class="flex items-center justify-between">
                                            <div class="text-xs text-gray-600">
                                                <i class="fas fa-layer-group text-gray-400 mr-1"></i>
                                                {{ number_format($barragem->volume_morto, 1) }} Mm³
                                            </div>
                                            <div class="text-xs text-gray-500">Vol. Morto</div>
                                        </div>
                                    @endif

                                    <!-- Cotas -->
                                    <div class="grid grid-cols-2 gap-1 mt-2">
                                        @if($barragem->cota_maxima)
                                            <div class="text-xs bg-blue-50 px-2 py-1 rounded">
                                                <i class="fas fa-arrow-up text-blue-500 mr-1"></i>
                                                <span class="font-medium">{{ number_format($barragem->cota_maxima, 1) }}m</span>
                                                <div class="text-gray-500">Máx</div>
                                            </div>
                                        @endif
                                        @if($barragem->cota_minima)
                                            <div class="text-xs bg-gray-50 px-2 py-1 rounded">
                                                <i class="fas fa-arrow-down text-gray-500 mr-1"></i>
                                                <span class="font-medium">{{ number_format($barragem->cota_minima, 1) }}m</span>
                                                <div class="text-gray-500">Min</div>
                                            </div>
                                        @endif
                                    </div>

                                    <!-- Níveis de Alerta -->
                                    @if($barragem->nivel_alerta || $barragem->nivel_emergencia)
                                        <div class="space-y-1 mt-2">
                                            @if($barragem->nivel_alerta)
                                                <div class="text-xs text-yellow-700 bg-yellow-100 px-2 py-1 rounded flex items-center">
                                                    <i class="fas fa-exclamation-triangle mr-1"></i>
                                                    Alerta: {{ number_format($barragem->nivel_alerta, 1) }}m
                                                </div>
                                            @endif
                                            @if($barragem->nivel_emergencia)
                                                <div class="text-xs text-red-700 bg-red-100 px-2 py-1 rounded flex items-center">
                                                    <i class="fas fa-exclamation-circle mr-1"></i>
                                                    Emerg.: {{ number_format($barragem->nivel_emergencia, 1) }}m
                                                </div>
                                            @endif
                                        </div>
                                    @endif

                                    <!-- Área da Bacia -->
                                    @if($barragem->area_bacia_hidraulica)
                                        <div class="text-xs text-gray-600 bg-gray-50 px-2 py-1 rounded">
                                            <i class="fas fa-expand-arrows-alt text-gray-400 mr-1"></i>
                                            Bacia: {{ number_format($barragem->area_bacia_hidraulica, 1) }} km²
                                        </div>
                                    @endif
                                </div>
                            </td>
                            
                            <td class="w-1/6 px-6 py-4 break-words">
                                @if($barragem->ultima_leitura)
                                    <div class="space-y-3">
                                        <!-- Data e Hora da Leitura -->
                                        <div class="flex items-center justify-between">
                                            <div class="text-xs font-medium text-gray-900">
                                                <i class="fas fa-clock text-blue-500 mr-1"></i>
                                                {{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}
                                            </div>
                                            <div class="text-xs text-gray-500">
                                                {{ $barragem->ultima_leitura->hora_leitura ?? '06:00' }}
                                            </div>
                                        </div>

                                        <!-- Cota Atual vs Máxima -->
                                        <div class="bg-blue-50 px-3 py-2 rounded">
                                            <div class="flex items-center justify-between mb-1">
                                                <span class="text-sm font-medium text-blue-900">
                                                    <i class="fas fa-ruler-vertical text-blue-600 mr-1"></i>
                                                    {{ number_format($barragem->ultima_leitura->cota_actual, 2) }}m
                                                </span>
                                                @if($barragem->cota_maxima)
                                                    <span class="text-xs text-blue-600">
                                                        /{{ number_format($barragem->cota_maxima, 1) }}m
                                                    </span>
                                                @endif
                                            </div>
                                            <div class="text-xs text-blue-700">Cota Atual</div>
                                        </div>

                                        <!-- Percentagem de Enchimento -->
                                        @if($barragem->ultima_leitura->percentagem_enchimento)
                                            @php
                                                $percentage = min(100, $barragem->ultima_leitura->percentagem_enchimento);
                                                $bgColor = 'bg-blue-600';
                                                $textColor = 'text-blue-900';
                                                $bgLight = 'bg-blue-50';
                                                if ($percentage >= 95) {
                                                    $bgColor = 'bg-red-600'; $textColor = 'text-red-900'; $bgLight = 'bg-red-50';
                                                } elseif ($percentage >= 85) {
                                                    $bgColor = 'bg-orange-500'; $textColor = 'text-orange-900'; $bgLight = 'bg-orange-50';
                                                } elseif ($percentage >= 70) {
                                                    $bgColor = 'bg-yellow-500'; $textColor = 'text-yellow-900'; $bgLight = 'bg-yellow-50';
                                                } elseif ($percentage >= 50) {
                                                    $bgColor = 'bg-green-500'; $textColor = 'text-green-900'; $bgLight = 'bg-green-50';
                                                }
                                            @endphp
                                            <div class="{{ $bgLight }} px-3 py-2 rounded">
                                                <div class="flex items-center justify-between mb-1">
                                                    <span class="text-sm font-medium {{ $textColor }}">
                                                        {{ number_format($percentage, 1) }}%
                                                    </span>
                                                    <i class="fas fa-tint {{ str_replace('text-', 'text-', $textColor) }}"></i>
                                                </div>
                                                <div class="w-full bg-gray-200 rounded-full h-2 mb-1">
                                                    <div class="{{ $bgColor }} h-2 rounded-full transition-all duration-500" 
                                                         style="width: {{ $percentage }}%"></div>
                                                </div>
                                                <div class="text-xs {{ $textColor }}">Enchimento</div>
                                            </div>
                                        @endif

                                        <!-- Volume Atual -->
                                        @if($barragem->ultima_leitura->volume_actual)
                                            <div class="text-xs text-gray-600 bg-gray-50 px-2 py-1 rounded">
                                                <i class="fas fa-cube text-gray-400 mr-1"></i>
                                                <span class="font-medium">{{ number_format($barragem->ultima_leitura->volume_actual, 2) }}</span> Mm³
                                                <span class="text-gray-500 ml-1">atual</span>
                                            </div>
                                        @endif

                                        <!-- Caudais -->
                                        @if($barragem->ultima_leitura->caudal_afluente || $barragem->ultima_leitura->caudal_efluente)
                                            <div class="grid grid-cols-2 gap-1">
                                                @if($barragem->ultima_leitura->caudal_afluente)
                                                    <div class="text-xs bg-green-50 px-2 py-1 rounded">
                                                        <i class="fas fa-arrow-right text-green-500 mr-1"></i>
                                                        <div class="font-medium">{{ number_format($barragem->ultima_leitura->caudal_afluente, 1) }}</div>
                                                        <div class="text-green-600">Afluente</div>
                                                    </div>
                                                @endif
                                                @if($barragem->ultima_leitura->caudal_efluente)
                                                    <div class="text-xs bg-blue-50 px-2 py-1 rounded">
                                                        <i class="fas fa-arrow-left text-blue-500 mr-1"></i>
                                                        <div class="font-medium">{{ number_format($barragem->ultima_leitura->caudal_efluente, 1) }}</div>
                                                        <div class="text-blue-600">Efluente</div>
                                                    </div>
                                                @endif
                                            </div>
                                        @endif

                                        <!-- Método de Leitura -->
                                        <div class="text-xs text-gray-500 border-t pt-2">
                                            <i class="fas fa-satellite text-gray-400 mr-1"></i>
                                            {{ ucfirst($barragem->ultima_leitura->metodo_leitura ?? 'Manual') }}
                                            @if($barragem->ultima_leitura->validado)
                                                <i class="fas fa-check-circle text-green-500 ml-2" title="Validado"></i>
                                            @else
                                                <i class="fas fa-clock text-orange-500 ml-2" title="Pendente validação"></i>
                                            @endif
                                        </div>
                                    </div>
                                @else
                                    <div class="text-center py-4">
                                        <div class="w-12 h-12 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-2">
                                            <i class="fas fa-database text-2xl text-gray-400"></i>
                                        </div>
                                        <div class="text-sm font-medium text-gray-500 mb-1">Sem dados</div>
                                        <div class="text-xs text-gray-400">Nenhuma leitura registada</div>
                                    </div>
                                @endif
                            </td>
                            
                            <td class="w-1/6 px-6 py-4 break-words">
                                @php
                                    $estadoConfig = [
                                        'activa' => [
                                            'class' => 'bg-green-100 text-green-800 border-green-200', 
                                            'icon' => 'fas fa-circle-check',
                                            'text' => 'Ativa'
                                        ],
                                        'inactiva' => [
                                            'class' => 'bg-red-100 text-red-800 border-red-200', 
                                            'icon' => 'fas fa-circle-xmark',
                                            'text' => 'Inativa'
                                        ],
                                        'manutencao' => [
                                            'class' => 'bg-yellow-100 text-yellow-800 border-yellow-200', 
                                            'icon' => 'fas fa-screwdriver-wrench',
                                            'text' => 'Manutenção'
                                        ]
                                    ];
                                    $config = $estadoConfig[$barragem->estado] ?? [
                                        'class' => 'bg-gray-100 text-gray-800 border-gray-200', 
                                        'icon' => 'fas fa-question-circle',
                                        'text' => ucfirst($barragem->estado)
                                    ];
                                @endphp
                                <div class="space-y-2">
                                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium border {{ $config['class'] }} shadow-sm">
                                        <i class="{{ $config['icon'] }} mr-2 text-sm"></i>
                                        {{ $config['text'] }}
                                    </span>
                                    
                                    @if($barragem->ultima_leitura && $barragem->nivel_alerta && $barragem->ultima_leitura->cota_actual >= $barragem->nivel_alerta)
                                        @if($barragem->nivel_emergencia && $barragem->ultima_leitura->cota_actual >= $barragem->nivel_emergencia)
                                            <div class="mt-1">
                                                <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800 border border-red-200 animate-pulse">
                                                    <i class="fas fa-exclamation-circle mr-1"></i>
                                                    EMERGÊNCIA
                                                </span>
                                            </div>
                                        @else
                                            <div class="mt-1">
                                                <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800 border border-yellow-200">
                                                    <i class="fas fa-exclamation-triangle mr-1"></i>
                                                    ALERTA
                                                </span>
                                            </div>
                                        @endif
                                    @endif
                                </div>
                            </td>
                            
                            <td class="px-4 py-3 whitespace-nowrap text-sm font-medium">
                                <div class="space-y-2">
                                    <!-- Ações Principais -->
                                    <div class="flex items-center space-x-1">
                                        <a href="{{ route('barragens.show', $barragem) }}" 
                                           class="inline-flex items-center px-2 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-blue-600 hover:bg-blue-700 transition-colors"
                                           title="Visualizar detalhes">
                                            <i class="fas fa-eye mr-1"></i>
                                            Ver
                                        </a>
                                        <a href="{{ route('barragens.edit', $barragem) }}" 
                                           class="inline-flex items-center px-2 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-green-600 hover:bg-green-700 transition-colors"
                                           title="Editar">
                                            <i class="fas fa-edit mr-1"></i>
                                            Editar
                                        </a>
                                    </div>

                                    <!-- Ações de Dados -->
                                    <div class="flex items-center space-x-1">
                                        <a href="{{ route('barragens.leituras', $barragem) }}" 
                                           class="inline-flex items-center px-2 py-1 border border-blue-300 text-xs leading-4 font-medium rounded text-blue-700 bg-blue-50 hover:bg-blue-100 transition-colors"
                                           title="Ver leituras">
                                            <i class="fas fa-chart-line mr-1"></i>
                                            Leituras
                                        </a>
                                        <a href="#" 
                                           class="inline-flex items-center px-2 py-1 border border-purple-300 text-xs leading-4 font-medium rounded text-purple-700 bg-purple-50 hover:bg-purple-100 transition-colors"
                                           title="Gerar boletim">
                                            <i class="fas fa-file-alt mr-1"></i>
                                            Boletim
                                        </a>
                                    </div>
                                    
                                    <!-- Nova Leitura - Botão Destacado -->
                                    <div class="flex items-center mt-2">
                                        <a href="{{ route('barragens.leituras', $barragem) }}" 
                                           class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-bold rounded-md text-white bg-green-500 hover:bg-green-600 transition-colors w-full justify-center shadow-md"
                                           title="Adicionar nova leitura">
                                            <i class="fas fa-plus mr-2"></i>
                                            Nova Leitura
                                        </a>
                                    </div>

                                    <!-- Status de Última Atualização -->
                                    @if($barragem->ultima_leitura)
                                        @php
                                            $ultimaAtualizacao = $barragem->ultima_leitura->data_leitura;
                                            $diasSemAtualizacao = $ultimaAtualizacao->diffInDays(now());
                                            $statusClass = 'bg-green-100 text-green-800';
                                            $statusIcon = 'fas fa-check-circle';
                                            $statusText = 'Atualizado';
                                            
                                            if ($diasSemAtualizacao > 7) {
                                                $statusClass = 'bg-red-100 text-red-800';
                                                $statusIcon = 'fas fa-exclamation-circle';
                                                $statusText = 'Desatualizado';
                                            } elseif ($diasSemAtualizacao > 3) {
                                                $statusClass = 'bg-yellow-100 text-yellow-800';
                                                $statusIcon = 'fas fa-clock';
                                                $statusText = 'Atrasado';
                                            }
                                        @endphp
                                        <div class="text-xs {{ $statusClass }} px-2 py-1 rounded flex items-center">
                                            <i class="{{ $statusIcon }} mr-1"></i>
                                            {{ $statusText }}
                                            @if($diasSemAtualizacao > 0)
                                                <span class="ml-1">({{ $diasSemAtualizacao }}d)</span>
                                            @endif
                                        </div>
                                    @else
                                        <div class="text-xs bg-gray-100 text-gray-800 px-2 py-1 rounded flex items-center">
                                            <i class="fas fa-minus-circle mr-1"></i>
                                            Sem dados
                                        </div>
                                    @endif

                                    <!-- Dropdown para mais ações -->
                                    <div class="relative">
                                        <button onclick="toggleDropdown('actions-{{ $barragem->id }}')" 
                                                class="inline-flex items-center px-2 py-1 border border-gray-300 text-xs leading-4 font-medium rounded text-gray-700 bg-white hover:bg-gray-50 transition-colors w-full justify-center">
                                            <i class="fas fa-ellipsis-h mr-1"></i>
                                            Mais
                                        </button>
                                        <div id="actions-{{ $barragem->id }}" class="hidden absolute right-0 mt-1 w-48 bg-white rounded-md shadow-lg z-10 border">
                                            <div class="py-1">
                                                <a href="#" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-download mr-2"></i>
                                                    Exportar dados
                                                </a>
                                                <a href="#" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-chart-bar mr-2"></i>
                                                    Relatório mensal
                                                </a>
                                                <a href="#" class="block px-4 py-2 text-xs text-gray-700 hover:bg-gray-100">
                                                    <i class="fas fa-bell mr-2"></i>
                                                    Configurar alertas
                                                </a>
                                                <div class="border-t border-gray-100">
                                                    <form method="POST" action="{{ route('barragens.destroy', $barragem) }}" 
                                                          class="inline w-full" onsubmit="return confirm('Tem certeza que deseja excluir esta barragem?')">
                                                        @csrf
                                                        @method('DELETE')
                                                        <button type="submit" class="block w-full text-left px-4 py-2 text-xs text-red-700 hover:bg-red-50">
                                                            <i class="fas fa-trash mr-2"></i>
                                                            Excluir barragem
                                                        </button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            @if($barragens->hasPages())
            <div class="px-6 py-4 border-t border-gray-200">
                {{ $barragens->withQueryString()->links() }}
            </div>
            @endif
        @else
            <div class="text-center py-12">
                <div class="w-24 h-24 bg-gradient-to-br from-blue-100 to-blue-200 rounded-full flex items-center justify-center mx-auto mb-4 shadow-lg">
                    <i class="fas fa-dam text-4xl text-blue-600"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-900 mb-2">Nenhuma barragem encontrada</h3>
                <p class="text-gray-500 mb-6 max-w-md mx-auto">
                    @if(request()->filled('search') || request()->filled('estado') || request()->filled('provincia'))
                        Nenhuma barragem corresponde aos filtros aplicados. Tente ajustar os critérios de busca.
                    @else
                        Comece adicionando sua primeira barragem ao sistema para começar o monitoramento hidrológico.
                    @endif
                </p>
                <div class="flex flex-col sm:flex-row gap-3 justify-center">
                    @if(request()->filled('search') || request()->filled('estado') || request()->filled('provincia'))
                        <a href="{{ route('barragens.index') }}" 
                           class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 transition-colors">
                            <i class="fas fa-times mr-2"></i>
                            Limpar Filtros
                        </a>
                    @endif
                    <a href="{{ route('barragens.create') }}" 
                       class="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 transition-colors">
                        <i class="fas fa-plus mr-2"></i>
                        Adicionar Barragem
                    </a>
                </div>
            </div>
        @endif
    </div>
</div>

<style>
@keyframes pulse {
    0%, 100% {
        opacity: 1;
    }
    50% {
        opacity: 0.5;
    }
}

.animate-pulse {
    animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
}

/* Enhanced progress bar styling */
.progress-enhanced {
    background: linear-gradient(90deg, #f3f4f6 0%, #e5e7eb 100%);
    border-radius: 0.375rem;
    position: relative;
    overflow: hidden;
}

.progress-enhanced::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    animation: shimmer 1.5s infinite;
}

@keyframes shimmer {
    0% {
        left: -100%;
    }
    100% {
        left: 100%;
    }
}

/* Responsive table */
@media (max-width: 768px) {
    .overflow-x-auto {
        -webkit-overflow-scrolling: touch;
    }
    
    table {
        min-width: 800px;
    }
}

/* Hover effects */
.hover\\:shadow-lg:hover {
    box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
}

/* Status badges with enhanced styling */
.status-badge {
    position: relative;
    overflow: hidden;
}

.status-badge::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(45deg, rgba(255,255,255,0.1) 25%, transparent 25%), 
                linear-gradient(-45deg, rgba(255,255,255,0.1) 25%, transparent 25%), 
                linear-gradient(45deg, transparent 75%, rgba(255,255,255,0.1) 75%), 
                linear-gradient(-45deg, transparent 75%, rgba(255,255,255,0.1) 75%);
    background-size: 8px 8px;
    background-position: 0 0, 0 4px, 4px -4px, -4px 0px;
}
</style>

<script>
function toggleDropdown(dropdownId) {
    const dropdown = document.getElementById(dropdownId);
    const isHidden = dropdown.classList.contains('hidden');
    
    // Fechar todos os outros dropdowns
    document.querySelectorAll('[id^="actions-"]').forEach(el => {
        if (el.id !== dropdownId) {
            el.classList.add('hidden');
        }
    });
    
    // Toggle do dropdown atual
    if (isHidden) {
        dropdown.classList.remove('hidden');
    } else {
        dropdown.classList.add('hidden');
    }
}

// Fechar dropdown ao clicar fora
document.addEventListener('click', function(event) {
    if (!event.target.closest('[onclick*="toggleDropdown"]') && !event.target.closest('[id^="actions-"]')) {
        document.querySelectorAll('[id^="actions-"]').forEach(el => {
            el.classList.add('hidden');
        });
    }
});
</script>

@endsection