@extends('layouts.app')

@section('title', 'Nova Leitura - ' . $estacao->nome)

@push('styles')
<link rel="stylesheet" href="{{ asset('css/leituras.css') }}">
@endpush

@section('content')
<div class="space-y-6">
    <!-- Breadcrumb e Header -->
    <div class="bg-white rounded-lg shadow">
        <!-- Breadcrumb -->
        <div class="px-6 py-3 border-b border-gray-200 bg-gray-50">
            <nav class="flex" aria-label="Breadcrumb">
                <ol class="inline-flex items-center space-x-2 md:space-x-4">
                    <li class="inline-flex items-center">
                        <a href="{{ route('dashboard') }}" class="text-gray-500 hover:text-gray-700">
                            <i class="fas fa-home mr-2"></i>
                            Dashboard
                        </a>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('estacoes.index') }}" class="text-gray-500 hover:text-gray-700">Estações</a>
                        </div>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('estacoes.show', $estacao) }}" class="text-gray-500 hover:text-gray-700">{{ $estacao->nome }}</a>
                        </div>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('estacoes.leituras', $estacao) }}" class="text-gray-500 hover:text-gray-700">Leituras</a>
                        </div>
                    </li>
                    <li aria-current="page">
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <span class="text-gray-800 font-medium">Nova Leitura</span>
                        </div>
                    </li>
                </ol>
            </nav>
        </div>

        <!-- Header Principal -->
        <div class="px-6 py-6">
            <div class="flex flex-col lg:flex-row lg:items-center lg:justify-between">
                <div class="flex-1">
                    <div class="flex items-center mb-2">
                        <div class="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-lg flex items-center justify-center mr-4">
                            <i class="fas fa-plus text-white text-xl"></i>
                        </div>
                        <div>
                            <h1 class="text-2xl font-bold text-gray-900">Nova Leitura</h1>
                            <p class="text-gray-600">{{ $estacao->nome }} • Código: {{ $estacao->codigo }}</p>
                        </div>
                    </div>

                    <div class="flex flex-wrap items-center gap-4 mt-4">
                        <div class="flex items-center bg-blue-50 px-3 py-1 rounded-full">
                            <i class="fas fa-satellite-dish text-blue-600 mr-2"></i>
                            <span class="text-sm font-medium text-blue-800">{{ ucfirst($estacao->tipo) }}</span>
                        </div>

                        <div class="flex items-center bg-green-50 px-3 py-1 rounded-full">
                            <i class="fas fa-map-marker-alt text-green-600 mr-2"></i>
                            <span class="text-sm font-medium text-green-800">{{ $estacao->provincia ?? 'N/A' }}</span>
                        </div>

                        <div class="flex items-center
                            @if($estacao->estado == 'activa') bg-green-50
                            @elseif($estacao->estado == 'inactiva') bg-red-50
                            @else bg-yellow-50 @endif px-3 py-1 rounded-full">
                            <i class="fas fa-circle
                                @if($estacao->estado == 'activa') text-green-600
                                @elseif($estacao->estado == 'inactiva') text-red-600
                                @else text-yellow-600 @endif mr-2"></i>
                            <span class="text-sm font-medium
                                @if($estacao->estado == 'activa') text-green-800
                                @elseif($estacao->estado == 'inactiva') text-red-800
                                @else text-yellow-800 @endif">{{ ucfirst($estacao->estado) }}</span>
                        </div>
                    </div>
                </div>

                <div class="mt-4 lg:mt-0">
                    <a href="{{ route('estacoes.leituras', $estacao) }}"
                       class="bg-gray-600 hover:bg-gray-700 text-white font-semibold px-6 py-3 rounded-lg shadow-md hover:shadow-lg transition-all duration-200 inline-flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Voltar às Leituras
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulário Principal -->
    <div class="bg-white rounded-lg shadow">
        <form method="POST" action="{{ route('estacoes.storeLeitura', $estacao) }}" id="leituraForm" enctype="multipart/form-data">
            @csrf

            <!-- Formulário Principal -->
            <div class="p-6 space-y-6">

                    <!-- Informações Básicas -->
                    <div class="bg-blue-50 rounded-lg p-4">
                        <h4 class="font-semibold text-blue-900 mb-3 flex items-center">
                            <i class="fas fa-info-circle mr-2"></i>
                            Informações da Leitura
                        </h4>
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <i class="fas fa-calendar mr-1"></i>
                                    Data da Leitura *
                                </label>
                                <input type="date" name="data_leitura" id="data_leitura" required
                                       value="{{ date('Y-m-d') }}"
                                       max="{{ date('Y-m-d') }}"
                                       class="w-full border-gray-300 rounded-lg shadow-sm focus:border-green-500 focus:ring-green-500">
                                <small class="text-gray-500">Máximo: hoje</small>
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <i class="fas fa-user mr-1"></i>
                                    Operador
                                </label>
                                <input type="text" value="{{ Auth::user()->name }}" readonly
                                       class="w-full border-gray-300 rounded-lg shadow-sm bg-gray-50">
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    <i class="fas fa-clock mr-1"></i>
                                    Hora de Registro
                                </label>
                                <input type="time" name="hora_registro" value="{{ date('H:i') }}"
                                       class="w-full border-gray-300 rounded-lg shadow-sm focus:border-green-500 focus:ring-green-500">
                            </div>
                        </div>
                    </div>

                    <!-- Verificação de Duplicatas -->
                    <div id="verificacao-duplicata" class="hidden bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-triangle text-yellow-600 mr-3"></i>
                            <div>
                                <h4 class="font-medium text-yellow-800">Leitura já existe para esta data</h4>
                                <p class="text-sm text-yellow-700">Deseja sobrescrever os dados existentes?</p>
                            </div>
                        </div>
                    </div>

                    <!-- Dados Hidrométricos -->
                    @if($estacao->tipo == 'hidrometrica')
                    <div class="bg-gradient-to-br from-blue-50 via-indigo-50 to-cyan-50 rounded-xl shadow-lg border-2 border-blue-200 p-6">
                        <div class="flex items-center justify-between mb-6">
                            <div class="flex items-center">
                                <div class="bg-blue-500 p-3 rounded-lg mr-4">
                                    <i class="fas fa-water text-white text-lg"></i>
                                </div>
                                <div>
                                    <h4 class="font-bold text-blue-900 text-lg">Leituras Hidrométricas</h4>
                                    <p class="text-blue-600 text-sm">Registre os níveis de água nos horários especificados</p>
                                </div>
                            </div>
                            <div class="bg-white px-3 py-1 rounded-full border border-blue-200">
                                <span class="text-xs font-medium text-blue-700">5 Horários</span>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-5 gap-6 mb-8">
                            <!-- Leitura 6h -->
                            <div class="bg-white rounded-xl shadow-lg border-4 border-orange-400 hover:border-orange-600 transition-all duration-200 hover:shadow-xl overflow-hidden">
                                <!-- Header do Horário -->
                                <div class="bg-gradient-to-r from-orange-700 to-orange-800 px-4 py-4 border-b border-orange-900">
                                    <div class="flex items-center justify-between">
                                        <h5 class="font-bold text-white flex items-center text-lg" style="text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
                                            <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center mr-3 shadow-lg ring-4 ring-yellow-300 ring-opacity-50">
                                                <i class="fas fa-sun text-orange-600 text-lg"></i>
                                            </div>
                                            06:00h
                                        </h5>
                                        <span class="text-xs text-orange-100 bg-orange-800 bg-opacity-60 px-3 py-1 rounded-full font-bold shadow-lg border-2 border-yellow-200">MANHÃ</span>
                                    </div>
                                </div>

                                <!-- Conteúdo dos Campos -->
                                <div class="p-5 space-y-5">
                                    <!-- Hora Exata -->
                                    <div class="bg-gray-50 rounded-lg p-4 border-2 border-gray-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-clock text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Hora Exata
                                            </label>
                                        </div>
                                        <input type="time" name="hora_6h" value="06:00"
                                               class="w-full border-2 border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white">
                                    </div>

                                    <!-- Nível de Água -->
                                    <div class="bg-blue-50 rounded-lg p-4 border-2 border-blue-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-tint text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Nível de Água
                                            </label>
                                        </div>

                                        <input type="number" step="0.001" name="nivel_6h" id="nivel_6h"
                                               placeholder="Ex: 2.450"
                                               class="w-full border-2 border-blue-300 rounded-lg shadow-sm focus:border-blue-600 focus:ring-2 focus:ring-blue-300 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white mb-3"
                                               onchange="calcularEstatisticas()">

                                        <!-- Sem Dados -->
                                        <div class="bg-white rounded-lg p-3 border-2 border-red-200">
                                            <label class="flex items-center text-sm text-red-700 cursor-pointer hover:text-red-800 transition-colors duration-200">
                                                <input type="checkbox" name="campos_sem_dados[]" value="nivel_6h" class="mr-3 w-4 h-4 rounded border-2 border-red-400 text-red-600 focus:ring-red-300 focus:ring-2">
                                                <div class="flex items-center">
                                                    <i class="fas fa-ban mr-2 text-red-600"></i>
                                                    <span class="font-bold">Sem Dados (F)</span>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Leitura 9h -->
                            <div class="bg-white rounded-xl shadow-lg border-4 border-yellow-400 hover:border-yellow-600 transition-all duration-200 hover:shadow-xl overflow-hidden">
                                <!-- Header do Horário -->
                                <div class="bg-gradient-to-r from-yellow-700 to-yellow-800 px-4 py-4 border-b border-yellow-900">
                                    <div class="flex items-center justify-between">
                                        <h5 class="font-bold text-white flex items-center text-lg" style="text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
                                            <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center mr-3 shadow-lg ring-4 ring-amber-300 ring-opacity-50">
                                                <i class="fas fa-sun text-yellow-600 text-lg"></i>
                                            </div>
                                            09:00h
                                        </h5>
                                        <span class="text-sm text-yellow-100 bg-yellow-800 bg-opacity-60 px-4 py-2 rounded-full font-bold shadow-lg border-2 border-amber-200">MANHÃ</span>
                                    </div>
                                </div>

                                <!-- Conteúdo dos Campos -->
                                <div class="p-5 space-y-5">
                                    <!-- Hora Exata -->
                                    <div class="bg-gray-50 rounded-lg p-4 border-2 border-gray-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-clock text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Hora Exata
                                            </label>
                                        </div>
                                        <input type="time" name="hora_9h" value="09:00"
                                               class="w-full border-2 border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white">
                                    </div>

                                    <!-- Nível de Água -->
                                    <div class="bg-blue-50 rounded-lg p-4 border-2 border-blue-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-tint text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Nível de Água
                                            </label>
                                        </div>

                                        <input type="number" step="0.001" name="nivel_9h" id="nivel_9h"
                                               placeholder="Ex: 2.460"
                                               class="w-full border-2 border-blue-300 rounded-lg shadow-sm focus:border-blue-600 focus:ring-2 focus:ring-blue-300 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white mb-3"
                                               onchange="calcularEstatisticas()">

                                        <!-- Sem Dados -->
                                        <div class="bg-white rounded-lg p-3 border-2 border-red-200">
                                            <label class="flex items-center text-sm text-red-700 cursor-pointer hover:text-red-800 transition-colors duration-200">
                                                <input type="checkbox" name="campos_sem_dados[]" value="nivel_9h" class="mr-3 w-4 h-4 rounded border-2 border-red-400 text-red-600 focus:ring-red-300 focus:ring-2">
                                                <div class="flex items-center">
                                                    <i class="fas fa-ban mr-2 text-red-600"></i>
                                                    <span class="font-bold">Sem Dados (F)</span>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Leitura 12h -->
                            <div class="bg-white rounded-xl shadow-lg border-4 border-amber-400 hover:border-amber-600 transition-all duration-200 hover:shadow-xl overflow-hidden">
                                <!-- Header do Horário -->
                                <div class="bg-gradient-to-r from-amber-700 to-amber-800 px-4 py-4 border-b border-amber-900">
                                    <div class="flex items-center justify-between">
                                        <h5 class="font-bold text-white flex items-center text-lg" style="text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
                                            <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center mr-3 shadow-lg ring-4 ring-yellow-300 ring-opacity-70">
                                                <i class="fas fa-sun text-amber-600 text-lg"></i>
                                            </div>
                                            12:00h
                                        </h5>
                                        <span class="text-xs text-amber-100 bg-amber-800 bg-opacity-60 px-3 py-1 rounded-full font-bold shadow-lg border-2 border-yellow-200">MEIO-DIA</span>
                                    </div>
                                </div>

                                <!-- Conteúdo dos Campos -->
                                <div class="p-5 space-y-5">
                                    <!-- Hora Exata -->
                                    <div class="bg-gray-50 rounded-lg p-4 border-2 border-gray-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-clock text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Hora Exata
                                            </label>
                                        </div>
                                        <input type="time" name="hora_12h" value="12:00"
                                               class="w-full border-2 border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white">
                                    </div>

                                    <!-- Nível de Água -->
                                    <div class="bg-blue-50 rounded-lg p-4 border-2 border-blue-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-tint text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Nível de Água
                                            </label>
                                        </div>

                                        <input type="number" step="0.001" name="nivel_12h" id="nivel_12h"
                                               placeholder="Ex: 2.470"
                                               class="w-full border-2 border-blue-300 rounded-lg shadow-sm focus:border-blue-600 focus:ring-2 focus:ring-blue-300 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white mb-3"
                                               onchange="calcularEstatisticas()">

                                        <!-- Sem Dados -->
                                        <div class="bg-white rounded-lg p-3 border-2 border-red-200">
                                            <label class="flex items-center text-sm text-red-700 cursor-pointer hover:text-red-800 transition-colors duration-200">
                                                <input type="checkbox" name="campos_sem_dados[]" value="nivel_12h" class="mr-3 w-4 h-4 rounded border-2 border-red-400 text-red-600 focus:ring-red-300 focus:ring-2">
                                                <div class="flex items-center">
                                                    <i class="fas fa-ban mr-2 text-red-600"></i>
                                                    <span class="font-bold">Sem Dados (F)</span>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Leitura 15h -->
                            <div class="bg-white rounded-xl shadow-lg border-4 border-red-400 hover:border-red-600 transition-all duration-200 hover:shadow-xl overflow-hidden">
                                <!-- Header do Horário -->
                                <div class="bg-gradient-to-r from-red-700 to-red-800 px-4 py-4 border-b border-red-900">
                                    <div class="flex items-center justify-between">
                                        <h5 class="font-bold text-white flex items-center text-lg" style="text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
                                            <div class="w-12 h-12 bg-white rounded-full flex items-center justify-center mr-3 shadow-lg ring-4 ring-orange-300 ring-opacity-50">
                                                <i class="fas fa-sun text-red-600 text-lg"></i>
                                            </div>
                                            15:00h
                                        </h5>
                                        <span class="text-xs text-red-100 bg-red-800 bg-opacity-60 px-3 py-1 rounded-full font-bold shadow-lg border-2 border-orange-200">TARDE</span>
                                    </div>
                                </div>

                                <!-- Conteúdo dos Campos -->
                                <div class="p-5 space-y-5">
                                    <!-- Hora Exata -->
                                    <div class="bg-gray-50 rounded-lg p-4 border-2 border-gray-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-clock text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Hora Exata
                                            </label>
                                        </div>
                                        <input type="time" name="hora_15h" value="15:00"
                                               class="w-full border-2 border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white">
                                    </div>

                                    <!-- Nível de Água -->
                                    <div class="bg-blue-50 rounded-lg p-4 border-2 border-blue-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-tint text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Nível de Água
                                            </label>
                                        </div>

                                        <input type="number" step="0.001" name="nivel_15h" id="nivel_15h"
                                               placeholder="Ex: 2.440"
                                               class="w-full border-2 border-blue-300 rounded-lg shadow-sm focus:border-blue-600 focus:ring-2 focus:ring-blue-300 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white mb-3"
                                               onchange="calcularEstatisticas()">

                                        <!-- Sem Dados -->
                                        <div class="bg-white rounded-lg p-3 border-2 border-red-200">
                                            <label class="flex items-center text-sm text-red-700 cursor-pointer hover:text-red-800 transition-colors duration-200">
                                                <input type="checkbox" name="campos_sem_dados[]" value="nivel_15h" class="mr-3 w-4 h-4 rounded border-2 border-red-400 text-red-600 focus:ring-red-300 focus:ring-2">
                                                <div class="flex items-center">
                                                    <i class="fas fa-ban mr-2 text-red-600"></i>
                                                    <span class="font-bold">Sem Dados (F)</span>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Leitura 18h -->
                            <div class="bg-white rounded-xl shadow-lg border-4 border-purple-400 hover:border-purple-600 transition-all duration-200 hover:shadow-xl overflow-hidden">
                                <!-- Header do Horário -->
                                <div class="bg-gradient-to-r from-purple-700 to-purple-800 px-4 py-4 border-b border-purple-900">
                                    <div class="flex items-center justify-between">
                                        <h5 class="font-bold text-white flex items-center text-lg" style="text-shadow: 1px 1px 3px rgba(0,0,0,0.7);">
                                            <div class="w-12 h-12 bg-yellow-400 rounded-full flex items-center justify-center mr-3 shadow-lg ring-4 ring-white ring-opacity-30">
                                                <i class="fas fa-moon text-purple-900 text-lg"></i>
                                            </div>
                                            18:00h
                                        </h5>
                                        <span class="text-xs text-yellow-200 bg-purple-800 bg-opacity-60 px-3 py-1 rounded-full font-bold shadow-lg border-2 border-yellow-300">ENTARDECER</span>
                                    </div>
                                </div>

                                <!-- Conteúdo dos Campos -->
                                <div class="p-5 space-y-5">
                                    <!-- Hora Exata -->
                                    <div class="bg-gray-50 rounded-lg p-4 border-2 border-gray-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-gray-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-clock text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Hora Exata
                                            </label>
                                        </div>
                                        <input type="time" name="hora_18h" value="18:00"
                                               class="w-full border-2 border-gray-300 rounded-lg shadow-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white">
                                    </div>

                                    <!-- Nível de Água -->
                                    <div class="bg-blue-50 rounded-lg p-4 border-2 border-blue-200">
                                        <div class="flex items-center mb-3">
                                            <div class="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center mr-3">
                                                <i class="fas fa-tint text-white text-sm"></i>
                                            </div>
                                            <label class="text-xs font-bold text-gray-800 uppercase tracking-wider">
                                                Nível de Água
                                            </label>
                                        </div>

                                        <input type="number" step="0.001" name="nivel_18h" id="nivel_18h"
                                               placeholder="Ex: 2.430"
                                               class="w-full border-2 border-blue-300 rounded-lg shadow-sm focus:border-blue-600 focus:ring-2 focus:ring-blue-300 text-sm font-bold transition-all duration-200 px-3 py-2 bg-white mb-3"
                                               onchange="calcularEstatisticas()">

                                        <!-- Sem Dados -->
                                        <div class="bg-white rounded-lg p-3 border-2 border-red-200">
                                            <label class="flex items-center text-sm text-red-700 cursor-pointer hover:text-red-800 transition-colors duration-200">
                                                <input type="checkbox" name="campos_sem_dados[]" value="nivel_18h" class="mr-3 w-4 h-4 rounded border-2 border-red-400 text-red-600 focus:ring-red-300 focus:ring-2">
                                                <div class="flex items-center">
                                                    <i class="fas fa-ban mr-2 text-red-600"></i>
                                                    <span class="font-bold">Sem Dados (F)</span>
                                                </div>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Cálculos Automáticos -->
                        <div class="bg-gradient-to-r from-emerald-50 to-teal-50 rounded-xl p-6 border-2 border-dashed border-emerald-300 shadow-md">
                            <div class="flex items-center mb-4">
                                <div class="bg-emerald-500 p-2 rounded-lg mr-3">
                                    <i class="fas fa-calculator text-white"></i>
                                </div>
                                <div>
                                    <h5 class="font-bold text-emerald-900">Cálculos Automáticos</h5>
                                    <p class="text-emerald-600 text-xs">Valores calculados automaticamente com base nas leituras</p>
                                </div>
                            </div>
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                <div class="bg-white rounded-xl p-5 border-2 border-emerald-200 shadow-sm hover:shadow-md transition-shadow duration-200">
                                    <label class="block text-sm font-semibold text-gray-700 mb-2 flex items-center">
                                        <i class="fas fa-chart-line text-emerald-500 mr-2"></i>
                                        Nível Médio
                                    </label>
                                    <div class="relative">
                                        <input type="number" step="0.001" id="nivel_medio" readonly
                                               class="w-full border-2 border-gray-200 rounded-lg shadow-sm bg-gradient-to-r from-gray-50 to-gray-100 text-lg font-bold text-gray-700 pl-3 pr-8 py-2">
                                        <span class="absolute right-3 top-2 text-sm text-gray-500">m</span>
                                    </div>
                                </div>
                                <div class="bg-white rounded-xl p-5 border-2 border-emerald-200 shadow-sm hover:shadow-md transition-shadow duration-200">
                                    <label class="block text-sm font-semibold text-gray-700 mb-2 flex items-center">
                                        <i class="fas fa-arrows-alt-v text-amber-500 mr-2"></i>
                                        Variação
                                    </label>
                                    <div class="relative">
                                        <input type="number" step="0.001" id="variacao_nivel" readonly
                                               class="w-full border-2 border-gray-200 rounded-lg shadow-sm bg-gradient-to-r from-gray-50 to-gray-100 text-lg font-bold text-gray-700 pl-3 pr-8 py-2">
                                        <span class="absolute right-3 top-2 text-sm text-gray-500">m</span>
                                    </div>
                                </div>
                                <div class="bg-white rounded-xl p-5 border-2 border-emerald-200 shadow-sm hover:shadow-md transition-shadow duration-200">
                                    <label class="block text-sm font-semibold text-gray-700 mb-2 flex items-center">
                                        <i class="fas fa-water text-blue-500 mr-2"></i>
                                        Caudal Estimado
                                    </label>
                                    <div class="relative">
                                        <input type="number" step="0.001" id="caudal_estimado" readonly
                                               class="w-full border-2 border-gray-200 rounded-lg shadow-sm bg-gradient-to-r from-gray-50 to-gray-100 text-lg font-bold text-gray-700 pl-3 pr-12 py-2">
                                        <span class="absolute right-3 top-2 text-sm text-gray-500">m³/s</span>
                                    </div>
                                </div>
                            </div>
                            <div class="mt-6 p-4 bg-white bg-opacity-80 rounded-lg border border-emerald-200">
                                <p class="text-xs text-emerald-700 flex items-center">
                                    <i class="fas fa-info-circle mr-2"></i>
                                    <strong>Automático:</strong> Os valores são calculados em tempo real conforme você insere as leituras de nível.
                                </p>
                            </div>
                        </div>
                    </div>
                    @endif

                    <!-- Seção de Cotas (se configurada para a estação) -->
                    @if($estacao->trabalhaComCotas() && $estacao->isHidrometrica())
                    <div class="bg-gradient-to-r from-purple-50 to-indigo-50 rounded-lg p-6">
                        <h4 class="font-semibold text-purple-900 mb-4 flex items-center">
                            <i class="fas fa-ruler-vertical mr-2"></i>
                            Leituras de Cotas
                            <span class="ml-2 px-2 py-1 bg-purple-200 text-purple-800 text-xs rounded-full">
                                {{ $estacao->recebeCotasManualmente() ? 'Manual' : 'Automática' }}
                            </span>
                        </h4>

                        @if($estacao->recebeCotasManualmente())
                        <!-- Entrada Manual de Cotas -->
                        <div class="grid grid-cols-1 md:grid-cols-5 gap-3">
                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota 6h (m)
                                </label>
                                <input type="number" step="0.001" name="cota_6h" id="cota_6h"
                                       placeholder="142.350"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500 text-sm">
                                <div class="mt-1">
                                    <label class="flex items-center text-xs text-gray-600">
                                        <input type="checkbox" name="campos_sem_dados[]" value="cota_6h" class="mr-1 text-red-600">
                                        Sem Dados (F)
                                    </label>
                                </div>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota 9h (m)
                                </label>
                                <input type="number" step="0.001" name="cota_9h" id="cota_9h"
                                       placeholder="142.365"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500 text-sm">
                                <div class="mt-1">
                                    <label class="flex items-center text-xs text-gray-600">
                                        <input type="checkbox" name="campos_sem_dados[]" value="cota_9h" class="mr-1 text-red-600">
                                        Sem Dados (F)
                                    </label>
                                </div>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota 12h (m)
                                </label>
                                <input type="number" step="0.001" name="cota_12h" id="cota_12h"
                                       placeholder="142.385"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500 text-sm">
                                <div class="mt-1">
                                    <label class="flex items-center text-xs text-gray-600">
                                        <input type="checkbox" name="campos_sem_dados[]" value="cota_12h" class="mr-1 text-red-600">
                                        Sem Dados (F)
                                    </label>
                                </div>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota 15h (m)
                                </label>
                                <input type="number" step="0.001" name="cota_15h" id="cota_15h"
                                       placeholder="142.375"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500 text-sm">
                                <div class="mt-1">
                                    <label class="flex items-center text-xs text-gray-600">
                                        <input type="checkbox" name="campos_sem_dados[]" value="cota_15h" class="mr-1 text-red-600">
                                        Sem Dados (F)
                                    </label>
                                </div>
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota 18h (m)
                                </label>
                                <input type="number" step="0.001" name="cota_18h" id="cota_18h"
                                       placeholder="142.370"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500 text-sm">
                                <div class="mt-1">
                                    <label class="flex items-center text-xs text-gray-600">
                                        <input type="checkbox" name="campos_sem_dados[]" value="cota_18h" class="mr-1 text-red-600">
                                        Sem Dados (F)
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="mt-4 p-3 bg-purple-100 rounded-lg">
                            <p class="text-sm text-purple-700">
                                <i class="fas fa-info-circle mr-2"></i>
                                <strong>Entrada Manual:</strong> Esta estação não possui escala ou régua. Insira as cotas obtidas por outros meios (GPS, topografia, etc.).
                            </p>
                        </div>

                        @else
                        <!-- Cálculo Automático das Cotas -->
                        <div class="grid grid-cols-1 md:grid-cols-5 gap-3">
                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota Calc. 6h (m)
                                </label>
                                <input type="number" step="0.001" id="cota_calc_6h" readonly
                                       class="w-full border-gray-300 rounded-md shadow-sm bg-gray-50 text-sm">
                                <input type="hidden" name="cota_6h" id="cota_hidden_6h">
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota Calc. 9h (m)
                                </label>
                                <input type="number" step="0.001" id="cota_calc_9h" readonly
                                       class="w-full border-gray-300 rounded-md shadow-sm bg-gray-50 text-sm">
                                <input type="hidden" name="cota_9h" id="cota_hidden_9h">
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota Calc. 12h (m)
                                </label>
                                <input type="number" step="0.001" id="cota_calc_12h" readonly
                                       class="w-full border-gray-300 rounded-md shadow-sm bg-gray-50 text-sm">
                                <input type="hidden" name="cota_12h" id="cota_hidden_12h">
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota Calc. 15h (m)
                                </label>
                                <input type="number" step="0.001" id="cota_calc_15h" readonly
                                       class="w-full border-gray-300 rounded-md shadow-sm bg-gray-50 text-sm">
                                <input type="hidden" name="cota_15h" id="cota_hidden_15h">
                            </div>

                            <div>
                                <label class="block text-xs font-medium text-gray-700 mb-2">
                                    Cota Calc. 18h (m)
                                </label>
                                <input type="number" step="0.001" id="cota_calc_18h" readonly
                                       class="w-full border-gray-300 rounded-md shadow-sm bg-gray-50 text-sm">
                                <input type="hidden" name="cota_18h" id="cota_hidden_18h">
                            </div>
                        </div>

                        <div class="mt-4 p-3 bg-indigo-100 rounded-lg">
                            <p class="text-sm text-indigo-700">
                                <i class="fas fa-calculator mr-2"></i>
                                <strong>Cálculo Automático:</strong> As cotas são calculadas automaticamente a partir dos níveis usando a fórmula:
                                <span class="font-mono">{{ $estacao->getFormulaTexto() ?? 'Cota = Nível + Referência' }}</span>
                            </p>
                        </div>
                        @endif
                    </div>
                    @endif

                    <!-- Dados Pluviométricos -->
                    @if(in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica']))
                    <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-lg p-6">
                        <h4 class="font-semibold text-green-900 mb-4 flex items-center">
                            <i class="fas fa-cloud-rain mr-2"></i>
                            Dados Pluviométricos
                        </h4>

                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Precipitação Total (mm) *
                                </label>
                                <input type="number" step="0.1" name="precipitacao_mm" id="precipitacao_mm"
                                       placeholder="Ex: 15.2"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-green-500 focus:ring-green-500">
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Intensidade Máxima (mm/h)
                                </label>
                                <input type="number" step="0.1" name="intensidade_maxima"
                                       placeholder="Ex: 5.2"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-green-500 focus:ring-green-500">
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Duração (horas)
                                </label>
                                <input type="number" step="0.1" name="duracao_precipitacao"
                                       placeholder="Ex: 3.5"
                                       class="w-full border-gray-300 rounded-md shadow-sm focus:border-green-500 focus:ring-green-500">
                            </div>

                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">
                                    Tipo de Precipitação
                                </label>
                                <select name="tipo_precipitacao"
                                        class="w-full border-gray-300 rounded-md shadow-sm focus:border-green-500 focus:ring-green-500">
                                    <option value="">Selecionar</option>
                                    <option value="chuva">Chuva</option>
                                    <option value="chuvisco">Chuvisco</option>
                                    <option value="granizo">Granizo</option>
                                    <option value="neve">Neve</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    @endif
                </div>

                <!-- Dados Meteorológicos (se aplicável) -->
                @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']))
                <div class="bg-gradient-to-r from-purple-50 to-pink-50 rounded-lg p-6">
                    <h4 class="font-semibold text-purple-900 mb-4 flex items-center">
                        <i class="fas fa-thermometer-half mr-2"></i>
                        Dados de Temperatura
                    </h4>

                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Temperatura Máxima (°C)
                            </label>
                            <input type="number" step="0.1" name="temperatura_max"
                                   placeholder="Ex: 28.5"
                                   class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Temperatura Mínima (°C)
                            </label>
                            <input type="number" step="0.1" name="temperatura_min"
                                   placeholder="Ex: 18.2"
                                   class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Humidade Relativa (%)
                            </label>
                            <input type="number" step="0.1" min="0" max="100" name="humidade_relativa"
                                   placeholder="Ex: 65.5"
                                   class="w-full border-gray-300 rounded-md shadow-sm focus:border-purple-500 focus:ring-purple-500">
                        </div>
                    </div>
                </div>
                @endif
            </div>

            <!-- Observações -->
            <div class="p-6 border-t bg-gray-50">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        <i class="fas fa-sticky-note mr-1"></i>
                        Observações
                    </label>
                    <textarea name="observacoes" rows="3"
                              class="w-full border-gray-300 rounded-md shadow-sm focus:border-green-500 focus:ring-green-500"
                              placeholder="Observações sobre as condições meteorológicas, equipamentos, anomalias observadas, etc..."></textarea>
                </div>
            </div>

            <!-- Footer de Ações -->
            <div class="bg-white px-6 py-6 border-t border-gray-200">
                <!-- Ações Principais -->
                <div class="flex justify-center gap-4">
                    <a href="{{ route('estacoes.leituras', $estacao) }}"
                       class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-3 rounded-lg font-semibold transition-all duration-200 inline-flex items-center">
                        <i class="fas fa-times mr-2"></i>
                        Cancelar
                    </a>
                    <button type="submit"
                            class="bg-green-600 hover:bg-green-700 text-white px-8 py-3 rounded-lg font-bold transition-all duration-200 inline-flex items-center shadow-md hover:shadow-lg">
                        <i class="fas fa-save mr-2"></i>
                        Salvar Leitura
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>


<!-- Scripts necessários -->
<script>
// Configurar variáveis globais para o JavaScript
window.estacaoId = {{ $estacao->id }};
window.estacaoTipo = '{{ $estacao->tipo }}';
window.estacaoCodigo = '{{ $estacao->codigo }}';
window.estacaoNome = '{{ $estacao->nome }}';
window.estacaoNivelAlerta = {{ $estacao->nivel_alerta ?? 'null' }};

// Calculation functions
function calcularEstatisticas() {
    const nivel6h = parseFloat(document.getElementById('nivel_6h')?.value || 0);
    const nivel9h = parseFloat(document.getElementById('nivel_9h')?.value || 0);
    const nivel12h = parseFloat(document.getElementById('nivel_12h')?.value || 0);
    const nivel15h = parseFloat(document.getElementById('nivel_15h')?.value || 0);
    const nivel18h = parseFloat(document.getElementById('nivel_18h')?.value || 0);

    const niveis = [nivel6h, nivel9h, nivel12h, nivel15h, nivel18h].filter(n => n > 0);

    if (niveis.length > 0) {
        const nivelMedio = niveis.reduce((sum, n) => sum + n, 0) / niveis.length;
        const variacao = niveis.length > 1 ? Math.max(...niveis) - Math.min(...niveis) : 0;

        document.getElementById('nivel_medio').value = nivelMedio.toFixed(2);
        document.getElementById('variacao_nivel').value = variacao.toFixed(2);

        // Cálculo estimado do caudal usando fórmula da estação
        @if($estacao->getFormulaAtiva())
            const formula = {
                a: {{ $estacao->getFormulaAtiva()->coeficiente_a }},
                b: {{ $estacao->getFormulaAtiva()->coeficiente_b }},
                c: {{ $estacao->getFormulaAtiva()->coeficiente_c }}
            };
            const caudalEstimado = formula.a * Math.pow((nivelMedio - formula.c), formula.b);
            document.getElementById('caudal_estimado').value = Math.max(0, caudalEstimado).toFixed(3);
        @else
            // Fórmula placeholder se não houver fórmula configurada
            const caudalEstimado = nivelMedio * 0.5;
            document.getElementById('caudal_estimado').value = caudalEstimado.toFixed(3);
        @endif
    }

    // Calcular cotas automaticamente se configurado
    @if($estacao->trabalhaComCotas() && $estacao->calculaCotasAutomaticamente())
    calcularCotasAutomaticamente();
    @endif
}

// Função para calcular cotas automaticamente
@if($estacao->trabalhaComCotas() && $estacao->calculaCotasAutomaticamente())
function calcularCotasAutomaticamente() {
    const nivel6h = parseFloat(document.getElementById('nivel_6h').value) || 0;
    const nivel9h = parseFloat(document.getElementById('nivel_9h').value) || 0;
    const nivel12h = parseFloat(document.getElementById('nivel_12h').value) || 0;
    const nivel15h = parseFloat(document.getElementById('nivel_15h').value) || 0;
    const nivel18h = parseFloat(document.getElementById('nivel_18h').value) || 0;

    const config = {
        tipo: '{{ $estacao->formula_cota_tipo }}',
        referencia: {{ $estacao->cota_referencia ?? 0 }},
        a: {{ $estacao->formula_cota_a ?? 1 }},
        b: {{ $estacao->formula_cota_b ?? 0 }},
        c: {{ $estacao->formula_cota_c ?? 0 }}
    };

    function calcularCota(nivel) {
        if (!nivel) return '';

        switch(config.tipo) {
            case 'simples':
                return nivel + config.referencia;
            case 'linear':
                return (config.a * nivel) + config.b;
            case 'completa':
                return (config.a * nivel * nivel) + (config.b * nivel) + config.c;
            default:
                return nivel + config.referencia;
        }
    }

    if (nivel6h > 0) {
        const cota6h = calcularCota(nivel6h);
        document.getElementById('cota_calc_6h').value = cota6h.toFixed(3);
        document.getElementById('cota_hidden_6h').value = cota6h.toFixed(3);
    }

    if (nivel9h > 0) {
        const cota9h = calcularCota(nivel9h);
        document.getElementById('cota_calc_9h').value = cota9h.toFixed(3);
        document.getElementById('cota_hidden_9h').value = cota9h.toFixed(3);
    }

    if (nivel12h > 0) {
        const cota12h = calcularCota(nivel12h);
        document.getElementById('cota_calc_12h').value = cota12h.toFixed(3);
        document.getElementById('cota_hidden_12h').value = cota12h.toFixed(3);
    }

    if (nivel15h > 0) {
        const cota15h = calcularCota(nivel15h);
        document.getElementById('cota_calc_15h').value = cota15h.toFixed(3);
        document.getElementById('cota_hidden_15h').value = cota15h.toFixed(3);
    }

    if (nivel18h > 0) {
        const cota18h = calcularCota(nivel18h);
        document.getElementById('cota_calc_18h').value = cota18h.toFixed(3);
        document.getElementById('cota_hidden_18h').value = cota18h.toFixed(3);
    }
}
@endif

// Form submission validation
document.getElementById('leituraForm').addEventListener('submit', function(e) {
    // Basic validation
    const dataLeitura = document.getElementById('data_leitura').value;
    if (!dataLeitura) {
        e.preventDefault();
        alert('Por favor, selecione a data da leitura.');
        return;
    }

    // Station type specific validation
    if (window.estacaoTipo === 'hidrometrica') {
        const nivel6h = document.getElementById('nivel_6h').value;
        const nivel9h = document.getElementById('nivel_9h').value;
        const nivel12h = document.getElementById('nivel_12h').value;
        const nivel15h = document.getElementById('nivel_15h').value;
        const nivel18h = document.getElementById('nivel_18h').value;

        if (!nivel6h && !nivel9h && !nivel12h && !nivel15h && !nivel18h) {
            e.preventDefault();
            alert('Por favor, preencha pelo menos um nível de água.');
            return;
        }
    }

    if (['pluviometrica', 'meteorologica', 'climatologica'].includes(window.estacaoTipo)) {
        const precipitacao = document.getElementById('precipitacao_mm').value;
        if (!precipitacao) {
            if (!confirm('Nenhum valor de precipitação informado. Confirma que não houve precipitação?')) {
                e.preventDefault();
                return;
            }
        }
    }
});
</script>
<script src="{{ asset('js/leitura-inteligente.js') }}"></script>
@endsection