@extends('layouts.app')

@section('title', 'Mapa Interativo')

@section('content')
<div class="space-y-6">
    <!-- Cabeçalho -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex justify-between items-center">
            <div>
                <h1 class="text-3xl font-bold text-gray-800">
                    <i class="fas fa-map-marked-alt mr-2 text-blue-600"></i>
                    Mapa Interativo
                </h1>
                @if($itemSelecionado)
                    <p class="text-gray-600 mt-2">
                        <i class="fas fa-{{ $tipo === 'estacao' ? 'tachometer-alt' : 'water' }} mr-1"></i>
                        Visualizando: <strong>{{ $itemSelecionado->nome }}</strong>
                    </p>
                @endif
            </div>
            <div>
                <button class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg flex items-center" onclick="history.back()">
                    <i class="fas fa-arrow-left mr-2"></i>Voltar
                </button>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-4 gap-6">
        <!-- Painel lateral com informações -->
        <div class="lg:col-span-1">
            <div class="bg-white rounded-lg shadow h-full">
                <div class="bg-blue-600 text-white rounded-t-lg p-4">
                    <h5 class="text-lg font-semibold">
                        <i class="fas fa-info-circle mr-2"></i>
                        Informações
                    </h5>
                </div>
                <div class="p-4 space-y-4">
                    @if($itemSelecionado)
                        <!-- Item selecionado -->
                        <div class="p-4 bg-gray-50 rounded-lg">
                            <h6 class="text-blue-600 font-semibold mb-2">
                                <i class="fas fa-{{ $tipo === 'estacao' ? 'tachometer-alt' : 'water' }} mr-1"></i>
                                {{ ucfirst($tipo) }} Selecionada
                            </h6>
                            <p class="font-medium mb-1">{{ $itemSelecionado->nome }}</p>
                            <p class="text-sm text-gray-600 mb-3">
                                <i class="fas fa-map-marker-alt mr-1"></i>
                                {{ number_format($lat, 6) }}°, {{ number_format($lng, 6) }}°
                            </p>

                            @if($tipo === 'estacao')
                                <div class="space-y-2 text-sm">
                                    <div class="flex justify-between">
                                        <span>Tipo:</span>
                                        <span class="bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs">{{ ucfirst($itemSelecionado->tipo) }}</span>
                                    </div>
                                    @if($itemSelecionado->ultima_leitura?->nivel_hidrometrico)
                                        <div class="flex justify-between">
                                            <span>Nível Atual:</span>
                                            <span class="text-blue-600 font-medium">{{ number_format($itemSelecionado->ultima_leitura->nivel_hidrometrico, 2) }}m</span>
                                        </div>
                                    @endif
                                    @if($itemSelecionado->nivel_alerta)
                                        <div class="flex justify-between">
                                            <span>Nível Alerta:</span>
                                            <span class="text-yellow-600 font-medium">{{ $itemSelecionado->nivel_alerta }}m</span>
                                        </div>
                                    @endif
                                </div>
                            @elseif($tipo === 'barragem')
                                <div class="space-y-2 text-sm">
                                    @if($itemSelecionado->ultima_leitura?->cota_actual)
                                        <div class="flex justify-between">
                                            <span>Cota Atual:</span>
                                            <span class="text-blue-600 font-medium">{{ number_format($itemSelecionado->ultima_leitura->cota_actual, 2) }}m</span>
                                        </div>
                                    @endif
                                    @if($itemSelecionado->nivel_alerta)
                                        <div class="flex justify-between">
                                            <span>Nível Alerta:</span>
                                            <span class="text-yellow-600 font-medium">{{ $itemSelecionado->nivel_alerta }}m</span>
                                        </div>
                                    @endif
                                    @if($itemSelecionado->capacidade_total)
                                        <div class="flex justify-between">
                                            <span>Capacidade:</span>
                                            <span class="text-green-600 font-medium">{{ number_format($itemSelecionado->capacidade_total) }}m³</span>
                                        </div>
                                    @endif
                                </div>
                            @endif

                            <div class="mt-3">
                                <a href="{{ $tipo === 'estacao' ? route('estacoes.show', $itemSelecionado->id) : route('barragens.show', $itemSelecionado->id) }}"
                                   class="inline-flex items-center px-3 py-2 text-sm bg-blue-600 hover:bg-blue-700 text-white rounded-lg">
                                    <i class="fas fa-eye mr-1"></i>Ver Detalhes
                                </a>
                            </div>
                        </div>
                    @endif

                    <!-- Legenda -->
                    <div>
                        <h6 class="text-gray-700 font-semibold mb-3">
                            <i class="fas fa-list mr-1"></i>
                            Legenda
                        </h6>
                        <div class="space-y-2 text-sm">
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-blue-500 rounded-full mr-3 border-2 border-white shadow"></div>
                                <span>Estações Hidrométricas</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-4 h-4 bg-green-500 rounded-full mr-3 border-2 border-white shadow"></div>
                                <span>Barragens</span>
                            </div>
                            <div class="flex items-center">
                                <div class="w-5 h-5 bg-red-500 rounded-full mr-3 border-2 border-white shadow animate-pulse"></div>
                                <span>Item Selecionado</span>
                            </div>
                        </div>
                    </div>

                    <!-- Estatísticas -->
                    <div>
                        <h6 class="text-gray-700 font-semibold mb-3">
                            <i class="fas fa-chart-bar mr-1"></i>
                            No Mapa
                        </h6>
                        <div class="space-y-2 text-sm">
                            <div class="flex justify-between">
                                <span>Estações:</span>
                                <span class="bg-gray-200 text-gray-800 px-2 py-1 rounded text-xs">{{ $todasEstacoes->count() }}</span>
                            </div>
                            <div class="flex justify-between">
                                <span>Barragens:</span>
                                <span class="bg-gray-200 text-gray-800 px-2 py-1 rounded text-xs">{{ $todasBarragens->count() }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Mapa principal -->
        <div class="lg:col-span-3">
            <div class="bg-white rounded-lg shadow overflow-hidden">
                <div id="mapa-interativo" style="height: 600px; width: 100%;">
                    <div class="h-full flex justify-center items-center">
                        <div class="text-center">
                            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mb-4"></div>
                            <p class="text-gray-600">Carregando mapa...</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('styles')
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<style>
.leaflet-popup-content-wrapper {
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.leaflet-popup-content {
    margin: 12px 16px;
    line-height: 1.4;
}
</style>
@endpush

@push('scripts')
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    console.log('Inicializando mapa...');

    // Configuração inicial do mapa
    const mapConfig = {
        center: [@if($lat && $lng) {{ $lat }}, {{ $lng }} @else -18.2436, 35.0599 @endif], // Moçambique por padrão
        zoom: @if($lat && $lng) 12 @else 6 @endif,
        itemSelecionado: @if($itemSelecionado) {
            id: {{ $itemSelecionado->id }},
            nome: "{{ $itemSelecionado->nome }}",
            tipo: "{{ $tipo }}",
            lat: {{ $lat }},
            lng: {{ $lng }}
        } @else null @endif
    };

    // Inicializar mapa
    const mapa = L.map('mapa-interativo').setView(mapConfig.center, mapConfig.zoom);

    // Adicionar tiles do OpenStreetMap
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
        maxZoom: 18
    }).addTo(mapa);

    console.log('Mapa inicializado, carregando dados...');

    // Ícones personalizados
    const iconeEstacao = L.divIcon({
        className: 'custom-div-icon',
        html: '<div style="background-color: #3b82f6; width: 16px; height: 16px; border-radius: 50%; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>',
        iconSize: [20, 20],
        iconAnchor: [10, 10]
    });

    const iconeBarragem = L.divIcon({
        className: 'custom-div-icon',
        html: '<div style="background-color: #10b981; width: 16px; height: 16px; border-radius: 50%; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>',
        iconSize: [20, 20],
        iconAnchor: [10, 10]
    });

    const iconeSelecionado = L.divIcon({
        className: 'custom-div-icon',
        html: '<div style="background-color: #ef4444; width: 20px; height: 20px; border-radius: 50%; border: 3px solid white; box-shadow: 0 3px 6px rgba(0,0,0,0.4); animation: pulse 2s infinite;"></div>',
        iconSize: [26, 26],
        iconAnchor: [13, 13]
    });

    // Carregar dados do mapa
    fetch('{{ route("mapas.dados") }}')
        .then(response => {
            console.log('Response status:', response.status);
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Dados recebidos:', data);

            // Adicionar estações
            data.estacoes.forEach(estacao => {
                const icone = (mapConfig.itemSelecionado &&
                             mapConfig.itemSelecionado.tipo === 'estacao' &&
                             mapConfig.itemSelecionado.id === estacao.id) ?
                             iconeSelecionado : iconeEstacao;

                const marker = L.marker([estacao.lat, estacao.lng], { icon: icone })
                    .bindPopup(`
                        <div class="text-center">
                            <h6 class="mb-2 text-blue-600 font-semibold">
                                <i class="fas fa-tachometer-alt mr-1"></i>
                                ${estacao.nome}
                            </h6>
                            <p class="text-sm mb-2">
                                <strong>Tipo:</strong> ${estacao.subtipo}<br>
                                ${estacao.nivel_atual ? `<strong>Nível:</strong> ${Number(estacao.nivel_atual).toFixed(2)}m<br>` : ''}
                                <strong>Status:</strong> <span class="text-green-600">${estacao.status || 'Ativo'}</span>
                            </p>
                            <a href="${estacao.url}" class="inline-block bg-blue-600 hover:bg-blue-700 text-white px-3 py-1 rounded text-sm">
                                <i class="fas fa-eye mr-1"></i>Ver Detalhes
                            </a>
                        </div>
                    `);

                marker.addTo(mapa);
            });

            // Adicionar barragens
            data.barragens.forEach(barragem => {
                const icone = (mapConfig.itemSelecionado &&
                             mapConfig.itemSelecionado.tipo === 'barragem' &&
                             mapConfig.itemSelecionado.id === barragem.id) ?
                             iconeSelecionado : iconeBarragem;

                const marker = L.marker([barragem.lat, barragem.lng], { icon: icone })
                    .bindPopup(`
                        <div class="text-center">
                            <h6 class="mb-2 text-green-600 font-semibold">
                                <i class="fas fa-water mr-1"></i>
                                ${barragem.nome}
                            </h6>
                            <p class="text-sm mb-2">
                                ${barragem.cota_atual ? `<strong>Cota:</strong> ${Number(barragem.cota_atual).toFixed(2)}m<br>` : ''}
                                ${barragem.nivel_alerta ? `<strong>Nível Alerta:</strong> ${barragem.nivel_alerta}m<br>` : ''}
                                ${barragem.capacidade ? `<strong>Capacidade:</strong> ${new Intl.NumberFormat('pt-PT').format(barragem.capacidade)}m³<br>` : ''}
                                <strong>Status:</strong> <span class="text-green-600">Ativo</span>
                            </p>
                            <a href="${barragem.url}" class="inline-block bg-green-600 hover:bg-green-700 text-white px-3 py-1 rounded text-sm">
                                <i class="fas fa-eye mr-1"></i>Ver Detalhes
                            </a>
                        </div>
                    `);

                marker.addTo(mapa);
            });

            console.log(`Mapa carregado: ${data.estacoes.length} estações, ${data.barragens.length} barragens`);

            // Se há item selecionado, abrir popup automaticamente
            if (mapConfig.itemSelecionado) {
                setTimeout(() => {
                    mapa.eachLayer(layer => {
                        if (layer instanceof L.Marker) {
                            const latLng = layer.getLatLng();
                            if (Math.abs(latLng.lat - mapConfig.itemSelecionado.lat) < 0.0001 &&
                                Math.abs(latLng.lng - mapConfig.itemSelecionado.lng) < 0.0001) {
                                layer.openPopup();
                            }
                        }
                    });
                }, 1000);
            }
        })
        .catch(error => {
            console.error('Erro ao carregar dados do mapa:', error);
            document.getElementById('mapa-interativo').innerHTML = `
                <div class="h-full flex justify-center items-center">
                    <div class="text-center text-red-600">
                        <i class="fas fa-exclamation-triangle text-4xl mb-4"></i>
                        <p>Erro ao carregar o mapa</p>
                        <p class="text-sm text-gray-600">${error.message}</p>
                    </div>
                </div>
            `;
        });
});
</script>
@endpush