# 🌊 Integração SEBA-HYDROMETRIE - Guia Completo

## 📋 Visão Geral

A integração com SEBA-HYDROMETRIE permite importação automática de dados hidrológicos diretamente dos equipamentos e plataforma SEBA para o sistema hidrológico de Moçambique.

### ✨ Funcionalidades Implementadas:

- **🔄 Importação Automática:** Dados importados automaticamente em intervalos configuráveis
- **📡 Múltiplos Protocolos:** REST API, SensorThings API, CSV Download, FTP
- **⚡ Jobs em Background:** Processamento assíncrono e resiliente
- **📊 Monitoramento Completo:** Logs detalhados e estatísticas
- **🚨 Alertas Automáticos:** Geração automática de alertas baseados nos dados importados
- **🔁 Sistema de Retry:** Tentativas automáticas em caso de falha

---

## 🛠️ Configuração

### **📋 ANTES DE COMEÇAR - URLs Reais do SEBA**

⚠️ **CRÍTICO:** Este sistema implementa a **estrutura** para integração SEBA, mas você **DEVE** obter as URLs e credenciais reais através de:

1. **Contato Direto SEBA:**
   - 🌐 Website: https://www.seba-hydrometrie.com/
   - 📧 Email: info@seba.de
   - 📞 Tel: +49 8341 9648-0

2. **Documentação Técnica:**
   - Manual da sua instalação SEBA Hydrocenter
   - Documentação API fornecida pela SEBA
   - Guia de integração do seu projeto

3. **Suporte Local:**
   - Representante SEBA em Moçambique
   - Equipe técnica que instalou os equipamentos
   - Fornecedor/integrador local

### 1. **Variáveis de Ambiente (.env)**

⚠️ **As URLs abaixo são EXEMPLOS**. Substitua pelas URLs reais:
- Sua instalação SEBA Hydrocenter
- Documentação técnica fornecida pela SEBA
- Suporte técnico SEBA

```env
# SEBA-HYDROMETRIE Integration (SUBSTITUA COM URLs REAIS!)
SEBA_API_URL=https://your-seba-server.com/api/v1
SEBA_USERNAME=seu_usuario_seba
SEBA_PASSWORD=sua_senha_seba
SEBA_API_KEY=sua_chave_api_opcional
SEBA_TIMEOUT=30

# Métodos alternativos (URLs DE EXEMPLO - substitua pelas reais!)
SEBA_SENSORTHINGS_URL=https://your-frost-server.com/FROST-Server/v1.1
SEBA_CSV_DOWNLOAD_URL=https://your-seba-server.com/data/export

# Configurações de integração
SEBA_AUTO_IMPORT_ENABLED=true
SEBA_DEFAULT_INTERVAL_MINUTES=60
SEBA_MAX_PARALLEL_REQUESTS=3
SEBA_RETRY_ATTEMPTS=3
```

### 2. **Configurar Estações**

Para cada estação que deve importar dados do SEBA:

```sql
UPDATE estacoes SET
    seba_station_id = 'SEBA_STATION_001',
    seba_station_name = 'Nome da Estação no SEBA',
    seba_integration_type = 'rest_api',
    auto_importacao_ativa = true,
    intervalo_importacao_minutos = 60
WHERE id = 1;
```

---

## 🚀 Comandos Disponíveis

### **1. Testar Conectividade**
```bash
php artisan seba:testar-conectividade
```
Verifica se o sistema consegue conectar com os serviços SEBA.

### **2. Importar Estação Específica**
```bash
# Importar últimas 24h
php artisan seba:importar-estacao 1

# Período específico
php artisan seba:importar-estacao 1 --inicio="2024-09-24 08:00" --fim="2024-09-24 18:00"

# Forçar importação mesmo se desabilitada
php artisan seba:importar-estacao 1 --force
```

### **3. Importar Todas as Estações**
```bash
# Importar todas as estações configuradas
php artisan seba:importar-todas

# Período específico
php artisan seba:importar-todas --inicio="2024-09-24 00:00" --fim="2024-09-24 23:59"

# Forçar mesmo estações desabilitadas
php artisan seba:importar-todas --force
```

---

## ⚙️ Automação via Cron

### **Configuração no Crontab:**

```bash
# Importação a cada 15 minutos
*/15 * * * * cd /caminho/para/projeto && php artisan seba:importar-todas >> /var/log/seba-import.log 2>&1

# Importação horária (mais conservador)
0 * * * * cd /caminho/para/projeto && php artisan seba:importar-todas

# Teste de conectividade diário
0 6 * * * cd /caminho/para/projeto && php artisan seba:testar-conectividade
```

### **Usando Laravel Scheduler (Recomendado):**

No arquivo `app/Console/Kernel.php`:

```php
protected function schedule(Schedule $schedule)
{
    // Importação SEBA a cada 30 minutos
    $schedule->command('seba:importar-todas')
             ->everyThirtyMinutes()
             ->withoutOverlapping()
             ->runInBackground();

    // Teste de conectividade às 6h
    $schedule->command('seba:testar-conectividade')
             ->dailyAt('06:00');
}
```

---

## 📊 Monitoramento e Logs

### **1. Logs de Integração**

Todos os logs são armazenados na tabela `seba_integracao_logs`:

```php
use App\Models\SebaIntegracaoLog;

// Estatísticas dos últimos 7 dias
$stats = SebaIntegracaoLog::getEstatisticas(7);

// Logs com erro
$erros = SebaIntegracaoLog::erro()->recentes(1)->get();

// Logs por estação
$logsEstacao = SebaIntegracaoLog::porEstacao(1)->get();
```

### **2. Monitorar via Laravel Log**

```bash
# Monitorar logs em tempo real
tail -f storage/logs/laravel.log | grep SEBA

# Filtrar apenas erros
tail -f storage/logs/laravel.log | grep "ERROR.*SEBA"
```

---

## 🔧 Protocolos de Integração

### **1. REST API (Recomendado)**

```php
// Configuração típica
SEBA_API_URL=https://hydrocenter.seba.com/api/v1
SEBA_USERNAME=usuario
SEBA_PASSWORD=senha
```

**Endpoint típico:** `GET /stations/{station_id}/data`

### **2. SensorThings API (OGC Standard)**

```php
SEBA_SENSORTHINGS_URL=https://sensorthings.seba.com/v1.1
```

**Query exemplo:**
```
GET /Datastreams(STATION_001)/Observations?$filter=phenomenonTime ge 2024-09-24T00:00:00Z
```

### **3. CSV Download**

```php
SEBA_CSV_DOWNLOAD_URL=https://hydrocenter.seba.com/data/export
```

Parâmetros: `station_id`, `start_date`, `end_date`, `format=csv`

---

## 📈 Estrutura de Dados

### **Mapeamento de Campos**

| Campo SEBA | Campo Sistema | Tipo | Descrição |
|------------|---------------|------|-----------|
| `timestamp` | `data_leitura`, `hora_leitura` | DateTime | Data/hora da medição |
| `water_level` | `nivel_hidrometrico` | Float | Nível da água em metros |
| `precipitation` | `precipitacao` | Float | Precipitação em mm |
| `temperature` | `temperatura` | Float | Temperatura em °C |
| `station_id` | `seba_station_id` | String | Identificador da estação |

### **Exemplo de Dados CSV:**

```csv
timestamp,station_id,water_level,precipitation,temperature
2024-09-24 08:00:00,SEBA_001,2.45,0.0,22.5
2024-09-24 08:15:00,SEBA_001,2.47,0.2,22.8
2024-09-24 08:30:00,SEBA_001,2.50,0.5,23.1
```

---

## 🚨 Alertas Automáticos

### **Geração de Alertas**

Os dados importados passam automaticamente pela verificação de limites:

```php
// Para estações hidrométricas
if ($estacao->tipo === 'hidrometrica' && $leitura->nivel_hidrometrico >= $estacao->nivel_alerta) {
    Alerta::criarAlertaEstacao($estacao, $leitura);
}

// Para barragens
if ($leitura->cota_actual >= $barragem->nivel_alerta) {
    Alerta::criarAlertaBarragem($barragem, $leitura);
}
```

### **Tipos de Alertas Gerados:**

- 🟡 **Atenção:** Níveis próximos ao limite
- 🟠 **Alerta:** Níveis acima do limite de segurança
- 🔴 **Emergência:** Níveis críticos
- ✅ **Resolução:** Quando níveis normalizam

---

## ⚡ Jobs e Performance

### **Sistema de Jobs:**

```php
use App\Jobs\SebaImportacaoAutomaticaJob;

// Disparar importação automática
SebaImportacaoAutomaticaJob::dispatch($estacao);

// Agendar para execução futura
SebaImportacaoAutomaticaJob::dispatch($estacao)
    ->delay(now()->addMinutes(60));
```

### **Configurações de Performance:**

- **Timeout:** 300 segundos por job
- **Tentativas:** 3 tentativas com backoff exponencial
- **Paralelismo:** Máximo 3 estações simultâneas
- **Intervalo:** Pausa de 0.5s entre requisições

---

## 🔍 Troubleshooting

### **Problemas Comuns:**

#### **1. Erro de Conectividade**
```bash
# Testar conectividade
php artisan seba:testar-conectividade

# Verificar URLs e credenciais no .env
```

#### **2. Dados Não Importados**
```bash
# Verificar configuração da estação
SELECT seba_station_id, auto_importacao_ativa, intervalo_importacao_minutos
FROM estacoes WHERE id = 1;

# Verificar logs de erro
SELECT * FROM seba_integracao_logs WHERE status = 'erro' ORDER BY created_at DESC;
```

#### **3. Jobs Travados**
```bash
# Verificar queue
php artisan queue:work --once

# Limpar jobs falhados
php artisan queue:clear

# Restart supervisor (se usando)
sudo supervisorctl restart laravel-worker:*
```

#### **4. Muitas Duplicatas**
```bash
# Ajustar intervalo de importação
UPDATE estacoes SET intervalo_importacao_minutos = 120 WHERE seba_station_id IS NOT NULL;
```

---

## 📚 APIs de Exemplo

### **1. REST API Response:**
```json
{
  "station_id": "SEBA_001",
  "measurements": [
    {
      "timestamp": "2024-09-24T08:00:00Z",
      "water_level": 2.45,
      "precipitation": 0.0,
      "temperature": 22.5
    }
  ]
}
```

### **2. SensorThings Response:**
```json
{
  "@iot.count": 100,
  "value": [
    {
      "@iot.id": 1,
      "phenomenonTime": "2024-09-24T08:00:00Z",
      "result": 2.45,
      "Datastream@iot.navigationLink": "Datastreams(1)"
    }
  ]
}
```

---

## ✅ Checklist de Implementação

### **Pré-requisitos:**
- [ ] Credenciais SEBA válidas
- [ ] Conexão de rede com servidores SEBA
- [ ] Queue worker configurado (`php artisan queue:work`)
- [ ] Cron configurado para execução automática

### **Configuração:**
- [ ] Variáveis de ambiente configuradas (`.env`)
- [ ] Migrations executadas (`php artisan migrate`)
- [ ] Estações configuradas com `seba_station_id`
- [ ] Auto-importação ativada nas estações

### **Testes:**
- [ ] Conectividade testada (`seba:testar-conectividade`)
- [ ] Importação manual testada (`seba:importar-estacao 1`)
- [ ] Jobs funcionando (`php artisan queue:work`)
- [ ] Logs sendo gerados corretamente

### **Produção:**
- [ ] Cron configurado para execução automática
- [ ] Monitoramento de logs implementado
- [ ] Alertas de falha configurados
- [ ] Backup de configurações

---

## 🎯 Resumo de Comandos

```bash
# Setup inicial
php artisan migrate
php artisan config:clear

# Testes
php artisan seba:testar-conectividade
php artisan seba:importar-estacao 1 --force

# Produção
php artisan seba:importar-todas
php artisan queue:work --daemon

# Monitoramento
tail -f storage/logs/laravel.log | grep SEBA
```

---

## 📞 Suporte

Para questões técnicas ou problemas de integração:

1. **Verificar logs:** `storage/logs/laravel.log`
2. **Consultar tabela:** `seba_integracao_logs`
3. **Testar conectividade:** `php artisan seba:testar-conectividade`
4. **Documentação SEBA:** [https://www.seba-hydrometrie.com/](https://www.seba-hydrometrie.com/)

---

**🚀 Sistema de Integração SEBA-HYDROMETRIE implementado com sucesso!**