#!/usr/bin/env python3
"""
Análise completa da folha GRAFICOS DE BARRAGEM
"""

import openpyxl
import sys

def analisar_graficos_barragem(ficheiro_path):
    """Análise detalhada da folha GRAFICOS DE BARRAGEM"""

    try:
        # Abrir ficheiro
        wb = openpyxl.load_workbook(ficheiro_path, data_only=False)

        if 'GRAFICOS DE BARRAGEM' not in wb.sheetnames:
            print("❌ Folha 'GRAFICOS DE BARRAGEM' não encontrada")
            return

        ws = wb['GRAFICOS DE BARRAGEM']

        print("=" * 80)
        print("RELATÓRIO COMPLETO: FOLHA 'GRAFICOS DE BARRAGEM'")
        print("=" * 80)
        print()

        # 1. INFORMAÇÕES GERAIS
        print("📊 1. INFORMAÇÕES GERAIS")
        print("-" * 40)
        print(f"Dimensões da folha: {ws.max_row} linhas × {ws.max_column} colunas")
        print(f"Número total de gráficos: {len(ws._charts) if hasattr(ws, '_charts') else 0}")
        print()

        if not hasattr(ws, '_charts') or not ws._charts:
            print("❌ Nenhum gráfico encontrado na folha")
            return

        # 2. ANÁLISE DE CADA GRÁFICO
        print("📈 2. ANÁLISE DETALHADA DOS GRÁFICOS")
        print("-" * 50)

        graficos_barras = []
        graficos_linha = []

        for i, chart in enumerate(ws._charts):
            # Extrair título do gráfico
            titulo_grafico = ""
            if hasattr(chart, 'title') and chart.title and chart.title.text:
                if hasattr(chart.title.text, 'rich') and chart.title.text.rich:
                    for p in chart.title.text.rich.p:
                        if hasattr(p, 'r') and p.r:
                            for r in p.r:
                                if hasattr(r, 't'):
                                    titulo_grafico += r.t

            tipo_grafico = type(chart).__name__

            print(f"🎯 GRÁFICO {i+1}: {titulo_grafico}")
            print(f"   Tipo: {tipo_grafico}")

            # Posição na folha
            if hasattr(chart, 'anchor') and chart.anchor:
                anchor = chart.anchor
                if hasattr(anchor, '_from') and hasattr(anchor, 'to'):
                    from_col = anchor._from.col + 1
                    from_row = anchor._from.row + 1
                    to_col = anchor.to.col + 1
                    to_row = anchor.to.row + 1

                    from_col_letter = chr(64 + from_col) if from_col <= 26 else f"A{chr(64 + from_col - 26)}"
                    to_col_letter = chr(64 + to_col) if to_col <= 26 else f"A{chr(64 + to_col - 26)}"

                    print(f"   Posição: {from_col_letter}{from_row} até {to_col_letter}{to_row}")

            # Dimensões
            if hasattr(chart, 'width') and hasattr(chart, 'height'):
                print(f"   Dimensões: {chart.width} × {chart.height} cm")

            # Eixos
            eixo_y = ""
            if hasattr(chart, 'y_axis') and chart.y_axis and hasattr(chart.y_axis, 'title'):
                if chart.y_axis.title and chart.y_axis.title.text:
                    if hasattr(chart.y_axis.title.text, 'rich'):
                        for p in chart.y_axis.title.text.rich.p:
                            if hasattr(p, 'r') and p.r:
                                for r in p.r:
                                    if hasattr(r, 't'):
                                        eixo_y += r.t
                        print(f"   Eixo Y: {eixo_y}")

            # Séries de dados
            if hasattr(chart, 'series') and chart.series:
                print(f"   Número de séries: {len(chart.series)}")

                for j, serie in enumerate(chart.series):
                    titulo_serie = ""
                    referencia_titulo = ""

                    if hasattr(serie, 'title') and serie.title:
                        if hasattr(serie.title, 'strRef') and serie.title.strRef:
                            referencia_titulo = serie.title.strRef.f
                            if hasattr(serie.title.strRef, 'strCache') and serie.title.strRef.strCache:
                                if hasattr(serie.title.strRef.strCache, 'pt') and serie.title.strRef.strCache.pt:
                                    for pt in serie.title.strRef.strCache.pt:
                                        titulo_serie = pt.v

                    print(f"     📌 Série {j+1}: {titulo_serie}")
                    if referencia_titulo:
                        folha_fonte = referencia_titulo.split('!')[0] if '!' in referencia_titulo else ""
                        print(f"        Folha fonte: {folha_fonte}")
                        print(f"        Referência: {referencia_titulo}")

            # Classificar gráficos por tipo
            if tipo_grafico == "BarChart":
                graficos_barras.append({
                    'num': i+1,
                    'titulo': titulo_grafico,
                    'series': len(chart.series) if hasattr(chart, 'series') and chart.series else 0
                })
            elif tipo_grafico == "LineChart":
                graficos_linha.append({
                    'num': i+1,
                    'titulo': titulo_grafico,
                    'series': len(chart.series) if hasattr(chart, 'series') and chart.series else 0
                })

            print()

        # 3. RESUMO POR TIPOS DE GRÁFICO
        print("📋 3. RESUMO POR TIPOS DE GRÁFICO")
        print("-" * 40)

        print(f"📊 Gráficos de Barras: {len(graficos_barras)}")
        for graf in graficos_barras:
            print(f"   • Gráfico {graf['num']}: {graf['titulo']} ({graf['series']} série{'s' if graf['series'] != 1 else ''})")

        print(f"\n📈 Gráficos de Linha: {len(graficos_linha)}")
        for graf in graficos_linha:
            print(f"   • Gráfico {graf['num']}: {graf['titulo']} ({graf['series']} série{'s' if graf['series'] != 1 else ''})")

        # 4. LAYOUT E ORGANIZAÇÃO
        print(f"\n🏗️ 4. LAYOUT E ORGANIZAÇÃO")
        print("-" * 40)
        print("A folha está organizada com gráficos distribuídos em diferentes áreas:")
        print("• Área superior esquerda: Gráfico de linha da Barragem de Nampula")
        print("• Área superior direita: Gráfico de barras da Barragem de Nacala")
        print("• Área central: Gráficos de barras das barragens Mitucue e Locomue")
        print("• Área inferior: Gráficos de barras das barragens Mugica e Chipembe")
        print("• Área direita: Gráfico adicional de linha da Barragem de Nampula")

        # 5. FONTES DE DADOS
        print(f"\n💾 5. FONTES DE DADOS")
        print("-" * 40)
        folhas_fonte = set()

        for chart in ws._charts:
            if hasattr(chart, 'series') and chart.series:
                for serie in chart.series:
                    if hasattr(serie, 'title') and serie.title:
                        if hasattr(serie.title, 'strRef') and serie.title.strRef:
                            ref = serie.title.strRef.f
                            if '!' in ref:
                                folha = ref.split('!')[0]
                                folhas_fonte.add(folha)

        print("Folhas fonte dos dados:")
        for folha in sorted(folhas_fonte):
            print(f"   • {folha}")

        print(f"\n✅ ANÁLISE CONCLUÍDA - {len(ws._charts)} gráficos analisados com sucesso!")

    except Exception as e:
        print(f"❌ Erro durante a análise: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    ficheiro = "ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"
    analisar_graficos_barragem(ficheiro)