#!/usr/bin/env python3
"""
Script para analisar especificamente as fórmulas e estrutura das barragens
"""

import pandas as pd
import sys
import os
from openpyxl import load_workbook
import warnings
warnings.filterwarnings('ignore')

def analyze_main_barragem_file():
    """Analisa o ficheiro principal das barragens focando nos cálculos"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print("ANÁLISE DETALHADA DO FICHEIRO PRINCIPAL: Cópia de BARRAGENS ACT 8.8.xlsm")
    print("="*80)

    try:
        # Carrega o workbook com fórmulas
        wb = load_workbook(file_path)
        print(f"Folhas disponíveis: {wb.sheetnames}")

        # Analisa as folhas principais de barragens
        barragem_sheets = ['NAMPULA', 'NACALA', 'MUGICA', 'CHIPEMBE', 'LOCUMUE', 'MITUCUE']

        for sheet_name in barragem_sheets:
            if sheet_name in wb.sheetnames:
                print(f"\n{'-'*50}")
                print(f"BARRAGEM: {sheet_name}")
                print(f"{'-'*50}")

                # Lê a folha com pandas para ver a estrutura
                df = pd.read_excel(file_path, sheet_name=sheet_name, engine='openpyxl', nrows=50)
                print(f"Primeiras colunas encontradas:")

                for i, col in enumerate(df.columns[:15]):
                    if pd.notna(col) and str(col).strip():
                        print(f"  {i+1}. {col}")

                # Procura por colunas com percentagens ou volumes
                relevant_cols = []
                for col in df.columns:
                    col_str = str(col).lower()
                    if any(keyword in col_str for keyword in ['%', 'perc', 'enchimento', 'volume', 'total', 'util', 'cota', 'nivel']):
                        relevant_cols.append(col)

                if relevant_cols:
                    print(f"\nColunas relevantes para cálculos:")
                    for col in relevant_cols:
                        print(f"  - {col}")

                # Analisa as fórmulas na folha openpyxl
                ws = wb[sheet_name]
                print(f"\nProcurando fórmulas de percentagem...")

                formulas_found = []
                for row_num in range(1, min(100, ws.max_row + 1)):  # Primeiras 100 linhas
                    for col_num in range(1, min(50, ws.max_column + 1)):  # Primeiras 50 colunas
                        cell = ws.cell(row=row_num, column=col_num)
                        if cell.data_type == 'f' and cell.value:
                            cell_value = str(cell.value).lower()
                            # Procura fórmulas com percentagem, divisão ou que parecem calcular enchimento
                            if any(keyword in cell_value for keyword in ['/', '*100', 'percentagem', '%']):
                                formulas_found.append({
                                    'cell': cell.coordinate,
                                    'formula': cell.value,
                                    'value': cell.display_value
                                })

                if formulas_found:
                    print(f"Fórmulas de percentagem encontradas:")
                    for formula in formulas_found[:10]:  # Primeiras 10
                        print(f"  {formula['cell']}: {formula['formula']} = {formula['value']}")

                # Procura dados nas primeiras linhas para entender estrutura
                print(f"\nEstrutura dos dados (primeiras linhas):")
                for row_num in range(1, 8):
                    row_data = []
                    for col_num in range(1, 16):
                        cell = ws.cell(row=row_num, column=col_num)
                        if cell.value is not None:
                            row_data.append(str(cell.value)[:30])  # Primeiros 30 caracteres
                        else:
                            row_data.append("")

                    if any(data.strip() for data in row_data):
                        print(f"  Linha {row_num}: {' | '.join(row_data[:10])}")

        # Analisa a folha de gráficos se existir
        if 'GRAFICOS DE BARRAGEM' in wb.sheetnames:
            print(f"\n{'-'*50}")
            print(f"FOLHA: GRAFICOS DE BARRAGEM")
            print(f"{'-'*50}")

            ws_graficos = wb['GRAFICOS DE BARRAGEM']
            print(f"Esta folha tem {ws_graficos.max_row} linhas e {ws_graficos.max_column} colunas")

    except Exception as e:
        print(f"Erro: {str(e)}")
        import traceback
        traceback.print_exc()

def analyze_other_files():
    """Analisa outros ficheiros Excel relevantes"""
    print(f"\n{'='*80}")
    print("ANÁLISE DOS OUTROS FICHEIROS")
    print(f"{'='*80}")

    files_to_analyze = [
        ("SAC-2024-25.xls", "Sistema de Armazenamento de Água"),
        ("P-332.xls", "Dados P-332"),
        ("E-127-2011-12.xls", "Dados E-127")
    ]

    ficheiros_dir = "/Users/salumsaidsalum/Desktop/script/ficheiros/"

    for filename, description in files_to_analyze:
        file_path = os.path.join(ficheiros_dir, filename)

        if os.path.exists(file_path):
            print(f"\n{'-'*50}")
            print(f"FICHEIRO: {filename} ({description})")
            print(f"{'-'*50}")

            try:
                xl_file = pd.ExcelFile(file_path)
                print(f"Folhas: {xl_file.sheet_names}")

                # Analisa a primeira folha de cada ficheiro
                first_sheet = xl_file.sheet_names[0]
                df = pd.read_excel(file_path, sheet_name=first_sheet, nrows=20)
                print(f"Primeira folha '{first_sheet}' - Dimensões: {df.shape}")

                print("Primeiras colunas:")
                for i, col in enumerate(df.columns[:10]):
                    print(f"  {i+1}. {col}")

            except Exception as e:
                print(f"Erro ao analisar {filename}: {str(e)}")

if __name__ == "__main__":
    analyze_main_barragem_file()
    analyze_other_files()