import openpyxl
from openpyxl.utils import get_column_letter
import datetime

def analyze_data_sheets():
    """Analisar as folhas de dados principais"""
    wb = openpyxl.load_workbook('ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm', data_only=False)

    # Folhas principais que contêm dados de barragens
    data_sheets = ['NAMPULA', 'NACALA', 'MUGICA', 'CHIPEMBE', 'LOCUMUE', 'MITUCUE']

    print('🏗️ ANÁLISE DAS FOLHAS DE DADOS DAS BARRAGENS')
    print('=' * 70)

    for sheet_name in data_sheets:
        if sheet_name in wb.sheetnames:
            ws = wb[sheet_name]
            print(f'')
            print(f'📊 FOLHA: {sheet_name}')
            print('-' * 40)

            # Dimensões
            print(f'Dimensões: {ws.max_row} linhas x {ws.max_column} colunas')

            # Analisar headers/cabeçalhos (primeiras 10 linhas)
            print('Estrutura dos cabeçalhos/dados:')
            found_data = False
            for row in range(1, min(20, ws.max_row + 1)):
                row_data = []
                has_meaningful_data = False
                
                for col in range(1, min(20, ws.max_column + 1)):  
                    cell = ws.cell(row=row, column=col)
                    value = cell.value
                    if value is not None:
                        str_val = str(value)[:25]  # Limitar tamanho
                        if str_val.strip() and str_val != 'None':
                            has_meaningful_data = True
                        row_data.append(str_val)
                    else:
                        row_data.append('')

                # Só mostrar se tiver dados relevantes
                if has_meaningful_data:
                    # Mostrar apenas primeiras 8 colunas para não sobrecarregar
                    display_data = []
                    for i, item in enumerate(row_data[:8]):
                        if item.strip():
                            display_data.append(f'Col{get_column_letter(i+1)}:{item}')
                        else:
                            display_data.append('')
                    
                    non_empty = [x for x in display_data if x]
                    if non_empty:
                        print(f'   Linha {row}: {" | ".join(non_empty[:6])}')
                        found_data = True
                    
                    # Limitar saída
                    if row > 10 and found_data:
                        break

            # Procurar padrões de dados temporais
            print('')
            print('🕒 Procurando dados temporais/datas:')
            date_patterns_found = []
            
            for row in range(1, min(50, ws.max_row + 1)):
                for col in range(1, min(15, ws.max_column + 1)):
                    cell = ws.cell(row=row, column=col)
                    if cell.value:
                        val_str = str(cell.value).lower()
                        # Procurar padrões de data
                        if ('data' in val_str or 
                            any(char.isdigit() for char in val_str) and ('/' in val_str or '-' in val_str) or
                            isinstance(cell.value, datetime.datetime) or
                            isinstance(cell.value, datetime.date)):
                            date_patterns_found.append(f'   {get_column_letter(col)}{row}: {cell.value}')
                            if len(date_patterns_found) >= 3:  # Limitar a 3 exemplos
                                break
                if len(date_patterns_found) >= 3:
                    break
            
            if date_patterns_found:
                for pattern in date_patterns_found:
                    print(pattern)
            else:
                print('   Nenhum padrão de data óbvio encontrado nas primeiras linhas')

            # Procurar por campos relacionados com barragem
            print('')
            print('🏗️ Procurando campos relacionados com barragem:')
            dam_fields = []
            keywords = ['npa', 'nme', 'volume', 'morto', 'nível', 'cota', 'água', 'barragem']
            
            for row in range(1, min(20, ws.max_row + 1)):
                for col in range(1, min(20, ws.max_column + 1)):
                    cell = ws.cell(row=row, column=col)
                    if cell.value:
                        val_str = str(cell.value).lower()
                        for keyword in keywords:
                            if keyword in val_str:
                                dam_fields.append(f'   {get_column_letter(col)}{row}: {cell.value}')
                                break
            
            # Mostrar apenas os primeiros 5 campos encontrados
            for field in dam_fields[:5]:
                print(field)
            
            if not dam_fields:
                print('   Nenhum campo relacionado com barragem encontrado nos cabeçalhos')

if __name__ == "__main__":
    try:
        analyze_data_sheets()
    except Exception as e:
        print(f'❌ Erro: {e}')
        import traceback
        traceback.print_exc()
