<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithCustomStartCell;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;

class EstatisticasExport implements FromArray, WithStyles, WithTitle, WithEvents, ShouldAutoSize, WithCustomStartCell
{
    protected $medias;
    protected $tipo;
    protected $periodo;
    protected $entidadeNome;
    protected $graficoLabels;
    protected $graficoDados;

    public function __construct($medias, $tipo, $periodo, $entidadeNome, $graficoLabels, $graficoDados)
    {
        $this->medias = $medias;
        $this->tipo = $tipo;
        $this->periodo = $periodo;
        $this->entidadeNome = $entidadeNome;
        $this->graficoLabels = $graficoLabels;
        $this->graficoDados = $graficoDados;
    }

    public function array(): array
    {
        \Log::info('Exportando Excel com ' . $this->medias->count() . ' registros');

        return $this->medias->map(function($media) {
            $periodo = '';
            if (isset($media->data)) {
                $periodo = Carbon::parse($media->data)->format('d/m/Y');
            } elseif (isset($media->semana)) {
                $periodo = "Semana {$media->semana}/{$media->ano}";
            } elseif (isset($media->mes)) {
                $periodo = Carbon::create($media->ano, $media->mes)->format('F/Y');
            } else {
                $periodo = $media->ano;
            }

            $row = [
                $periodo,
                (float) $media->media,
            ];

            if ($this->tipo === 'precipitacao') {
                $row[] = (float) $media->total;
            }

            $row[] = (float) $media->minimo;
            $row[] = (float) $media->maximo;
            $row[] = (int) $media->leituras;

            return $row;
        })->toArray();
    }

    public function headings(): array
    {
        $unidade = $this->tipo === 'precipitacao' ? 'mm' : 'm';

        $headings = [
            'Período',
            "Média ($unidade)",
        ];

        if ($this->tipo === 'precipitacao') {
            $headings[] = "Total ($unidade)";
        }

        $headings[] = "Mínimo ($unidade)";
        $headings[] = "Máximo ($unidade)";
        $headings[] = 'Nº Leituras';

        return $headings;
    }

    public function styles(Worksheet $sheet)
    {
        return [];
    }

    public function title(): string
    {
        return 'Estatísticas';
    }

    public function startCell(): string
    {
        return 'A5'; // Data starts at row 5, after title (row 1), subtitle (row 2), spacing (row 3), and headers (row 4)
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                $lastColumn = $this->tipo === 'precipitacao' ? 'F' : 'E';

                // Título
                $sheet->mergeCells("A1:{$lastColumn}1");
                $sheet->setCellValue('A1', 'ESTATÍSTICAS - ' . strtoupper($this->tipo));
                $sheet->getStyle('A1')->applyFromArray([
                    'font' => ['size' => 18, 'bold' => true, 'color' => ['rgb' => '1F2937']],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                ]);
                $sheet->getRowDimension(1)->setRowHeight(30);

                // Subtítulo
                $sheet->mergeCells("A2:{$lastColumn}2");
                $sheet->setCellValue('A2', $this->entidadeNome . ' | ' . ucfirst($this->periodo) . ' | ' . Carbon::now()->format('d/m/Y H:i'));
                $sheet->getStyle('A2')->applyFromArray([
                    'font' => ['size' => 12],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER],
                ]);
                $sheet->getRowDimension(2)->setRowHeight(20);

                // Espaçamento
                $sheet->getRowDimension(3)->setRowHeight(10);

                // Cabeçalhos da tabela (linha 4)
                $sheet->getStyle("A4:{$lastColumn}4")->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF'], 'size' => 11],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '1F2937']],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
                ]);
                $sheet->getRowDimension(4)->setRowHeight(25);

                // Inserir cabeçalhos na linha 4
                $headings = $this->headings();
                foreach ($headings as $index => $heading) {
                    $columnLetter = chr(65 + $index);
                    $sheet->setCellValue($columnLetter . '4', $heading);
                }

                // Dados começam na linha 5
                $dataStartRow = 5;
                $dataEndRow = $dataStartRow + $this->medias->count() - 1;

                // Zebra striping
                for ($row = $dataStartRow; $row <= $dataEndRow; $row++) {
                    if ($row % 2 == 0) {
                        $sheet->getStyle("A{$row}:{$lastColumn}{$row}")->applyFromArray([
                            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F3F4F6']],
                        ]);
                    }
                }

                // Centralizar dados
                $sheet->getStyle("A{$dataStartRow}:{$lastColumn}{$dataEndRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

                // Destacar coluna de média (coluna B)
                $sheet->getStyle("B{$dataStartRow}:B{$dataEndRow}")->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['rgb' => '2563EB']],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'EFF6FF']],
                ]);

                // Bordas na tabela
                $sheet->getStyle("A4:{$lastColumn}{$dataEndRow}")->applyFromArray([
                    'borders' => [
                        'allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => 'D1D5DB']],
                    ],
                ]);

                // Inserir gráfico como imagem usando GD Library (PHP nativo)
                try {
                    \Log::info('Tentando gerar gráfico', [
                        'num_labels' => count($this->graficoLabels),
                        'num_dados' => count($this->graficoDados),
                        'labels' => $this->graficoLabels,
                        'dados' => $this->graficoDados
                    ]);

                    if (!empty($this->graficoLabels) && !empty($this->graficoDados)) {
                        $graficoInfo = $this->gerarGraficoGD();

                        if ($graficoInfo && file_exists($graficoInfo['file'])) {
                            \Log::info('Arquivo de gráfico gerado', [
                                'path' => $graficoInfo['file'],
                                'size' => filesize($graficoInfo['file']),
                                'width' => $graficoInfo['width'],
                                'height' => $graficoInfo['height']
                            ]);

                            // Criar objeto Drawing
                            $drawing = new Drawing();
                            $drawing->setName('Gráfico');
                            $drawing->setDescription('Gráfico de Estatísticas');
                            $drawing->setPath($graficoInfo['file']);

                            // Posicionar ao lado da tabela (coluna H ou G dependendo do tipo)
                            $graficoCol = $this->tipo === 'precipitacao' ? 'H' : 'G';
                            $drawing->setCoordinates("{$graficoCol}4");

                            // Ajustar tamanho baseado nas dimensões reais da imagem
                            $drawing->setHeight($graficoInfo['height']);
                            $drawing->setWidth($graficoInfo['width']);

                            // Offset para garantir que fique bem posicionado
                            $drawing->setOffsetX(20);
                            $drawing->setOffsetY(10);

                            // Adicionar à worksheet
                            $drawing->setWorksheet($sheet);

                            \Log::info('Gráfico inserido como imagem com sucesso', [
                                'position' => "{$graficoCol}4",
                                'height' => $graficoInfo['height'],
                                'width' => $graficoInfo['width']
                            ]);

                            // Agendar remoção do arquivo temporário
                            register_shutdown_function(function() use ($graficoInfo) {
                                if (file_exists($graficoInfo['file'])) {
                                    @unlink($graficoInfo['file']);
                                }
                            });
                        } else {
                            \Log::warning('Não foi possível gerar a imagem do gráfico', [
                                'graficoInfo' => $graficoInfo
                            ]);
                        }
                    } else {
                        \Log::warning('Dados do gráfico vazios', [
                            'labels_empty' => empty($this->graficoLabels),
                            'dados_empty' => empty($this->graficoDados)
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::error('Erro ao inserir gráfico como imagem: ' . $e->getMessage(), [
                        'trace' => $e->getTraceAsString()
                    ]);
                }
            },
        ];
    }

    /**
     * Gerar gráfico de barras usando GD Library
     */
    private function gerarGraficoGD()
    {
        try {
            // Validar dados
            if (empty($this->graficoDados) || max($this->graficoDados) == 0) {
                \Log::warning('Dados do gráfico inválidos ou zerados');
                return null;
            }

            // Para período diário, mostrar TODOS os dados
            $numDados = count($this->graficoDados);
            $dados = $this->graficoDados;
            $labels = $this->graficoLabels;
            $reduzido = false;

            // Apenas reduzir se NÃO for diário e houver muitos dados
            if ($this->periodo !== 'diario' && $numDados > 50) {
                $step = ceil($numDados / 50);
                $dadosReduzidos = [];
                $labelsReduzidos = [];
                for ($i = 0; $i < $numDados; $i += $step) {
                    $dadosReduzidos[] = $this->graficoDados[$i];
                    $labelsReduzidos[] = $this->graficoLabels[$i];
                }
                $dados = $dadosReduzidos;
                $labels = $labelsReduzidos;
                $reduzido = true;
            }

            // Dimensões do gráfico (ajustadas dinamicamente para melhor qualidade)
            $numBars = count($dados);
            $barMinWidth = 25;
            $barSpacing = 8;

            // Largura baseada no número de barras
            $width = max(1200, $numBars * ($barMinWidth + $barSpacing) + 160);
            $height = 600;
            $marginLeft = 100;
            $marginRight = 60;
            $marginTop = 100;
            $marginBottom = 150;

            // Criar imagem em alta qualidade
            $image = imagecreatetruecolor($width, $height);
            if (!$image) {
                \Log::error('Falha ao criar imagem GD');
                return null;
            }

            // Ativar anti-aliasing para melhor qualidade
            imageantialias($image, true);

            // Cores
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $blue = imagecolorallocate($image, 37, 99, 235); // Azul mais vibrante
            $lightBlue = imagecolorallocatealpha($image, 96, 165, 250, 40); // Transparência melhorada
            $gray = imagecolorallocate($image, 229, 231, 235);
            $darkGray = imagecolorallocate($image, 75, 85, 99);
            $mediumGray = imagecolorallocate($image, 156, 163, 175);

            // Preencher fundo branco
            imagefill($image, 0, 0, $white);

            // Título
            $chartTitle = ($this->tipo === 'precipitacao' ? 'Precipitação' : ($this->tipo === 'cotas' ? 'Cotas de Barragens' : 'Níveis Hidrométricos')) . ' - ' . ucfirst($this->periodo);
            $fontPath = $this->getFontPath();

            \Log::info('Gerando gráfico', [
                'font_path' => $fontPath,
                'title' => $chartTitle,
                'dados_originais' => $numDados,
                'dados_mostrados' => count($dados)
            ]);

            if ($fontPath) {
                imagettftext($image, 18, 0, $marginLeft, 50, $black, $fontPath, $chartTitle);
                // Subtítulo se houver redução de dados
                if ($reduzido) {
                    $subtitle = "Mostrando " . count($dados) . " de {$numDados} registros";
                    imagettftext($image, 11, 0, $marginLeft, 75, $mediumGray, $fontPath, $subtitle);
                }
            } else {
                imagestring($image, 5, $marginLeft, 30, $chartTitle, $black);
                if ($reduzido) {
                    $subtitle = "Mostrando " . count($dados) . " de {$numDados}";
                    imagestring($image, 3, $marginLeft, 50, $subtitle, $mediumGray);
                }
            }

            // Calcular área do gráfico
            $chartWidth = $width - $marginLeft - $marginRight;
            $chartHeight = $height - $marginTop - $marginBottom;

            // Encontrar valor máximo para escala
            $maxValue = max($dados);
            if ($maxValue <= 0) {
                $maxValue = 1;
            }
            $maxValue = $maxValue * 1.2; // 20% a mais para margem

            // Desenhar linhas de grade horizontais e verticais
            $numLines = 6;
            for ($i = 0; $i <= $numLines; $i++) {
                $y = $marginTop + ($chartHeight / $numLines) * $i;
                imagesetthickness($image, 1);
                imageline($image, $marginLeft, $y, $width - $marginRight, $y, $gray);

                // Valor da escala no eixo Y
                $value = $maxValue - ($maxValue / $numLines) * $i;
                $valueText = number_format($value, 1, ',', '.');
                if ($fontPath) {
                    imagettftext($image, 11, 0, 10, $y + 5, $darkGray, $fontPath, $valueText);
                } else {
                    imagestring($image, 3, 10, $y - 5, $valueText, $darkGray);
                }
            }

            // Desenhar barras com espaçamento adequado
            $totalSpacing = $chartWidth * 0.15; // 15% para espaçamento
            $totalBarWidth = $chartWidth - $totalSpacing;
            $barWidth = $totalBarWidth / $numBars;
            $spacing = $totalSpacing / ($numBars + 1);

            foreach ($dados as $index => $value) {
                $barHeight = ($value / $maxValue) * $chartHeight;
                $x = $marginLeft + $spacing + ($index * ($barWidth + $spacing));
                $y = $marginTop + $chartHeight - $barHeight;

                // Desenhar barra com gradiente e borda
                imagefilledrectangle($image, $x, $y, $x + $barWidth, $marginTop + $chartHeight, $lightBlue);
                imagesetthickness($image, 2);
                imagerectangle($image, $x, $y, $x + $barWidth, $marginTop + $chartHeight, $blue);
                imagesetthickness($image, 1);

                // Label do eixo X (vertical para melhor legibilidade)
                $label = $labels[$index] ?? '';
                // Truncar label se muito longo
                if (strlen($label) > 12) {
                    $label = substr($label, 0, 10) . '..';
                }

                $labelX = $x + ($barWidth / 2);
                $labelY = $marginTop + $chartHeight + 25;

                if ($fontPath) {
                    // Rotacionar 90 graus para período diário, 45 graus para outros
                    $angle = ($this->periodo === 'diario') ? 90 : 65;
                    imagettftext($image, 10, $angle, $labelX, $labelY, $black, $fontPath, $label);
                } else {
                    imagestringup($image, 3, $labelX, $labelY, $label, $black);
                }

                // Valor acima da barra (sempre mostrar para diário)
                if ($this->periodo === 'diario' || $barHeight > 25) {
                    $valueText = number_format($value, 1, ',', '.');
                    $textX = $x + ($barWidth / 2) - 15;
                    $textY = max($y - 10, $marginTop + 12);

                    if ($fontPath) {
                        imagettftext($image, 10, 0, $textX, $textY, $darkGray, $fontPath, $valueText);
                    } else {
                        imagestring($image, 2, $textX, $textY - 10, $valueText, $darkGray);
                    }
                }
            }

            // Eixos principais (mais grossos)
            imagesetthickness($image, 3);
            imageline($image, $marginLeft, $marginTop, $marginLeft, $marginTop + $chartHeight, $black);
            imageline($image, $marginLeft, $marginTop + $chartHeight, $width - $marginRight, $marginTop + $chartHeight, $black);
            imagesetthickness($image, 1);

            // Salvar em arquivo temporário com máxima qualidade
            $tempFile = tempnam(sys_get_temp_dir(), 'chart_') . '.png';
            $success = imagepng($image, $tempFile, 0); // Qualidade máxima (0 = sem compressão)
            imagedestroy($image);

            if ($success && file_exists($tempFile)) {
                \Log::info('Imagem PNG criada com sucesso', [
                    'path' => $tempFile,
                    'size' => filesize($tempFile),
                    'dimensions' => "{$width}x{$height}"
                ]);
                return [
                    'file' => $tempFile,
                    'width' => $width,
                    'height' => $height
                ];
            } else {
                \Log::error('Falha ao salvar PNG');
                return null;
            }
        } catch (\Exception $e) {
            \Log::error('Erro ao gerar gráfico GD: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Obter caminho da fonte TTF
     */
    private function getFontPath()
    {
        $possiblePaths = [
            '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf',
            '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',
            '/System/Library/Fonts/Helvetica.ttc',
            '/System/Library/Fonts/Supplemental/Arial.ttf',
            'C:\\Windows\\Fonts\\arial.ttf',
        ];

        foreach ($possiblePaths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }

        return null;
    }
}
