<?php

namespace App\Helpers;

use Carbon\Carbon;

class HydrologicalYear
{
    /**
     * Obtém o ano hidrológico atual
     * Ano hidrológico: Outubro a Setembro
     * Se estamos em Jan-Set = ano hidrológico começou no ano anterior
     * Se estamos em Out-Dez = ano hidrológico começou neste ano
     */
    public static function getCurrentHydrologicalYear()
    {
        $now = Carbon::now();
        
        // Se estamos entre Janeiro e Setembro, o ano hidrológico começou no ano anterior
        if ($now->month >= 1 && $now->month <= 9) {
            return $now->year - 1;
        }
        
        // Se estamos entre Outubro e Dezembro, o ano hidrológico começou neste ano
        return $now->year;
    }
    
    /**
     * Obtém o ano hidrológico para uma data específica
     */
    public static function getHydrologicalYear(Carbon $date)
    {
        // Se estamos entre Janeiro e Setembro, o ano hidrológico começou no ano anterior
        if ($date->month >= 1 && $date->month <= 9) {
            return $date->year - 1;
        }
        
        // Se estamos entre Outubro e Dezembro, o ano hidrológico começou neste ano
        return $date->year;
    }
    
    /**
     * Obtém as datas de início e fim do ano hidrológico
     */
    public static function getHydrologicalYearRange($hydrologicalYear = null)
    {
        if ($hydrologicalYear === null) {
            $hydrologicalYear = self::getCurrentHydrologicalYear();
        }
        
        $start = Carbon::create($hydrologicalYear, 10, 1, 0, 0, 0); // 1º de Outubro
        $end = Carbon::create($hydrologicalYear + 1, 9, 30, 23, 59, 59); // 30 de Setembro do ano seguinte
        
        return [
            'start' => $start,
            'end' => $end,
            'year' => $hydrologicalYear,
            'label' => $hydrologicalYear . '/' . ($hydrologicalYear + 1)
        ];
    }
    
    /**
     * Obtém lista dos últimos anos hidrológicos
     */
    public static function getRecentHydrologicalYears($count = 5)
    {
        $currentYear = self::getCurrentHydrologicalYear();
        $years = [];
        
        for ($i = 0; $i < $count; $i++) {
            $year = $currentYear - $i;
            $range = self::getHydrologicalYearRange($year);
            $years[] = [
                'year' => $year,
                'label' => $range['label'],
                'start' => $range['start'],
                'end' => $range['end'],
                'is_current' => $i === 0
            ];
        }
        
        return $years;
    }
    
    /**
     * Verifica se uma data está no ano hidrológico atual
     */
    public static function isCurrentHydrologicalYear(Carbon $date)
    {
        $currentYear = self::getCurrentHydrologicalYear();
        $range = self::getHydrologicalYearRange($currentYear);
        
        return $date->between($range['start'], $range['end']);
    }
    
    /**
     * Obtém o período atual do ano hidrológico (estação seca/chuvosa)
     */
    public static function getCurrentSeason()
    {
        $now = Carbon::now();
        
        // Estação chuvosa: Outubro a Março
        if (($now->month >= 10 && $now->month <= 12) || ($now->month >= 1 && $now->month <= 3)) {
            return [
                'season' => 'chuvosa',
                'label' => 'Estação Chuvosa',
                'months' => 'Outubro - Março',
                'color' => 'blue'
            ];
        }
        
        // Estação seca: Abril a Setembro
        return [
            'season' => 'seca',
            'label' => 'Estação Seca', 
            'months' => 'Abril - Setembro',
            'color' => 'orange'
        ];
    }
    
    /**
     * Formata o ano hidrológico para exibição
     */
    public static function formatHydrologicalYear($year)
    {
        return $year . '/' . ($year + 1);
    }
    
    /**
     * Obtém estatísticas do ano hidrológico atual vs anterior
     */
    public static function getYearComparison()
    {
        $currentYear = self::getCurrentHydrologicalYear();
        $previousYear = $currentYear - 1;
        
        return [
            'current' => self::getHydrologicalYearRange($currentYear),
            'previous' => self::getHydrologicalYearRange($previousYear),
            'progress' => self::getYearProgress()
        ];
    }
    
    /**
     * Calcula o progresso do ano hidrológico atual (0-100%)
     */
    public static function getYearProgress()
    {
        $range = self::getHydrologicalYearRange();
        $now = Carbon::now();
        
        // Se ainda não começou o ano hidrológico
        if ($now->lt($range['start'])) {
            return 0;
        }
        
        // Se já terminou o ano hidrológico
        if ($now->gt($range['end'])) {
            return 100;
        }
        
        // Calcular progresso
        $totalDays = $range['start']->diffInDays($range['end']);
        $elapsedDays = $range['start']->diffInDays($now);
        
        return round(($elapsedDays / $totalDays) * 100, 1);
    }
}