<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DivisaoAdministrativa;
use App\Models\DivisaoRolePermission;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

/**
 * Controller para gestao de permissoes customizadas por divisao
 */
class DivisaoPermissionController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'permission:configurar-sistema']);
    }

    /**
     * Listar divisoes com permissoes customizadas
     */
    public function index()
    {
        $divisoes = DivisaoAdministrativa::ativo()
            ->withCount('rolePermissions')
            ->orderBy('tipo')
            ->orderBy('nome')
            ->paginate(20);

        return view('admin.divisoes.permissions.index', compact('divisoes'));
    }

    /**
     * Ver permissoes customizadas de uma divisao
     */
    public function show(DivisaoAdministrativa $divisao)
    {
        $overrides = $divisao->rolePermissions()
            ->with(['role', 'permission'])
            ->get()
            ->groupBy('role_id');

        $roles = Role::orderBy('name')->get();
        $permissions = Permission::orderBy('name')->get();

        return view('admin.divisoes.permissions.show', compact('divisao', 'overrides', 'roles', 'permissions'));
    }

    /**
     * Adicionar override de permissao
     */
    public function store(Request $request, DivisaoAdministrativa $divisao)
    {
        $validated = $request->validate([
            'role_id' => 'required|exists:roles,id',
            'permission_id' => 'required|exists:permissions,id',
            'tipo' => 'required|in:grant,revoke',
        ]);

        DivisaoRolePermission::updateOrCreate(
            [
                'divisao_administrativa_id' => $divisao->id,
                'role_id' => $validated['role_id'],
                'permission_id' => $validated['permission_id'],
            ],
            ['tipo' => $validated['tipo']]
        );

        $role = Role::find($validated['role_id']);
        $permission = Permission::find($validated['permission_id']);
        $accao = $validated['tipo'] === 'grant' ? 'concedida' : 'revogada';

        return redirect()
            ->route('admin.divisoes.permissions.show', $divisao)
            ->with('success', "Permissao '{$permission->name}' {$accao} para '{$role->name}' nesta divisao.");
    }

    /**
     * Remover override de permissao
     */
    public function destroy(DivisaoAdministrativa $divisao, DivisaoRolePermission $override)
    {
        $override->delete();

        return redirect()
            ->route('admin.divisoes.permissions.show', $divisao)
            ->with('success', 'Override de permissao removido com sucesso!');
    }

    /**
     * Adicionar multiplos overrides de uma vez
     */
    public function bulkStore(Request $request, DivisaoAdministrativa $divisao)
    {
        $validated = $request->validate([
            'role_id' => 'required|exists:roles,id',
            'permissions' => 'required|array|min:1',
            'permissions.*' => 'exists:permissions,id',
            'tipo' => 'required|in:grant,revoke',
        ]);

        foreach ($validated['permissions'] as $permissionId) {
            DivisaoRolePermission::updateOrCreate(
                [
                    'divisao_administrativa_id' => $divisao->id,
                    'role_id' => $validated['role_id'],
                    'permission_id' => $permissionId,
                ],
                ['tipo' => $validated['tipo']]
            );
        }

        $role = Role::find($validated['role_id']);
        $accao = $validated['tipo'] === 'grant' ? 'concedidas' : 'revogadas';
        $count = count($validated['permissions']);

        return redirect()
            ->route('admin.divisoes.permissions.show', $divisao)
            ->with('success', "{$count} permissao(oes) {$accao} para '{$role->name}' nesta divisao.");
    }

    /**
     * Copiar permissoes de uma divisao para outra
     */
    public function copy(Request $request, DivisaoAdministrativa $divisao)
    {
        $validated = $request->validate([
            'target_divisao_id' => 'required|exists:divisoes_administrativas,id|different:' . $divisao->id,
            'overwrite' => 'boolean',
        ]);

        $targetDivisao = DivisaoAdministrativa::find($validated['target_divisao_id']);

        // Se overwrite, limpar permissoes existentes
        if ($request->boolean('overwrite')) {
            $targetDivisao->rolePermissions()->delete();
        }

        // Copiar permissoes
        $copied = 0;
        foreach ($divisao->rolePermissions as $override) {
            $exists = DivisaoRolePermission::where([
                'divisao_administrativa_id' => $targetDivisao->id,
                'role_id' => $override->role_id,
                'permission_id' => $override->permission_id,
            ])->exists();

            if (!$exists) {
                DivisaoRolePermission::create([
                    'divisao_administrativa_id' => $targetDivisao->id,
                    'role_id' => $override->role_id,
                    'permission_id' => $override->permission_id,
                    'tipo' => $override->tipo,
                ]);
                $copied++;
            }
        }

        return redirect()
            ->route('admin.divisoes.permissions.show', $targetDivisao)
            ->with('success', "{$copied} permissao(oes) copiada(s) de '{$divisao->nome}' para '{$targetDivisao->nome}'.");
    }
}
