<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

/**
 * Controller para gestao de Roles e Permissoes
 */
class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'permission:configurar-sistema']);
    }

    /**
     * Listar todos os roles com suas permissoes
     */
    public function index()
    {
        $roles = Role::with('permissions')->orderBy('name')->get();
        $permissions = Permission::orderBy('name')->get();

        // Agrupar permissoes por categoria
        $permissoesPorCategoria = $permissions->groupBy(function ($permission) {
            $partes = explode('-', $permission->name);
            return $partes[0] ?? 'outros';
        });

        return view('admin.roles.index', compact('roles', 'permissions', 'permissoesPorCategoria'));
    }

    /**
     * Formulario de criacao de role
     */
    public function create()
    {
        $permissions = Permission::orderBy('name')->get();

        $permissoesPorCategoria = $permissions->groupBy(function ($permission) {
            $partes = explode('-', $permission->name);
            return $partes[0] ?? 'outros';
        });

        return view('admin.roles.create', compact('permissions', 'permissoesPorCategoria'));
    }

    /**
     * Criar novo role
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:roles,name',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::create(['name' => $validated['name']]);

        if (!empty($validated['permissions'])) {
            $permissions = Permission::whereIn('id', $validated['permissions'])->get();
            $role->syncPermissions($permissions);
        }

        return redirect()
            ->route('admin.roles.index')
            ->with('success', "Role '{$role->name}' criado com sucesso!");
    }

    /**
     * Ver detalhes do role
     */
    public function show(Role $role)
    {
        $role->load('permissions', 'users');

        $permissions = Permission::orderBy('name')->get();
        $permissoesPorCategoria = $permissions->groupBy(function ($permission) {
            $partes = explode('-', $permission->name);
            return $partes[0] ?? 'outros';
        });

        return view('admin.roles.show', compact('role', 'permissions', 'permissoesPorCategoria'));
    }

    /**
     * Formulario de edicao de role
     */
    public function edit(Role $role)
    {
        $permissions = Permission::orderBy('name')->get();

        $permissoesPorCategoria = $permissions->groupBy(function ($permission) {
            $partes = explode('-', $permission->name);
            return $partes[0] ?? 'outros';
        });

        return view('admin.roles.edit', compact('role', 'permissions', 'permissoesPorCategoria'));
    }

    /**
     * Actualizar role
     */
    public function update(Request $request, Role $role)
    {
        // Nao permitir editar role Administrador
        if ($role->name === 'Administrador') {
            return back()->with('error', 'O role Administrador nao pode ser editado.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:100|unique:roles,name,' . $role->id,
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->update(['name' => $validated['name']]);

        if (isset($validated['permissions'])) {
            $permissions = Permission::whereIn('id', $validated['permissions'])->get();
            $role->syncPermissions($permissions);
        } else {
            $role->syncPermissions([]);
        }

        return redirect()
            ->route('admin.roles.index')
            ->with('success', "Role '{$role->name}' actualizado com sucesso!");
    }

    /**
     * Eliminar role
     */
    public function destroy(Role $role)
    {
        // Nao permitir eliminar roles padrao
        $rolesProtegidos = ['Administrador', 'Tecnico Senior', 'Tecnico', 'Operador', 'Visualizador'];

        if (in_array($role->name, $rolesProtegidos)) {
            return back()->with('error', 'Este role nao pode ser eliminado.');
        }

        // Verificar se tem utilizadores
        if ($role->users()->count() > 0) {
            return back()->with('error', 'Este role tem utilizadores atribuidos. Remova-os primeiro.');
        }

        $roleName = $role->name;
        $role->delete();

        return redirect()
            ->route('admin.roles.index')
            ->with('success', "Role '{$roleName}' eliminado com sucesso!");
    }

    /**
     * Matriz visual de permissoes
     */
    public function matrix()
    {
        $roles = Role::with('permissions')->orderBy('name')->get();
        $permissions = Permission::orderBy('name')->get();

        // Agrupar permissoes por categoria
        $permissoesPorCategoria = $permissions->groupBy(function ($permission) {
            $partes = explode('-', $permission->name);
            return $partes[0] ?? 'outros';
        });

        return view('admin.roles.matrix', compact('roles', 'permissions', 'permissoesPorCategoria'));
    }

    /**
     * Actualizar permissao via AJAX (para matriz)
     */
    public function togglePermission(Request $request, Role $role)
    {
        // Nao permitir editar Administrador
        if ($role->name === 'Administrador') {
            return response()->json(['error' => 'Role Administrador nao pode ser editado'], 403);
        }

        $validated = $request->validate([
            'permission_id' => 'required|exists:permissions,id',
            'action' => 'required|in:grant,revoke',
        ]);

        $permission = Permission::find($validated['permission_id']);

        if ($validated['action'] === 'grant') {
            $role->givePermissionTo($permission);
        } else {
            $role->revokePermissionTo($permission);
        }

        return response()->json([
            'success' => true,
            'message' => $validated['action'] === 'grant'
                ? "Permissao '{$permission->name}' concedida"
                : "Permissao '{$permission->name}' revogada"
        ]);
    }
}
