<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserDivisao;
use App\Models\DivisaoAdministrativa;
use App\Services\RegionalAccessService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Spatie\Permission\Models\Role;

/**
 * Controller para gestao de utilizadores
 */
class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'permission:gerir-usuarios']);
    }

    /**
     * Listar todos os utilizadores
     */
    public function index(Request $request)
    {
        $query = User::with(['roles', 'divisoesAtivas.divisao', 'divisaoPrincipal']);

        // Pesquisa
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('cargo', 'like', "%{$search}%");
            });
        }

        // Filtro por role
        if ($request->filled('role')) {
            $query->role($request->role);
        }

        // Filtro por status
        if ($request->filled('ativo')) {
            $query->where('ativo', $request->ativo === 'sim');
        }

        // Filtro por divisao
        if ($request->filled('divisao_id')) {
            $query->whereHas('divisoesAtivas', function ($q) use ($request) {
                $q->where('divisao_administrativa_id', $request->divisao_id);
            });
        }

        $users = $query->orderBy('name')->paginate(20);
        $roles = Role::orderBy('name')->get();
        $divisoes = DivisaoAdministrativa::ativo()
            ->orderBy('tipo')
            ->orderBy('nome')
            ->get()
            ->groupBy('tipo');

        return view('admin.usuarios.index', compact('users', 'roles', 'divisoes'));
    }

    /**
     * Formulario de criacao
     */
    public function create()
    {
        $roles = Role::orderBy('name')->get();
        $divisoes = DivisaoAdministrativa::ativo()
            ->orderBy('tipo')
            ->orderBy('nome')
            ->get()
            ->groupBy('tipo');

        return view('admin.usuarios.create', compact('roles', 'divisoes'));
    }

    /**
     * Criar novo utilizador
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => ['required', Password::defaults()],
            'telefone' => 'nullable|string|max:20',
            'cargo' => 'nullable|string|max:100',
            'departamento' => 'nullable|string|max:100',
            'roles' => 'required|array|min:1',
            'roles.*' => 'exists:roles,id',
            'divisoes' => 'nullable|array',
            'divisoes.*' => 'exists:divisoes_administrativas,id',
            'divisao_principal_id' => 'nullable|exists:divisoes_administrativas,id',
            'acesso_global' => 'boolean',
            'ativo' => 'boolean',
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'telefone' => $validated['telefone'] ?? null,
            'cargo' => $validated['cargo'] ?? null,
            'departamento' => $validated['departamento'] ?? null,
            'divisao_principal_id' => $validated['divisao_principal_id'] ?? null,
            'acesso_global' => $request->boolean('acesso_global'),
            'ativo' => $request->boolean('ativo', true),
        ]);

        // Atribuir roles
        $roles = Role::whereIn('id', $validated['roles'])->get();
        $user->syncRoles($roles);

        // Atribuir divisoes
        if (!empty($validated['divisoes'])) {
            foreach ($validated['divisoes'] as $index => $divisaoId) {
                $user->divisoes()->attach($divisaoId, [
                    'nivel_acesso' => 'full',
                    'is_primary' => $index === 0,
                    'ativo' => true,
                ]);
            }
        }

        return redirect()
            ->route('admin.usuarios.show', $user)
            ->with('success', 'Utilizador criado com sucesso!');
    }

    /**
     * Ver detalhes do utilizador
     */
    public function show(User $user)
    {
        $user->load(['roles', 'divisoesAtivas.divisao', 'divisaoPrincipal']);

        return view('admin.usuarios.show', compact('user'));
    }

    /**
     * Formulario de edicao
     */
    public function edit(User $user)
    {
        $user->load(['roles', 'divisoesAtivas.divisao']);

        $roles = Role::orderBy('name')->get();
        $divisoes = DivisaoAdministrativa::ativo()
            ->orderBy('tipo')
            ->orderBy('nome')
            ->get()
            ->groupBy('tipo');

        return view('admin.usuarios.edit', compact('user', 'roles', 'divisoes'));
    }

    /**
     * Actualizar utilizador
     */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => ['nullable', Password::defaults()],
            'telefone' => 'nullable|string|max:20',
            'cargo' => 'nullable|string|max:100',
            'departamento' => 'nullable|string|max:100',
            'roles' => 'required|array|min:1',
            'roles.*' => 'exists:roles,id',
            'divisao_principal_id' => 'nullable|exists:divisoes_administrativas,id',
            'acesso_global' => 'boolean',
            'ativo' => 'boolean',
        ]);

        $userData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'telefone' => $validated['telefone'] ?? null,
            'cargo' => $validated['cargo'] ?? null,
            'departamento' => $validated['departamento'] ?? null,
            'divisao_principal_id' => $validated['divisao_principal_id'] ?? null,
            'acesso_global' => $request->boolean('acesso_global'),
            'ativo' => $request->boolean('ativo', true),
        ];

        if (!empty($validated['password'])) {
            $userData['password'] = Hash::make($validated['password']);
        }

        $user->update($userData);

        // Actualizar roles
        $roles = Role::whereIn('id', $validated['roles'])->get();
        $user->syncRoles($roles);

        // Limpar cache de acesso
        RegionalAccessService::clearUserCache($user->id);

        return redirect()
            ->route('admin.usuarios.show', $user)
            ->with('success', 'Utilizador actualizado com sucesso!');
    }

    /**
     * Eliminar utilizador
     */
    public function destroy(User $user)
    {
        // Nao permitir eliminar o proprio utilizador
        if ($user->id === auth()->id()) {
            return back()->with('error', 'Nao pode eliminar a sua propria conta.');
        }

        $user->delete();

        return redirect()
            ->route('admin.usuarios.index')
            ->with('success', 'Utilizador eliminado com sucesso!');
    }
}
