<?php

namespace App\Http\Controllers;

use App\Models\Alerta;
use Illuminate\Http\Request;

class AlertaController extends Controller
{
    public function index()
    {
        $query = Alerta::with(['barragem', 'estacao']);

        if (request('tipo')) {
            $query->where('tipo', request('tipo'));
        }

        if (request('nivel')) {
            $query->where('nivel', request('nivel'));
        }

        if (request('estado')) {
            $query->where('estado', request('estado'));
        }

        $alertas = $query->orderBy('created_at', 'desc')->paginate(20);

        $stats = [
            'total' => Alerta::count(),
            'ativos' => Alerta::where('estado', 'activo')->count(),
            'emergencia' => Alerta::where('estado', 'activo')->where('nivel', 'emergencia')->count(),
            'alerta' => Alerta::where('estado', 'activo')->where('nivel', 'alerta')->count(),
        ];

        return view('alertas.index', compact('alertas', 'stats'));
    }

    public function show(Alerta $alerta)
    {
        $alerta->load(['barragem', 'estacao']);
        return view('alertas.show', compact('alerta'));
    }

    public function resolve(Request $request, Alerta $alerta)
    {
        $validated = $request->validate([
            'acoes_tomadas' => 'required|string',
        ]);

        $alerta->resolver($validated['acoes_tomadas']);

        // Processar notificações de resolução
        app(\App\Services\NotificationService::class)->processarResolucaoAlerta($alerta);

        return redirect()->back()
            ->with('success', 'Alerta resolvido com sucesso!');
    }

    public function cancel(Request $request, Alerta $alerta)
    {
        $validated = $request->validate([
            'motivo' => 'required|string',
        ]);

        $alerta->cancelar($validated['motivo']);

        // Processar notificações de resolução (cancelamento é tratado como resolução)
        app(\App\Services\NotificationService::class)->processarResolucaoAlerta($alerta);

        return redirect()->back()
            ->with('success', 'Alerta cancelado com sucesso!');
    }

    public function getAtivos()
    {
        $alertas = Alerta::with(['barragem', 'estacao'])
            ->where('estado', 'activo')
            ->orderBy('nivel', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($alertas);
    }
}
