<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\EstatisticasService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class EstatisticasController extends Controller
{
    protected $estatisticasService;

    public function __construct(EstatisticasService $estatisticasService)
    {
        $this->estatisticasService = $estatisticasService;
    }

    /**
     * GET /api/estatisticas/precipitacao
     * Retorna médias aritméticas de precipitação
     */
    public function mediasPrecipitacao(Request $request): JsonResponse
    {
        $estacao_id = $request->get('estacao_id');
        $periodo = $request->get('periodo', '30d');
        $data_inicio = $request->get('data_inicio');
        $data_fim = $request->get('data_fim');

        $medias = $this->estatisticasService->calcularMediasPrecipitacao(
            $estacao_id,
            $periodo,
            $data_inicio,
            $data_fim
        );

        return response()->json([
            'success' => true,
            'data' => $medias,
            'periodo' => $periodo,
            'estacao_id' => $estacao_id
        ]);
    }

    /**
     * GET /api/estatisticas/cotas-barragens
     * Retorna médias aritméticas de cotas nas barragens
     */
    public function mediasCotasBarragens(Request $request): JsonResponse
    {
        $barragem_id = $request->get('barragem_id');
        $periodo = $request->get('periodo', '30d');
        $data_inicio = $request->get('data_inicio');
        $data_fim = $request->get('data_fim');

        $medias = $this->estatisticasService->calcularMediasCotasBarragens(
            $barragem_id,
            $periodo,
            $data_inicio,
            $data_fim
        );

        return response()->json([
            'success' => true,
            'data' => $medias,
            'periodo' => $periodo,
            'barragem_id' => $barragem_id
        ]);
    }

    /**
     * GET /api/estatisticas/niveis-estacoes
     * Retorna médias aritméticas de níveis nas estações
     */
    public function mediasNiveisEstacoes(Request $request): JsonResponse
    {
        $estacao_id = $request->get('estacao_id');
        $periodo = $request->get('periodo', '30d');
        $data_inicio = $request->get('data_inicio');
        $data_fim = $request->get('data_fim');

        $medias = $this->estatisticasService->calcularMediasNiveisEstacoes(
            $estacao_id,
            $periodo,
            $data_inicio,
            $data_fim
        );

        return response()->json([
            'success' => true,
            'data' => $medias,
            'periodo' => $periodo,
            'estacao_id' => $estacao_id
        ]);
    }

    /**
     * GET /api/estatisticas/resumo
     * Retorna um resumo consolidado de todas as estatísticas
     */
    public function resumo(Request $request): JsonResponse
    {
        $periodo = $request->get('periodo', '30d');
        $data_inicio = $request->get('data_inicio');
        $data_fim = $request->get('data_fim');

        $precipitacao = $this->estatisticasService->calcularMediasPrecipitacao(
            null,
            $periodo,
            $data_inicio,
            $data_fim
        );

        $cotas = $this->estatisticasService->calcularMediasCotasBarragens(
            null,
            $periodo,
            $data_inicio,
            $data_fim
        );

        $niveis = $this->estatisticasService->calcularMediasNiveisEstacoes(
            null,
            $periodo,
            $data_inicio,
            $data_fim
        );

        return response()->json([
            'success' => true,
            'data' => [
                'precipitacao' => $precipitacao,
                'cotas_barragens' => $cotas,
                'niveis_estacoes' => $niveis
            ],
            'periodo' => $periodo
        ]);
    }
}
