<?php

namespace App\Http\Controllers;

use App\Models\Boletim;
use App\Services\BoletimService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class BoletimController extends Controller
{
    protected $boletimService;

    public function __construct(BoletimService $boletimService)
    {
        $this->boletimService = $boletimService;
    }

    public function index()
    {
        $query = Boletim::with(['elaborador', 'aprovador']);

        if (request('tipo')) {
            $query->where('tipo', request('tipo'));
        }

        if (request('estado')) {
            $query->where('estado', request('estado'));
        }

        if (request('data_inicio') && request('data_fim')) {
            $query->whereBetween('data_emissao', [request('data_inicio'), request('data_fim')]);
        }

        $boletins = $query->orderBy('data_emissao', 'desc')->paginate(15);

        return view('boletins.index', compact('boletins'));
    }

    public function create()
    {
        return view('boletins.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'tipo' => 'required|in:diario,semanal,mensal,emergencia,hidrologico_regional,hidrologico_provincial,informativo_barragens',
            'data_inicio' => 'nullable|date',
            'data_fim' => 'nullable|date|after_or_equal:data_inicio',
            'provincia' => 'nullable|string|in:Nampula,Cabo Delgado,Niassa',
        ]);

        try {
            if ($validated['tipo'] === 'hidrologico_regional') {
                $boletim = $this->boletimService->gerarBoletimHidrologicoRegional(
                    $validated['data_inicio'] ?? now()->subDays(7)->toDateString(),
                    $validated['data_fim'] ?? now()->toDateString()
                );
            } elseif ($validated['tipo'] === 'hidrologico_provincial') {
                $boletim = $this->boletimService->gerarBoletimHidrologicoProvincial(
                    $validated['data_fim'] ?? now()->toDateString(),
                    $validated['provincia'] ?? 'Nampula'
                );
            } else {
                $boletim = $this->boletimService->gerarBoletimInformativoBarragens(
                    $validated['data_fim'] ?? now()->toDateString()
                );
            }

            return redirect()->route('boletins.show', $boletim)
                ->with('success', 'Boletim gerado com sucesso!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erro ao gerar boletim: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Boletim $boletim)
    {
        $boletim->load(['elaborador', 'aprovador']);
        return view('boletins.show', compact('boletim'));
    }

    public function edit(Boletim $boletim)
    {
        if (!$boletim->podeSerEditado()) {
            return redirect()->route('boletins.show', $boletim)
                ->with('error', 'Este boletim não pode ser editado.');
        }

        return view('boletins.edit', compact('boletim'));
    }

    public function approve(Boletim $boletim)
    {
        if (!$boletim->podeSerAprovado()) {
            return redirect()->back()
                ->with('error', 'Este boletim não pode ser aprovado.');
        }

        $boletim->update([
            'estado' => 'aprovado',
            'aprovado_por' => Auth::id(),
        ]);

        return redirect()->route('boletins.show', $boletim)
            ->with('success', 'Boletim aprovado com sucesso!');
    }

    public function publish(Boletim $boletim)
    {
        if (!$boletim->podeSerPublicado()) {
            return redirect()->back()
                ->with('error', 'Este boletim não pode ser publicado.');
        }

        try {
            // Gerar PDF
            $this->boletimService->gerarPDF($boletim);
            
            $boletim->update(['estado' => 'publicado']);

            return redirect()->route('boletins.show', $boletim)
                ->with('success', 'Boletim publicado com sucesso!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erro ao publicar boletim: ' . $e->getMessage());
        }
    }

    public function downloadPdf(Boletim $boletim)
    {
        if (!$boletim->arquivo_pdf || !Storage::disk('local')->exists($boletim->arquivo_pdf)) {
            return redirect()->back()
                ->with('error', 'Arquivo PDF não encontrado.');
        }

        return Storage::disk('local')->download($boletim->arquivo_pdf);
    }

    public function export(Boletim $boletim, $format)
    {
        try {
            switch ($format) {
                case 'pdf':
                    return $this->boletimService->exportPDF($boletim);
                case 'word':
                case 'excel':
                    // TODO: Reactivar após demo
                    // case 'word':
                    //     return $this->boletimService->exportWord($boletim);
                    // case 'excel':
                    //     return $this->boletimService->exportExcel($boletim);
                    return redirect()->back()
                        ->with('info', 'Funcionalidade em Desenvolvimento');
                default:
                    return redirect()->back()
                        ->with('error', 'Formato de exportação inválido.');
            }
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erro ao exportar boletim: ' . $e->getMessage());
        }
    }

    public function update(Request $request, Boletim $boletim)
    {
        if (!$boletim->podeSerEditado()) {
            return redirect()->route('boletins.show', $boletim)
                ->with('error', 'Este boletim não pode ser editado.');
        }

        $validated = $request->validate([
            'numero_edicao' => 'nullable|string|max:10',
            'data_emissao' => 'required|date',
            'periodo_inicio' => 'nullable|date',
            'periodo_fim' => 'nullable|date|after_or_equal:periodo_inicio',
            'situacao_prevalecente' => 'nullable|string',
            'previsao_hidrologica' => 'nullable|string',
            'alertas_precaucoes' => 'nullable|string',
            'estado' => 'required|in:rascunho,revisao',
        ]);

        $boletim->update($validated);

        return redirect()->route('boletins.show', $boletim)
            ->with('success', 'Boletim atualizado com sucesso!');
    }

    public function destroy(Boletim $boletim)
    {
        // Verificar se o boletim pode ser deletado (apenas rascunhos)
        if ($boletim->estado !== 'rascunho') {
            return redirect()->back()
                ->with('error', 'Apenas boletins em rascunho podem ser deletados.');
        }

        try {
            // Deletar arquivo PDF se existir
            if ($boletim->arquivo_pdf && Storage::disk('local')->exists($boletim->arquivo_pdf)) {
                Storage::disk('local')->delete($boletim->arquivo_pdf);
            }

            $boletim->delete();

            return redirect()->route('boletins.index')
                ->with('success', 'Boletim deletado com sucesso!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erro ao deletar boletim: ' . $e->getMessage());
        }
    }
}