<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ChartsController extends Controller
{
    public function index()
    {
        return view('charts.index');
    }

    public function barragens()
    {
        $barragens = Barragem::with(['ultima_leitura'])->get();
        return view('charts.barragens', compact('barragens'));
    }

    public function barragemDetail(Barragem $barragem)
    {
        $barragem->load(['bacia_hidrografica', 'ultima_leitura']);
        
        // Dados para gráfico de enchimento (últimos 30 dias)
        $enchimento_30dias = $barragem->leituras()
            ->where('data_leitura', '>=', now()->subDays(30))
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'percentagem_enchimento', 'cota_actual']);

        // Dados para gráfico de enchimento (último ano)
        $enchimento_ano = $barragem->leituras()
            ->where('data_leitura', '>=', now()->subYear())
            ->whereDay('data_leitura', 1) // Primeiro dia de cada mês
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'percentagem_enchimento', 'volume_actual']);

        // Comparação com anos anteriores
        $comparacao_anos = DB::table('leituras_barragens')
            ->select(
                DB::raw('YEAR(data_leitura) as ano'),
                DB::raw('MONTH(data_leitura) as mes'),
                DB::raw('AVG(percentagem_enchimento) as media_enchimento')
            )
            ->where('barragem_id', $barragem->id)
            ->where('data_leitura', '>=', now()->subYears(5))
            ->groupBy('ano', 'mes')
            ->orderBy('ano')
            ->orderBy('mes')
            ->get();

        return view('charts.barragem-detail', compact(
            'barragem',
            'enchimento_30dias',
            'enchimento_ano',
            'comparacao_anos'
        ));
    }

    public function estacoes()
    {
        $estacoes = Estacao::with(['ultima_leitura'])->get();
        return view('charts.estacoes', compact('estacoes'));
    }

    public function estacaoDetail(Estacao $estacao)
    {
        $estacao->load(['bacia_hidrografica', 'ultima_leitura']);
        
        // Dados para gráfico (últimos 30 dias)
        $leituras_30dias = $estacao->leituras()
            ->where('data_leitura', '>=', now()->subDays(30))
            ->orderBy('data_leitura')
            ->get();

        // Dados mensais (último ano)
        $leituras_ano = DB::table('leituras_estacoes')
            ->select(
                DB::raw('YEAR(data_leitura) as ano'),
                DB::raw('MONTH(data_leitura) as mes'),
                DB::raw('AVG(nivel_hidrometrico) as media_nivel'),
                DB::raw('SUM(precipitacao_mm) as total_precipitacao'),
                DB::raw('AVG(temperatura_max) as media_temp_max'),
                DB::raw('AVG(temperatura_min) as media_temp_min')
            )
            ->where('estacao_id', $estacao->id)
            ->where('data_leitura', '>=', now()->subYear())
            ->groupBy('ano', 'mes')
            ->orderBy('ano')
            ->orderBy('mes')
            ->get();

        return view('charts.estacao-detail', compact(
            'estacao',
            'leituras_30dias',
            'leituras_ano'
        ));
    }

    public function regional()
    {
        // Dados regionais consolidados
        $barragens_stats = Barragem::with(['ultima_leitura'])
            ->get()
            ->map(function ($barragem) {
                $ultima_leitura = $barragem->ultima_leitura;
                return [
                    'nome' => $barragem->nome,
                    'percentagem' => $ultima_leitura ? $ultima_leitura->percentagem_enchimento : 0,
                    'volume' => $ultima_leitura ? $ultima_leitura->volume_actual : 0,
                    'cota' => $ultima_leitura ? $ultima_leitura->cota_actual : 0,
                    'status' => $ultima_leitura ? $barragem->getStatusNivel($ultima_leitura->cota_actual) : 'unknown'
                ];
            });

        // Precipitação regional (últimos 30 dias)
        $precipitacao_regional = DB::table('leituras_estacoes')
            ->join('estacoes', 'leituras_estacoes.estacao_id', '=', 'estacoes.id')
            ->select(
                'leituras_estacoes.data_leitura',
                DB::raw('AVG(precipitacao_mm) as media_precipitacao')
            )
            ->where('estacoes.tipo', 'pluviometrica')
            ->where('leituras_estacoes.data_leitura', '>=', now()->subDays(30))
            ->groupBy('leituras_estacoes.data_leitura')
            ->orderBy('leituras_estacoes.data_leitura')
            ->get();

        return view('charts.regional', compact(
            'barragens_stats',
            'precipitacao_regional'
        ));
    }

    public function apiBarragemData(Barragem $barragem, Request $request)
    {
        $days = $request->get('days', 30);
        
        $data = $barragem->leituras()
            ->where('data_leitura', '>=', now()->subDays($days))
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'percentagem_enchimento', 'cota_actual', 'volume_actual']);

        return response()->json($data);
    }

    public function apiEstacaoData(Estacao $estacao, Request $request)
    {
        $days = $request->get('days', 30);
        
        $data = $estacao->leituras()
            ->where('data_leitura', '>=', now()->subDays($days))
            ->orderBy('data_leitura')
            ->get();

        return response()->json($data);
    }
}