<?php

namespace App\Http\Controllers;

use App\Services\TwilioSmsService;
use App\Models\User;
use App\Models\NotificacaoEnviada;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;

class ConfiguracoesController extends Controller
{
    protected $twilioService;

    public function __construct(TwilioSmsService $twilioService)
    {
        $this->twilioService = $twilioService;
    }

    /**
     * Página principal de configurações
     */
    public function index()
    {
        $configuracoes = [
            'geral' => [
                'nome_sistema' => config('app.name', 'Sistema Hidrológico'),
                'timezone' => config('app.timezone'),
                'locale' => config('app.locale'),
                'debug_mode' => config('app.debug'),
                'url_sistema' => config('app.url'),
            ],
            'twilio' => [
                'sid' => config('services.twilio.sid') ? '***configurado***' : null,
                'auth_token' => config('services.twilio.auth_token') ? '***configurado***' : null,
                'phone_number' => config('services.twilio.phone_number'),
                'sms_enabled' => config('services.notifications.emergency_sms_enabled', true),
            ],
            'email' => [
                'mailer' => config('mail.default'),
                'host' => config('mail.mailers.smtp.host'),
                'port' => config('mail.mailers.smtp.port'),
                'username' => config('mail.mailers.smtp.username'),
                'from_address' => config('mail.from.address'),
                'from_name' => config('mail.from.name'),
                'email_enabled' => config('services.notifications.alert_email_enabled', true),
            ],
            'alertas' => [
                'escalacao_enabled' => config('services.notifications.escalation_enabled', true),
                'tempo_escalacao' => config('services.notifications.escalation_time_minutes', 30),
                'horario_relatorio' => config('services.notifications.daily_report_time', '08:00'),
                'auto_resolve_time' => config('alertas.auto_resolve_time', 24),
            ],
            'sistema' => [
                'backup_enabled' => config('backup.enabled', false),
                'backup_frequency' => config('backup.frequency', 'daily'),
                'log_level' => config('logging.default'),
                'cache_driver' => config('cache.default'),
                'queue_driver' => config('queue.default'),
            ]
        ];

        $stats = $this->getSystemStats();

        return view('configuracoes.index', compact('configuracoes', 'stats'));
    }

    /**
     * Configurações de SMS/Twilio
     */
    public function sms()
    {
        $config = [
            'sid' => config('services.twilio.sid'),
            'auth_token' => config('services.twilio.auth_token'),
            'phone_number' => config('services.twilio.phone_number'),
            'sms_enabled' => config('services.notifications.emergency_sms_enabled', true),
        ];

        $stats = [
            'sms_enviados_hoje' => NotificacaoEnviada::tipo('sms')->hoje()->count(),
            'sms_enviados_mes' => NotificacaoEnviada::tipo('sms')->whereMonth('created_at', now()->month)->count(),
            'taxa_entrega' => $this->calcularTaxaEntregaSMS(),
            'usuarios_com_sms' => User::where('notificacao_sms', true)->where('telefone', '!=', null)->count(),
        ];

        return view('configuracoes.sms', compact('config', 'stats'));
    }

    /**
     * Atualizar configurações SMS
     */
    public function updateSms(Request $request)
    {
        $request->validate([
            'twilio_sid' => 'required|string',
            'twilio_auth_token' => 'required|string',
            'twilio_phone_number' => 'required|string',
            'sms_enabled' => 'boolean',
        ]);

        $this->updateEnvFile([
            'TWILIO_SID' => $request->twilio_sid,
            'TWILIO_AUTH_TOKEN' => $request->twilio_auth_token,
            'TWILIO_PHONE_NUMBER' => $request->twilio_phone_number,
            'EMERGENCY_SMS_ENABLED' => $request->sms_enabled ? 'true' : 'false',
        ]);

        // Limpar cache de configuração
        Artisan::call('config:clear');

        return redirect()->back()->with('success', 'Configurações SMS atualizadas com sucesso!');
    }

    /**
     * Testar configuração SMS
     */
    public function testSms(Request $request)
    {
        $request->validate([
            'telefone_teste' => 'required|string',
        ]);

        try {
            $resultado = $this->twilioService->testConnection($request->telefone_teste);
            
            if ($resultado['success']) {
                return redirect()->back()->with('success', 'SMS de teste enviado com sucesso!');
            } else {
                return redirect()->back()->withErrors(['teste' => 'Falha no teste: ' . $resultado['error']]);
            }
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['teste' => 'Erro no teste: ' . $e->getMessage()]);
        }
    }

    /**
     * Configurações de Email
     */
    public function email()
    {
        $config = [
            'mailer' => config('mail.default'),
            'host' => config('mail.mailers.smtp.host'),
            'port' => config('mail.mailers.smtp.port'),
            'username' => config('mail.mailers.smtp.username'),
            'password' => config('mail.mailers.smtp.password') ? '***configurado***' : null,
            'encryption' => config('mail.mailers.smtp.encryption'),
            'from_address' => config('mail.from.address'),
            'from_name' => config('mail.from.name'),
            'email_enabled' => config('services.notifications.alert_email_enabled', true),
        ];

        $stats = [
            'emails_enviados_hoje' => NotificacaoEnviada::tipo('email')->hoje()->count(),
            'emails_enviados_mes' => NotificacaoEnviada::tipo('email')->whereMonth('created_at', now()->month)->count(),
            'usuarios_com_email' => User::where('notificacao_email', true)->count(),
        ];

        return view('configuracoes.email', compact('config', 'stats'));
    }

    /**
     * Atualizar configurações de Email
     */
    public function updateEmail(Request $request)
    {
        $request->validate([
            'mail_host' => 'required|string',
            'mail_port' => 'required|integer',
            'mail_username' => 'required|string',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|string|in:tls,ssl',
            'mail_from_address' => 'required|email',
            'mail_from_name' => 'required|string',
            'email_enabled' => 'boolean',
        ]);

        $envVars = [
            'MAIL_HOST' => $request->mail_host,
            'MAIL_PORT' => $request->mail_port,
            'MAIL_USERNAME' => $request->mail_username,
            'MAIL_ENCRYPTION' => $request->mail_encryption,
            'MAIL_FROM_ADDRESS' => $request->mail_from_address,
            'MAIL_FROM_NAME' => '"' . $request->mail_from_name . '"',
            'ALERT_EMAIL_ENABLED' => $request->email_enabled ? 'true' : 'false',
        ];

        if ($request->filled('mail_password')) {
            $envVars['MAIL_PASSWORD'] = $request->mail_password;
        }

        $this->updateEnvFile($envVars);

        Artisan::call('config:clear');

        return redirect()->back()->with('success', 'Configurações de email atualizadas com sucesso!');
    }

    /**
     * Configurações de Alertas
     */
    public function alertas()
    {
        $config = [
            'escalacao_enabled' => config('services.notifications.escalation_enabled', true),
            'tempo_escalacao' => config('services.notifications.escalation_time_minutes', 30),
            'horario_relatorio' => config('services.notifications.daily_report_time', '08:00'),
            'auto_resolve_time' => config('alertas.auto_resolve_time', 24),
            'niveis_criticos' => ['alerta', 'emergencia'],
        ];

        $stats = [
            'alertas_ativos' => \App\Models\Alerta::where('estado', 'activo')->count(),
            'alertas_hoje' => \App\Models\Alerta::whereDate('created_at', today())->count(),
            'emergencias_mes' => \App\Models\Alerta::where('nivel', 'emergencia')->whereMonth('created_at', now()->month)->count(),
            'tempo_resposta_medio' => $this->calcularTempoRespostaAlerta(),
        ];

        return view('configuracoes.alertas', compact('config', 'stats'));
    }

    /**
     * Atualizar configurações de Alertas
     */
    public function updateAlertas(Request $request)
    {
        $request->validate([
            'escalacao_enabled' => 'boolean',
            'tempo_escalacao' => 'required|integer|min:1|max:480',
            'horario_relatorio' => 'required|date_format:H:i',
            'auto_resolve_time' => 'required|integer|min:1|max:168',
        ]);

        $this->updateEnvFile([
            'NOTIFICATION_ESCALATION_ENABLED' => $request->escalacao_enabled ? 'true' : 'false',
            'NOTIFICATION_ESCALATION_TIME' => $request->tempo_escalacao,
            'DAILY_REPORT_TIME' => $request->horario_relatorio,
            'ALERT_AUTO_RESOLVE_TIME' => $request->auto_resolve_time,
        ]);

        Artisan::call('config:clear');

        return redirect()->back()->with('success', 'Configurações de alertas atualizadas com sucesso!');
    }

    /**
     * Configurações do Sistema
     */
    public function sistema()
    {
        $config = [
            'nome_sistema' => config('app.name'),
            'timezone' => config('app.timezone'),
            'locale' => config('app.locale'),
            'debug_mode' => config('app.debug'),
            'cache_driver' => config('cache.default'),
            'queue_driver' => config('queue.default'),
            'log_level' => config('logging.default'),
        ];

        $stats = $this->getSystemStats();

        return view('configuracoes.sistema', compact('config', 'stats'));
    }

    /**
     * Atualizar configurações do Sistema
     */
    public function updateSistema(Request $request)
    {
        $request->validate([
            'app_name' => 'required|string|max:100',
            'app_timezone' => 'required|string',
            'app_locale' => 'required|string',
            'app_debug' => 'boolean',
        ]);

        $this->updateEnvFile([
            'APP_NAME' => '"' . $request->app_name . '"',
            'APP_TIMEZONE' => $request->app_timezone,
            'APP_LOCALE' => $request->app_locale,
            'APP_DEBUG' => $request->app_debug ? 'true' : 'false',
        ]);

        Artisan::call('config:clear');

        return redirect()->back()->with('success', 'Configurações do sistema atualizadas com sucesso!');
    }

    /**
     * Limpar caches do sistema
     */
    public function clearCaches()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');

            return redirect()->back()->with('success', 'Caches limpos com sucesso!');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['cache' => 'Erro ao limpar caches: ' . $e->getMessage()]);
        }
    }

    /**
     * Informações do sistema
     */
    public function info()
    {
        $info = [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'N/A',
            'database' => config('database.default'),
            'cache_driver' => config('cache.default'),
            'queue_driver' => config('queue.default'),
            'storage_disk' => config('filesystems.default'),
            'timezone' => config('app.timezone'),
            'locale' => config('app.locale'),
            'environment' => config('app.env'),
            'debug_mode' => config('app.debug') ? 'Ativado' : 'Desativado',
        ];

        return view('configuracoes.info', compact('info'));
    }

    /**
     * Atualizar arquivo .env
     */
    private function updateEnvFile(array $data)
    {
        $envFile = base_path('.env');
        $envContent = File::get($envFile);

        foreach ($data as $key => $value) {
            $pattern = "/^{$key}=.*/m";
            $replacement = "{$key}={$value}";

            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, $replacement, $envContent);
            } else {
                $envContent .= "\n{$replacement}";
            }
        }

        File::put($envFile, $envContent);
    }

    /**
     * Calcular taxa de entrega de SMS
     */
    private function calcularTaxaEntregaSMS()
    {
        $total = NotificacaoEnviada::tipo('sms')->whereMonth('created_at', now()->month)->count();
        $entregues = NotificacaoEnviada::tipo('sms')->whereMonth('created_at', now()->month)->where('status', 'entregue')->count();

        return $total > 0 ? round(($entregues / $total) * 100, 1) : 0;
    }

    /**
     * Calcular tempo médio de resposta a alertas
     */
    private function calcularTempoRespostaAlerta()
    {
        $alertasResolvidos = \App\Models\Alerta::whereNotNull('data_fim')
            ->whereMonth('created_at', now()->month)
            ->get();

        if ($alertasResolvidos->count() === 0) {
            return 0;
        }

        $tempoTotal = 0;
        foreach ($alertasResolvidos as $alerta) {
            $tempoTotal += $alerta->created_at->diffInMinutes($alerta->data_fim);
        }

        return round($tempoTotal / $alertasResolvidos->count());
    }

    /**
     * Obter estatísticas do sistema
     */
    private function getSystemStats()
    {
        return [
            'usuarios_total' => User::count(),
            'usuarios_ativos' => User::where('ativo', true)->count(),
            'barragens_total' => \App\Models\Barragem::count(),
            'estacoes_total' => \App\Models\Estacao::count(),
            'alertas_ativos' => \App\Models\Alerta::where('estado', 'activo')->count(),
            'leituras_hoje' => \App\Models\LeituraBarragem::whereDate('created_at', today())->count() +
                             \App\Models\LeituraEstacao::whereDate('created_at', today())->count(),
            'notificacoes_hoje' => NotificacaoEnviada::hoje()->count(),
            'uptime' => $this->getSystemUptime(),
        ];
    }

    /**
     * Obter uptime do sistema
     */
    private function getSystemUptime()
    {
        if (function_exists('sys_getloadavg')) {
            return 'Disponível';
        }
        return 'N/A';
    }
}