<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\Alerta;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use App\Models\BaciaHidrografica;
use App\Helpers\HydrologicalYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        // Informações do ano hidrológico
        $anoHidrologico = HydrologicalYear::getHydrologicalYearRange();
        $estacaoAtual = HydrologicalYear::getCurrentSeason();
        $progressoAno = HydrologicalYear::getYearProgress();

        // Estatísticas gerais
        $stats = [
            'total_barragens' => Barragem::where('estado', 'activa')->count(),
            'total_estacoes' => Estacao::where('estado', 'activa')->count(),
            'alertas_ativos' => Alerta::where('estado', 'activo')->count(),
            'leituras_hoje' => $this->getLeiturasDoDia(),
            'leituras_ano_hidrologico' => $this->getLeiturasAnoHidrologico(),
            'ano_hidrologico' => $anoHidrologico,
            'estacao_atual' => $estacaoAtual,
            'progresso_ano' => $progressoAno,
        ];

        // Dados para o mapa - filtrar coordenadas válidas para Norte de Moçambique
        $barragens_mapa = Barragem::whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->with('ultima_leitura')
            ->get()
            ->map(function ($barragem) {
                $ultima_leitura = $barragem->ultima_leitura;
                return [
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'tipo' => $barragem->tipo_barragem ?? 'Barragem',
                    'altura' => $barragem->altura_barragem,
                    'latitude' => $barragem->latitude,
                    'longitude' => $barragem->longitude,
                    'cota_atual' => $ultima_leitura?->cota_actual,
                    'percentagem_enchimento' => $ultima_leitura?->percentagem_enchimento,
                ];
            })
            ->filter(function ($barragem) {
                // Filtrar apenas coordenadas válidas para Norte de Moçambique
                $lat = $barragem['latitude'];
                $lng = $barragem['longitude'];
                return $lat >= -20 && $lat <= -10 && $lng >= 30 && $lng <= 45;
            });

        $estacoes_mapa = Estacao::whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->get()
            ->map(function ($estacao) {
                return [
                    'id' => $estacao->id,
                    'nome' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'tipo' => $estacao->tipo,
                    'estado' => $estacao->estado,
                    'provincia' => $estacao->provincia,
                    'latitude' => $estacao->latitude,
                    'longitude' => $estacao->longitude,
                ];
            })
            ->filter(function ($estacao) {
                // Filtrar apenas coordenadas válidas para Norte de Moçambique
                $lat = $estacao['latitude'];
                $lng = $estacao['longitude'];
                return $lat >= -20 && $lat <= -10 && $lng >= 30 && $lng <= 45;
            });

        // Para as bacias, vamos usar coordenadas centrais aproximadas das províncias do norte de Moçambique
        $bacias_coords = [
            'Cabo Delgado' => ['latitude' => -11.27, 'longitude' => 40.51],
            'Nampula' => ['latitude' => -15.12, 'longitude' => 39.27],
            'Niassa' => ['latitude' => -13.27, 'longitude' => 36.64],
            'Tete' => ['latitude' => -16.16, 'longitude' => 33.59],
            'Zambézia' => ['latitude' => -17.72, 'longitude' => 36.89],
        ];

        $bacias_mapa = BaciaHidrografica::all()
            ->map(function ($bacia) use ($bacias_coords) {
                // Tenta encontrar coordenadas baseadas na província
                $coords = null;
                if ($bacia->provincias_abrangidas) {
                    $provincias = is_array($bacia->provincias_abrangidas)
                        ? $bacia->provincias_abrangidas
                        : explode(',', $bacia->provincias_abrangidas);
                    $primeira_provincia = trim($provincias[0] ?? '');
                    if (isset($bacias_coords[$primeira_provincia])) {
                        $coords = $bacias_coords[$primeira_provincia];
                        // Adiciona pequena variação para evitar sobreposição
                        $coords['latitude'] += (rand(-100, 100) / 1000);
                        $coords['longitude'] += (rand(-100, 100) / 1000);
                    }
                }

                return [
                    'id' => $bacia->id,
                    'nome' => $bacia->nome,
                    'codigo' => $bacia->codigo,
                    'area_km2' => $bacia->area_km2,
                    'provincias_abrangidas' => is_array($bacia->provincias_abrangidas)
                        ? implode(', ', $bacia->provincias_abrangidas)
                        : $bacia->provincias_abrangidas,
                    'latitude' => $coords['latitude'] ?? null,
                    'longitude' => $coords['longitude'] ?? null,
                ];
            })->filter(function ($bacia) {
                return $bacia['latitude'] && $bacia['longitude'];
            });

        $bacias_count = BaciaHidrografica::count();

        // Alertas ativos recentes (filtrar apenas os que têm barragem OU estação)
        $alertas_ativos = Alerta::with(['barragem', 'estacao'])
            ->where('estado', 'activo')
            ->where(function($query) {
                $query->whereNotNull('barragem_id')
                      ->orWhereNotNull('estacao_id');
            })
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        return view('dashboard', compact(
            'stats',
            'barragens_mapa',
            'estacoes_mapa',
            'bacias_mapa',
            'bacias_count',
            'alertas_ativos'
        ));
    }

    public function backup()
    {
        // Informações do ano hidrológico
        $anoHidrologico = HydrologicalYear::getHydrologicalYearRange();
        $estacaoAtual = HydrologicalYear::getCurrentSeason();
        $progressoAno = HydrologicalYear::getYearProgress();

        // Estatísticas gerais (filtradas pelo ano hidrológico atual)
        $stats = [
            'total_barragens' => Barragem::where('estado', 'activa')->count(),
            'total_estacoes' => Estacao::where('estado', 'activa')->count(),
            'alertas_ativos' => Alerta::where('estado', 'activo')->count(),
            'leituras_hoje' => $this->getLeiturasDoDia(),
            'leituras_ano_hidrologico' => $this->getLeiturasAnoHidrologico(),
            'ano_hidrologico' => $anoHidrologico,
            'estacao_atual' => $estacaoAtual,
            'progresso_ano' => $progressoAno,
        ];

        // Barragens principais com última leitura
        $barragens_principais = Barragem::with(['ultima_leitura', 'bacia_hidrografica'])
            ->whereIn('codigo', ['BAR-001', 'BAR-002', 'BAR-003', 'BAR-004', 'BAR-005', 'BAR-006'])
            ->get()
            ->map(function ($barragem) {
                $ultima_leitura = $barragem->ultima_leitura;
                return [
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'cota_atual' => $ultima_leitura?->cota_actual,
                    'percentagem_enchimento' => $ultima_leitura?->percentagem_enchimento,
                    'comparacao_ano_anterior' => $ultima_leitura?->comparacao_ano_anterior,
                    'status' => $ultima_leitura ? $barragem->getStatusNivel($ultima_leitura->cota_actual) : 'sem_dados',
                    'data_leitura' => $ultima_leitura?->data_leitura,
                    'bacia' => $barragem->bacia_hidrografica?->nome,
                ];
            });

        // Alertas ativos recentes (filtrar apenas os que têm barragem OU estação)
        $alertas_ativos = Alerta::with(['barragem', 'estacao'])
            ->where('estado', 'activo')
            ->where(function($query) {
                $query->whereNotNull('barragem_id')
                      ->orWhereNotNull('estacao_id');
            })
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        // Tendências dos últimos 7 dias
        $tendencias = $this->getTendenciasSemanais();

        return view('dashboard-backup', compact(
            'stats',
            'barragens_principais',
            'alertas_ativos',
            'tendencias'
        ));
    }

    public function getStats()
    {
        $stats = [
            'barragens' => [
                'total' => Barragem::count(),
                'ativas' => Barragem::where('estado', 'activa')->count(),
                'inativas' => Barragem::where('estado', 'inactiva')->count(),
                'manutencao' => Barragem::where('estado', 'manutencao')->count(),
            ],
            'estacoes' => [
                'total' => Estacao::count(),
                'hidrometricas' => Estacao::where('tipo', 'hidrometrica')->count(),
                'pluviometricas' => Estacao::where('tipo', 'pluviometrica')->count(),
                'meteorologicas' => Estacao::whereIn('tipo', ['meteorologica', 'climatologica'])->count(),
            ],
            'alertas' => [
                'ativos' => Alerta::where('estado', 'activo')->count(),
                'emergencia' => Alerta::where('estado', 'activo')->where('nivel', 'emergencia')->count(),
                'alerta' => Alerta::where('estado', 'activo')->where('nivel', 'alerta')->count(),
                'atencao' => Alerta::where('estado', 'activo')->where('nivel', 'atencao')->count(),
            ],
            'leituras' => [
                'hoje' => $this->getLeiturasDoDia(),
                'semana' => $this->getLeiturasUltimos7Dias(),
                'pendentes_validacao' => $this->getLeiturasParaValidacao(),
            ]
        ];

        return response()->json($stats);
    }

    private function getLeiturasDoDia()
    {
        $hoje = Carbon::today();
        
        $barragens = LeituraBarragem::whereDate('data_leitura', $hoje)->count();
        $estacoes = LeituraEstacao::whereDate('data_leitura', $hoje)->count();
        
        return $barragens + $estacoes;
    }

    private function getLeiturasUltimos7Dias()
    {
        $inicio = Carbon::today()->subDays(6);
        $fim = Carbon::today();
        
        $barragens = LeituraBarragem::whereBetween('data_leitura', [$inicio, $fim])->count();
        $estacoes = LeituraEstacao::whereBetween('data_leitura', [$inicio, $fim])->count();
        
        return $barragens + $estacoes;
    }

    private function getLeiturasParaValidacao()
    {
        $barragens = LeituraBarragem::where('validado', false)
            ->where('created_at', '>', Carbon::now()->subDays(7))
            ->count();
            
        $estacoes = LeituraEstacao::where('validado', false)
            ->where('created_at', '>', Carbon::now()->subDays(7))
            ->count();
        
        return $barragens + $estacoes;
    }

    private function getLeiturasAnoHidrologico()
    {
        try {
            $anoHidrologico = HydrologicalYear::getHydrologicalYearRange();

            $barragens = LeituraBarragem::whereBetween('data_leitura', [
                $anoHidrologico['start']->toDateString(),
                $anoHidrologico['end']->toDateString()
            ])->count();

            $estacoes = LeituraEstacao::whereBetween('data_leitura', [
                $anoHidrologico['start']->toDateString(),
                $anoHidrologico['end']->toDateString()
            ])->count();

            $total = $barragens + $estacoes;

            // Se não houver leituras no ano hidrológico atual, mostrar todas as leituras
            if ($total == 0) {
                $barragens = LeituraBarragem::count();
                $estacoes = LeituraEstacao::count();
                $total = $barragens + $estacoes;
            }

            return $total;
        } catch (\Exception $e) {
            // Fallback: retornar todas as leituras se houver erro
            $barragens = LeituraBarragem::count();
            $estacoes = LeituraEstacao::count();
            return $barragens + $estacoes;
        }
    }

    private function getTendenciasSemanais()
    {
        $dias = [];
        for ($i = 6; $i >= 0; $i--) {
            $data = Carbon::today()->subDays($i);
            
            $leituras_barragens = LeituraBarragem::whereDate('data_leitura', $data)->count();
            $leituras_estacoes = LeituraEstacao::whereDate('data_leitura', $data)->count();
            $alertas_criados = Alerta::whereDate('created_at', $data)->count();
            
            $dias[] = [
                'data' => $data->format('d/m'),
                'leituras' => $leituras_barragens + $leituras_estacoes,
                'alertas' => $alertas_criados,
            ];
        }
        
        return $dias;
    }
}
