<?php

namespace App\Http\Controllers;

use App\Models\Estacao;
use App\Models\BaciaHidrografica;
use App\Models\LeituraEstacao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EstacaoController extends Controller
{
    public function index()
    {
        $query = Estacao::with(['bacia_hidrografica', 'ultima_leitura', 'alertas']);

        if (request('search')) {
            $query->where(function($q) {
                $q->where('nome', 'like', '%' . request('search') . '%')
                  ->orWhere('codigo', 'like', '%' . request('search') . '%');
            });
        }

        if (request('tipo')) {
            $query->where('tipo', request('tipo'));
        }

        if (request('provincia')) {
            $query->where('provincia', request('provincia'));
        }

        if (request('estado')) {
            $query->where('estado', request('estado'));
        }

        if (request('bacia_id')) {
            $query->where('bacia_hidrografica_id', request('bacia_id'));
        }

        $estacoes = $query->orderBy('nome')->paginate(15);

        return view('estacoes.index', compact('estacoes'));
    }

    public function create()
    {
        $bacias = BaciaHidrografica::orderBy('nome')->get();
        return view('estacoes.create', compact('bacias'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:estacoes',
            'tipo' => 'required|in:hidrometrica,pluviometrica,meteorologica,climatologica',
            'bacia_hidrografica_id' => 'nullable|exists:bacias_hidrograficas,id',
            'divisao_administrativa_id' => 'required|exists:divisoes_administrativas,id',
            'rio' => 'nullable|string|max:100',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'altitude' => 'nullable|numeric|min:0',
            'nivel_alerta' => 'nullable|numeric|min:0',
            'data_instalacao' => 'nullable|date',
            'equipamento' => 'nullable|string|max:255',
            'frequencia_leitura' => 'required|in:horaria,diaria,6_horas,12_horas',
            'estado' => 'required|in:activa,inactiva,manutencao',
            'observacoes' => 'nullable|string',
        ]);

        Estacao::create($validated);

        return redirect()->route('estacoes.index')
            ->with('success', 'Estação criada com sucesso!');
    }

    public function show(Estacao $estacao)
    {
        $estacao->load(['bacia_hidrografica', 'ultima_leitura', 'alertas' => function($query) {
            $query->where('estado', 'activo')->orderBy('created_at', 'desc');
        }]);

        $leituras_recentes = $estacao->leituras()
            ->with('operador')
            ->orderBy('data_leitura', 'desc')
            ->take(10)
            ->get();

        $historico_niveis = $estacao->leituras()
            ->where('data_leitura', '>=', now()->subDays(30))
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'nivel_hidrometrico', 'precipitacao_mm']);

        return view('estacoes.show', compact('estacao', 'leituras_recentes', 'historico_niveis'));
    }

    public function edit(Estacao $estacao)
    {
        $bacias = BaciaHidrografica::orderBy('nome')->get();
        return view('estacoes.edit', compact('estacao', 'bacias'));
    }

    public function update(Request $request, Estacao $estacao)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:estacoes,codigo,' . $estacao->id,
            'tipo' => 'required|in:hidrometrica,pluviometrica,meteorologica,climatologica',
            'bacia_hidrografica_id' => 'nullable|exists:bacias_hidrograficas,id',
            'divisao_administrativa_id' => 'required|exists:divisoes_administrativas,id',
            'rio' => 'nullable|string|max:100',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'altitude' => 'nullable|numeric|min:0',
            'nivel_alerta' => 'nullable|numeric|min:0',
            'data_instalacao' => 'nullable|date',
            'equipamento' => 'nullable|string|max:255',
            'frequencia_leitura' => 'required|in:horaria,diaria,6_horas,12_horas',
            'estado' => 'required|in:activa,inactiva,manutencao',
            'observacoes' => 'nullable|string',
        ]);

        $estacao->update($validated);

        return redirect()->route('estacoes.index')
            ->with('success', 'Estação atualizada com sucesso!');
    }

    public function destroy(Estacao $estacao)
    {
        try {
            $estacao->delete();
            return redirect()->route('estacoes.index')
                ->with('success', 'Estação removida com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('estacoes.index')
                ->with('error', 'Erro ao remover estação. Verifique se não há dados dependentes.');
        }
    }

    public function leituras(Estacao $estacao)
    {
        $leituras = $estacao->leituras()
            ->with('operador')
            ->orderBy('data_leitura', 'desc')
            ->paginate(20);

        return view('estacoes.leituras', compact('estacao', 'leituras'));
    }

    public function leiturasGrid(Estacao $estacao, Request $request)
    {
        // Determinar período (padrão: mês actual)
        $mesParam = $request->input('mes', now()->format('Y-m'));
        $dataInicio = \Carbon\Carbon::createFromFormat('Y-m', $mesParam)->startOfMonth();
        $dataFim = $dataInicio->copy()->endOfMonth();

        // Buscar leituras do período
        $leituras = $estacao->leituras()
            ->whereBetween('data_leitura', [$dataInicio, $dataFim])
            ->orderBy('data_leitura')
            ->get()
            ->keyBy(function ($item) {
                return $item->data_leitura->format('Y-m-d');
            });

        // Gerar array de datas do mês
        $diasDoMes = [];
        $dataAtual = $dataInicio->copy();
        while ($dataAtual <= $dataFim) {
            $diasDoMes[] = $dataAtual->copy();
            $dataAtual->addDay();
        }

        return view('estacoes.leituras-grid', compact('estacao', 'leituras', 'diasDoMes', 'mesParam', 'dataInicio', 'dataFim'));
    }

    public function storeLeiturasBulk(Request $request, Estacao $estacao)
    {
        $leituras = $request->input('leituras', []);
        $salvos = 0;
        $erros = 0;

        foreach ($leituras as $leituraData) {
            if (empty($leituraData['data_leitura'])) continue;

            // Verificar se tem pelo menos um valor preenchido
            $temDados = !empty($leituraData['nivel_6h']) || !empty($leituraData['nivel_9h']) ||
                        !empty($leituraData['nivel_12h']) || !empty($leituraData['nivel_15h']) ||
                        !empty($leituraData['nivel_18h']) || !empty($leituraData['precipitacao_mm']);

            if (!$temDados) continue;

            try {
                $estacao->leituras()->updateOrCreate(
                    ['data_leitura' => $leituraData['data_leitura']],
                    array_filter([
                        'nivel_6h' => $leituraData['nivel_6h'] ?? null,
                        'nivel_9h' => $leituraData['nivel_9h'] ?? null,
                        'nivel_12h' => $leituraData['nivel_12h'] ?? null,
                        'nivel_15h' => $leituraData['nivel_15h'] ?? null,
                        'nivel_18h' => $leituraData['nivel_18h'] ?? null,
                        'cota_6h' => $leituraData['cota_6h'] ?? null,
                        'cota_9h' => $leituraData['cota_9h'] ?? null,
                        'cota_12h' => $leituraData['cota_12h'] ?? null,
                        'cota_15h' => $leituraData['cota_15h'] ?? null,
                        'cota_18h' => $leituraData['cota_18h'] ?? null,
                        'precipitacao_mm' => $leituraData['precipitacao_mm'] ?? null,
                        'operador_id' => auth()->id(),
                    ], fn($v) => $v !== null)
                );
                $salvos++;
            } catch (\Exception $e) {
                $erros++;
            }
        }

        return response()->json([
            'success' => true,
            'message' => "{$salvos} leituras salvas com sucesso" . ($erros > 0 ? " ({$erros} erros)" : ''),
            'salvos' => $salvos,
            'erros' => $erros
        ]);
    }

    public function createLeitura(Estacao $estacao)
    {
        return view('components.leitura-form-page', compact('estacao'));
    }

    public function showImportForm(Estacao $estacao)
    {
        return view('components.importacao-form-page', compact('estacao'));
    }

    public function storeLeitura(Request $request, Estacao $estacao)
    {
        $rules = [
            'data_leitura' => 'required|date',
            'observacoes' => 'nullable|string',
        ];

        if ($estacao->tipo === 'hidrometrica') {
            $rules = array_merge($rules, [
                'hora_6h' => 'nullable|date_format:H:i',
                'hora_12h' => 'nullable|date_format:H:i',
                'hora_18h' => 'nullable|date_format:H:i',
                'nivel_6h' => 'nullable|numeric|min:0',
                'nivel_12h' => 'nullable|numeric|min:0',
                'nivel_18h' => 'nullable|numeric|min:0',
            ]);

            // Se a estação trabalha com cotas, adicionar validação
            if ($estacao->trabalhaComCotas()) {
                $rules = array_merge($rules, [
                    'cota_6h' => 'nullable|numeric',
                    'cota_12h' => 'nullable|numeric',
                    'cota_18h' => 'nullable|numeric',
                    'cota_hidrometrica' => 'nullable|numeric',
                ]);
            }
        }

        if (in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica', 'evaporimetrica'])) {
            $rules = array_merge($rules, [
                'precipitacao_mm' => 'nullable|numeric|min:0',
                'temperatura_max' => 'nullable|numeric',
                'temperatura_min' => 'nullable|numeric',
                'humidade_relativa' => 'nullable|numeric|min:0|max:100',
                'evaporacao' => 'nullable|numeric|min:0',
            ]);
        }

        $validated = $request->validate($rules);
        $validated['estacao_id'] = $estacao->id;
        $validated['operador_id'] = Auth::id();

        $leitura = LeituraEstacao::create($validated);

        // Verificar alertas
        if ($estacao->tipo === 'hidrometrica') {
            \App\Models\Alerta::criarAlertaEstacao($estacao, $leitura);
        }

        return redirect()->route('estacoes.leituras', $estacao)
            ->with('success', 'Leitura registrada com sucesso!');
    }

    /**
     * API: Verificar se existe leitura para uma data específica
     */
    public function verificarLeitura(Estacao $estacao, $data)
    {
        $existe = $estacao->leituras()
            ->whereDate('data_leitura', $data)
            ->exists();

        $leitura = null;
        if ($existe) {
            $leitura = $estacao->leituras()
                ->whereDate('data_leitura', $data)
                ->with('operador')
                ->first();
        }

        return response()->json([
            'existe' => $existe,
            'leitura' => $leitura
        ]);
    }

    /**
     * API: Obter dados históricos para comparação
     */
    public function getDadosHistoricos(Estacao $estacao, $data)
    {
        $dataAtual = new \Carbon\Carbon($data);

        // Buscar dados dos últimos 5 anos para a mesma data (±15 dias)
        $historico = $estacao->leituras()
            ->whereMonth('data_leitura', $dataAtual->month)
            ->whereDay('data_leitura', '>=', max(1, $dataAtual->day - 15))
            ->whereDay('data_leitura', '<=', min(31, $dataAtual->day + 15))
            ->where('data_leitura', '<', $dataAtual)
            ->orderBy('data_leitura', 'desc')
            ->limit(100)
            ->get();

        $dados = [];
        foreach ($historico as $leitura) {
            if ($estacao->tipo === 'hidrometrica') {
                $valor = $leitura->nivel_hidrometrico;
            } elseif ($estacao->tipo === 'pluviometrica') {
                $valor = $leitura->precipitacao_mm;
            } else {
                $valor = ($leitura->temperatura_max + $leitura->temperatura_min) / 2;
            }

            if ($valor) {
                $dados[] = [
                    'data' => $leitura->data_leitura->format('Y-m-d'),
                    'valor' => $valor,
                    'ano' => $leitura->data_leitura->year
                ];
            }
        }

        return response()->json($dados);
    }

    /**
     * API: Importar leituras de arquivo Excel/CSV
     */
    public function importarLeituras(Request $request, Estacao $estacao)
    {
        $request->validate([
            'arquivo_importacao' => 'required|file|mimes:xlsx,xls,csv|max:10240', // 10MB
        ]);

        try {
            $arquivo = $request->file('arquivo_importacao');
            $extensao = strtolower($arquivo->getClientOriginalExtension());

            // Log para debug
            \Log::info("Importando arquivo: " . $arquivo->getClientOriginalName() . " (extensão: $extensao)");

            // Verificar se o arquivo é válido
            if (!$arquivo->isValid()) {
                throw new \Exception("Arquivo inválido ou corrompido");
            }

            // Verificar tamanho mínimo
            if ($arquivo->getSize() < 10) {
                throw new \Exception("Arquivo muito pequeno ou vazio");
            }

            $dados = [];
            if (in_array($extensao, ['xlsx', 'xls'])) {
                $dados = $this->lerArquivoExcel($arquivo);
            } elseif ($extensao === 'csv') {
                $dados = $this->lerArquivoCSV($arquivo);
            } else {
                throw new \Exception("Formato de arquivo não suportado: $extensao");
            }

            if (empty($dados)) {
                throw new \Exception("Nenhum dado encontrado no arquivo");
            }

            $importadas = 0;
            $atualizadas = 0;
            $erros = [];

            \DB::beginTransaction();

            foreach ($dados as $index => $registro) {
                try {
                    $resultado = $this->criarLeituraDeRegistro($estacao, $registro);
                    if ($resultado === 'criada') {
                        $importadas++;
                    } elseif ($resultado === 'atualizada') {
                        $atualizadas++;
                    }
                } catch (\Exception $e) {
                    $erros[] = "Linha " . ($index + 2) . ": " . $e->getMessage();

                    // Se muitos erros, interromper
                    if (count($erros) > 10) {
                        $erros[] = "Muitos erros encontrados. Importação interrompida.";
                        break;
                    }
                }
            }

            \DB::commit();

            $total = $importadas + $atualizadas;
            $message = "$total leituras processadas";
            if ($importadas > 0) $message .= " ($importadas novas";
            if ($atualizadas > 0) $message .= $importadas > 0 ? ", $atualizadas atualizadas)" : " ($atualizadas atualizadas)";
            if ($importadas > 0 && $atualizadas == 0) $message .= ")";

            return response()->json([
                'success' => true,
                'importadas' => $total,
                'novas' => $importadas,
                'atualizadas' => $atualizadas,
                'erros' => $erros,
                'total_erros' => count($erros),
                'message' => $message
            ]);

        } catch (\Exception $e) {
            \DB::rollback();

            \Log::error("Erro na importação: " . $e->getMessage(), [
                'estacao_id' => $estacao->id,
                'arquivo' => $arquivo ? $arquivo->getClientOriginalName() : 'N/A'
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao processar arquivo: ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * API: Preview de dados do arquivo antes da importação
     */
    public function previewLeituras(Request $request, Estacao $estacao)
    {
        $request->validate([
            'arquivo_importacao' => 'required|file|mimes:xlsx,xls,csv|max:10240',
        ]);

        try {
            $arquivo = $request->file('arquivo_importacao');
            $extensao = strtolower($arquivo->getClientOriginalExtension());

            // Ler o arquivo
            $dados = [];
            if (in_array($extensao, ['xlsx', 'xls'])) {
                $dados = $this->lerArquivoExcel($arquivo);
            } elseif ($extensao === 'csv') {
                $dados = $this->lerArquivoCSV($arquivo);
            }

            // Para preview, vamos mostrar dados RAW como estão no arquivo
            $previewRaw = $this->lerArquivoRawParaPreview($arquivo);

            // Limitar a 50 primeiras linhas para preview dos dados processados
            $preview = array_slice($dados, 0, 50);

            // Obter estatísticas do arquivo
            $totalLinhas = count($dados);
            $colunas = !empty($dados) ? array_keys($dados[0]) : [];

            // Detectar período dos dados (se houver coluna de data)
            $periodo = null;
            if (!empty($dados)) {
                $datas = [];
                foreach ($dados as $linha) {
                    foreach ($linha as $campo => $valor) {
                        if (in_array(strtolower($campo), ['data_leitura', 'data', 'date']) && !empty($valor)) {
                            try {
                                $data = \Carbon\Carbon::parse($valor);
                                $datas[] = $data;
                            } catch (\Exception $e) {
                                // Ignorar datas inválidas
                            }
                        }
                    }
                }

                if (!empty($datas)) {
                    $dataMin = min($datas);
                    $dataMax = max($datas);
                    $periodo = $dataMin->format('M/Y') . ' - ' . $dataMax->format('M/Y');
                }
            }

            // Detectar tipo de template
            $tipoTemplate = $this->detectarTipoTemplate($colunas);

            return response()->json([
                'success' => true,
                'dados' => $preview,
                'dados_raw' => $previewRaw,
                'estatisticas' => [
                    'total_linhas' => $totalLinhas,
                    'total_colunas' => count($colunas),
                    'colunas' => $colunas,
                    'periodo' => $periodo ?: 'Não detectado',
                    'tipo_template' => ucfirst(str_replace('_', ' ', $tipoTemplate)),
                    'arquivo_nome' => $arquivo->getClientOriginalName(),
                    'arquivo_tamanho' => $this->formatFileSize($arquivo->getSize())
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao processar arquivo: ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * Ler arquivo RAW para preview (como aparece no Excel)
     */
    private function lerArquivoRawParaPreview($arquivo)
    {
        $extensao = strtolower($arquivo->getClientOriginalExtension());
        $dados = [];

        try {
            if (in_array($extensao, ['xlsx', 'xls'])) {
                // Para Excel, ler as primeiras 11 linhas (cabeçalho + 10 dados)
                $inputFileType = \PhpOffice\PhpSpreadsheet\IOFactory::identify($arquivo->getPathname());
                $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader($inputFileType);
                $reader->setReadDataOnly(true);
                $spreadsheet = $reader->load($arquivo->getPathname());
                $worksheet = $spreadsheet->getActiveSheet();

                $highestColumn = $worksheet->getHighestColumn();
                $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

                // Ler as primeiras 51 linhas (incluindo cabeçalho)
                for ($row = 1; $row <= min(51, $worksheet->getHighestRow()); $row++) {
                    $linha = [];
                    for ($col = 1; $col <= $highestColumnIndex; $col++) {
                        $cell = $worksheet->getCellByColumnAndRow($col, $row);
                        $valor = $cell->getCalculatedValue();

                        // Converter datas Excel se necessário
                        if ($cell->getDataType() === \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_NUMERIC) {
                            if (\PhpOffice\PhpSpreadsheet\Shared\Date::isDateTime($cell)) {
                                $valor = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($valor)->format('d/m/Y');
                            }
                        }

                        $linha[] = $valor;
                    }
                    $dados[] = $linha;
                }

            } elseif ($extensao === 'csv') {
                // Para CSV, ler as primeiras 51 linhas
                if (($handle = fopen($arquivo->getPathname(), 'r')) !== FALSE) {
                    $contador = 0;
                    while (($linha = fgetcsv($handle, 1000, ',')) !== FALSE && $contador < 51) {
                        $dados[] = $linha;
                        $contador++;
                    }
                    fclose($handle);
                }
            }

        } catch (\Exception $e) {
            // Em caso de erro, retornar array vazio
            $dados = [];
        }

        return $dados;
    }

    /**
     * Formatar tamanho de arquivo
     */
    private function formatFileSize($bytes)
    {
        if ($bytes === 0) return '0 Bytes';
        $k = 1024;
        $sizes = ['Bytes', 'KB', 'MB', 'GB'];
        $i = floor(log($bytes) / log($k));
        return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
    }

    /**
     * Ler arquivo Excel
     */
    private function lerArquivoExcel($arquivo)
    {
        try {
            // Detectar automaticamente o tipo de arquivo
            $inputFileType = \PhpOffice\PhpSpreadsheet\IOFactory::identify($arquivo->getPathname());
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader($inputFileType);

            // Configurar reader para ser mais robusto
            $reader->setReadDataOnly(true);
            $reader->setReadEmptyCells(false);

            $spreadsheet = $reader->load($arquivo->getPathname());
            $worksheet = $spreadsheet->getActiveSheet();

            $dados = [];
            $cabecalho = [];
            $primeiraLinha = true;

            // Usar getHighestRow/Column para melhor performance
            $highestRow = $worksheet->getHighestRow();
            $highestColumn = $worksheet->getHighestColumn();
            $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

            for ($row = 1; $row <= $highestRow; $row++) {
                $valores = [];

                for ($col = 1; $col <= $highestColumnIndex; $col++) {
                    $cell = $worksheet->getCellByColumnAndRow($col, $row);
                    $valor = $cell->getCalculatedValue();

                    // Converter datas Excel para formato legível
                    if ($cell->getDataType() === \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_NUMERIC) {
                        if (\PhpOffice\PhpSpreadsheet\Shared\Date::isDateTime($cell)) {
                            $valor = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($valor)->format('Y-m-d');
                        }
                    }

                    $valores[] = $valor;
                }

                if ($primeiraLinha) {
                    $cabecalho = array_map('strtolower', array_map('trim', array_filter($valores)));
                    $primeiraLinha = false;
                } else {
                    // Verificar se a linha não está vazia
                    if (!empty(array_filter($valores, function($v) { return !empty(trim($v)); }))) {
                        // Ajustar array se houver diferença de tamanho
                        $valores = array_pad($valores, count($cabecalho), '');
                        $registro = array_combine($cabecalho, array_slice($valores, 0, count($cabecalho)));
                        $dados[] = $registro;
                    }
                }
            }

            return $dados;

        } catch (\Exception $e) {
            throw new \Exception("Erro ao ler arquivo Excel: " . $e->getMessage());
        }
    }

    /**
     * Ler arquivo CSV
     */
    private function lerArquivoCSV($arquivo)
    {
        $dados = [];
        $cabecalho = [];

        if (($handle = fopen($arquivo->getPathname(), 'r')) !== FALSE) {
            while (($linha = fgetcsv($handle, 1000, ',')) !== FALSE) {
                if (empty($cabecalho)) {
                    $cabecalho = array_map('strtolower', array_map('trim', $linha));
                } else {
                    $registro = array_combine($cabecalho, $linha);
                    if (!empty(array_filter($linha))) {
                        $dados[] = $registro;
                    }
                }
            }
            fclose($handle);
        }

        return $dados;
    }

    /**
     * Criar leitura a partir de registro importado
     */
    private function criarLeituraDeRegistro(Estacao $estacao, array $registro)
    {
        // Mapear campos - versão expandida para suportar templates existentes
        $mapeamento = [
            // Data - múltiplos formatos possíveis
            'data_leitura' => [
                'data_leitura', 'data', 'date', 'fecha', 'datum',
                'data da leitura', 'data leitura', 'data de leitura',
                'dia', 'day', 'jour', 'giorno'
            ],

            // Níveis hidrométricos - formatos existentes e novos
            'nivel_6h' => [
                'nivel_6h', 'nivel6h', 'nivel_06h', 'nível_6h', 'nivel 6h',
                'nível 6h', 'nível às 6h', 'nivel as 6h', 'leitura_6h',
                'leitura 6h', 'h06', '6h', 'manha', 'manhã', 'morning',
                'nivel da manha', 'nivel da manhã', 'altura_6h'
            ],
            'nivel_12h' => [
                'nivel_12h', 'nivel12h', 'nível_12h', 'nivel 12h',
                'nível 12h', 'nível às 12h', 'nivel as 12h', 'leitura_12h',
                'leitura 12h', 'h12', '12h', 'meio-dia', 'meio dia',
                'noon', 'nivel do meio-dia', 'altura_12h'
            ],
            'nivel_18h' => [
                'nivel_18h', 'nivel18h', 'nível_18h', 'nivel 18h',
                'nível 18h', 'nível às 18h', 'nivel as 18h', 'leitura_18h',
                'leitura 18h', 'h18', '18h', 'tarde', 'evening',
                'nivel da tarde', 'altura_18h'
            ],

            // Horas das leituras
            'hora_6h' => [
                'hora_6h', 'hora6h', 'hora 6h', 'h_6h', 'tempo_6h',
                'horario_6h', 'horário_6h', 'time_6h'
            ],
            'hora_12h' => [
                'hora_12h', 'hora12h', 'hora 12h', 'h_12h', 'tempo_12h',
                'horario_12h', 'horário_12h', 'time_12h'
            ],
            'hora_18h' => [
                'hora_18h', 'hora18h', 'hora 18h', 'h_18h', 'tempo_18h',
                'horario_18h', 'horário_18h', 'time_18h'
            ],

            // Precipitação - formatos diversos
            'precipitacao_mm' => [
                'precipitacao_mm', 'precipitacao', 'precipitação', 'chuva',
                'precipitação (mm)', 'precipitacao (mm)', 'pp', 'rain',
                'pluviosidade', 'chuva_mm', 'chuva (mm)', 'precipitação total',
                'precipitacao total', 'total de chuva', 'rainfall',
                'precip', 'prec', 'pluviometria', 'pluviométrica'
            ],

            // Temperaturas - formatos existentes
            'temperatura_max' => [
                'temperatura_max', 'temp_max', 'temperatura_máxima', 'tmax',
                'temperatura máxima', 'temperatura maxima', 'temp max',
                'temp máx', 'temp_máx', 't_max', 'max_temp',
                'temperatura_alta', 'temp_alta', 'high_temp', 'temp_h'
            ],
            'temperatura_min' => [
                'temperatura_min', 'temp_min', 'temperatura_mínima', 'tmin',
                'temperatura mínima', 'temperatura minima', 'temp min',
                'temp mín', 'temp_mín', 't_min', 'min_temp',
                'temperatura_baixa', 'temp_baixa', 'low_temp', 'temp_l'
            ],
            'temperatura_atual' => [
                'temperatura_atual', 'temp_atual', 'temperatura atual',
                'temperatura presente', 'temp_presente', 'current_temp',
                't_atual', 'temp_corrente', 'temperatura_corrente'
            ],

            // Humidade - variações
            'humidade_relativa' => [
                'humidade_relativa', 'humidade', 'umidade', 'hr', 'rh',
                'humidade relativa', 'umidade relativa', 'humidity',
                'humidade (%)', 'umidade (%)', 'humidade_rel', 'umidade_rel',
                'h_relativa', 'u_relativa', 'relative_humidity'
            ],

            // Pressão atmosférica
            'pressao_atmosferica' => [
                'pressao_atmosferica', 'pressao', 'pressão', 'pressure',
                'pressão atmosférica', 'pressao atmosferica', 'pa', 'p_atm',
                'pressao_atm', 'pressão_atm', 'atmospheric_pressure',
                'barometric_pressure', 'pressao_barometrica'
            ],

            // Vento
            'velocidade_vento' => [
                'velocidade_vento', 'velocidade do vento', 'vel_vento',
                'wind_speed', 'vento_velocidade', 'vento_vel', 'v_vento',
                'velocidade_do_vento', 'wind_velocity', 'vel_do_vento'
            ],
            'direcao_vento' => [
                'direcao_vento', 'direção_vento', 'direção do vento',
                'direcao do vento', 'wind_direction', 'dir_vento',
                'vento_direcao', 'vento_dir', 'd_vento', 'wind_dir'
            ],

            // Evaporação
            'evaporacao' => [
                'evaporacao', 'evaporação', 'evaporation', 'evap',
                'evaporação (mm)', 'evaporacao (mm)', 'evap_mm',
                'evaporacao_mm', 'evaporação_mm', 'e_mm'
            ],

            // Insolação
            'insolacao' => [
                'insolacao', 'insolação', 'sunshine', 'sol', 'insolacao_h',
                'insolação (h)', 'insolacao (h)', 'horas_sol', 'horas de sol',
                'sunshine_hours', 'insol', 'brilho_solar'
            ],

            // Observações - formatos diversos
            'observacoes' => [
                'observacoes', 'observações', 'obs', 'comentario', 'comentário',
                'notas', 'notes', 'remarks', 'observacion', 'annotation',
                'anotacoes', 'anotações', 'comentarios', 'comentários',
                'descricao', 'descrição', 'description', 'detalhes'
            ]
        ];

        $dadosLeitura = [
            'estacao_id' => $estacao->id,
            'operador_id' => Auth::id()
        ];

        // Mapear campos do registro para campos da leitura
        foreach ($mapeamento as $campoLeitura => $camposRegistro) {
            foreach ($camposRegistro as $campoRegistro) {
                if (isset($registro[$campoRegistro]) && !empty($registro[$campoRegistro])) {
                    if ($campoLeitura === 'data_leitura') {
                        // Processar data - múltiplos formatos suportados
                        $dadosLeitura[$campoLeitura] = $this->processarData($registro[$campoRegistro]);
                    } else {
                        // Processar valores numéricos
                        $valor = str_replace(',', '.', $registro[$campoRegistro]);
                        if (is_numeric($valor)) {
                            $dadosLeitura[$campoLeitura] = floatval($valor);
                        } else {
                            $dadosLeitura[$campoLeitura] = $registro[$campoRegistro];
                        }
                    }
                    break;
                }
            }
        }

        // Validar se tem data
        if (!isset($dadosLeitura['data_leitura'])) {
            throw new \Exception("Data da leitura não encontrada");
        }

        // Verificar se já existe leitura para esta data
        $leituraExistente = $estacao->leituras()
            ->whereDate('data_leitura', $dadosLeitura['data_leitura'])
            ->first();

        if ($leituraExistente) {
            // Atualizar leitura existente
            $leituraExistente->update($dadosLeitura);
            return 'atualizada';
        } else {
            // Criar nova leitura
            LeituraEstacao::create($dadosLeitura);
            return 'criada';
        }
    }

    /**
     * API: Exportar leituras para Excel
     */
    public function exportarLeituras(Request $request, Estacao $estacao)
    {
        $query = $estacao->leituras()->with('operador');

        // Filtros
        if ($request->data_inicio) {
            $query->whereDate('data_leitura', '>=', $request->data_inicio);
        }

        if ($request->data_fim) {
            $query->whereDate('data_leitura', '<=', $request->data_fim);
        }

        $leituras = $query->orderBy('data_leitura', 'desc')->get();

        // Criar planilha Excel
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Cabeçalho
        $cabecalho = ['Data', 'Hora 6h', 'Nível 6h', 'Hora 12h', 'Nível 12h', 'Hora 18h', 'Nível 18h'];

        if (in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica'])) {
            $cabecalho = array_merge($cabecalho, ['Precipitação (mm)', 'Temp. Max (°C)', 'Temp. Min (°C)', 'Humidade (%)']);
        }

        $cabecalho[] = 'Operador';
        $cabecalho[] = 'Observações';

        $sheet->fromArray($cabecalho, null, 'A1');

        // Dados
        $linha = 2;
        foreach ($leituras as $leitura) {
            $dados = [
                $leitura->data_leitura->format('d/m/Y'),
                $leitura->hora_6h ? $leitura->hora_6h->format('H:i') : '',
                $leitura->nivel_6h,
                $leitura->hora_12h ? $leitura->hora_12h->format('H:i') : '',
                $leitura->nivel_12h,
                $leitura->hora_18h ? $leitura->hora_18h->format('H:i') : '',
                $leitura->nivel_18h
            ];

            if (in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica'])) {
                $dados = array_merge($dados, [
                    $leitura->precipitacao_mm,
                    $leitura->temperatura_max,
                    $leitura->temperatura_min,
                    $leitura->humidade_relativa
                ]);
            }

            $dados[] = $leitura->operador->name ?? '';
            $dados[] = $leitura->observacoes ?? '';

            $sheet->fromArray($dados, null, 'A' . $linha);
            $linha++;
        }

        // Configurar response para download
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $filename = "leituras_{$estacao->codigo}_" . date('Y-m-d') . ".xlsx";

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Processar data em múltiplos formatos
     */
    private function processarData($dataString)
    {
        $dataString = trim($dataString);

        // Lista de formatos de data suportados
        $formatos = [
            'd/m/Y',       // 01/12/2024
            'd-m-Y',       // 01-12-2024
            'd.m.Y',       // 01.12.2024
            'Y-m-d',       // 2024-12-01
            'Y/m/d',       // 2024/12/01
            'd/m/y',       // 01/12/24
            'd-m-y',       // 01-12-24
            'm/d/Y',       // 12/01/2024 (formato americano)
            'm-d-Y',       // 12-01-2024
            'Y.m.d',       // 2024.12.01
            'd M Y',       // 01 Dec 2024
            'd F Y',       // 01 December 2024
            'j/n/Y',       // 1/12/2024 (sem zeros à esquerda)
            'j-n-Y',       // 1-12-2024
            'D, d M Y',    // Mon, 01 Dec 2024
        ];

        // Tentar cada formato
        foreach ($formatos as $formato) {
            try {
                $data = \Carbon\Carbon::createFromFormat($formato, $dataString);
                if ($data && $data->format($formato) === $dataString) {
                    return $data->format('Y-m-d');
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        // Se nenhum formato específico funcionou, tentar o parser geral do Carbon
        try {
            $data = \Carbon\Carbon::parse($dataString);
            return $data->format('Y-m-d');
        } catch (\Exception $e) {
            // Última tentativa: se for um número (serial do Excel)
            if (is_numeric($dataString)) {
                try {
                    // Excel usa 1900-01-01 como base (com bug do ano 1900)
                    $baseDate = \Carbon\Carbon::create(1900, 1, 1);
                    $data = $baseDate->addDays($dataString - 2); // -2 por causa do bug do Excel
                    return $data->format('Y-m-d');
                } catch (\Exception $e2) {
                    // Se tudo falhar
                    throw new \Exception("Data inválida: " . $dataString . ". Use formatos como dd/mm/aaaa, aaaa-mm-dd, etc.");
                }
            }

            throw new \Exception("Data inválida: " . $dataString . ". Use formatos como dd/mm/aaaa, aaaa-mm-dd, etc.");
        }
    }

    /**
     * Detectar tipo de template baseado nas colunas
     */
    private function detectarTipoTemplate(array $colunas)
    {
        $colunas = array_map('strtolower', array_map('trim', $colunas));

        // Indicadores para diferentes tipos de template
        $indicadores = [
            'existente_hidrometrica' => [
                'palavras_chave' => ['altura', 'cota', 'nivel da manha', 'nivel do meio-dia', 'nivel da tarde'],
                'peso' => 0
            ],
            'existente_pluviometrica' => [
                'palavras_chave' => ['pluviometria', 'precip', 'chuva_mm', 'pluviosidade'],
                'peso' => 0
            ],
            'existente_meteorologica' => [
                'palavras_chave' => ['brilho solar', 'insolacao', 'pressao_barometrica', 'vel_vento'],
                'peso' => 0
            ],
            'novo_formato' => [
                'palavras_chave' => ['nivel_6h', 'nivel_12h', 'nivel_18h', 'temperatura_max'],
                'peso' => 0
            ]
        ];

        // Calcular pesos baseado na presença de palavras-chave
        foreach ($colunas as $coluna) {
            foreach ($indicadores as $tipo => &$config) {
                foreach ($config['palavras_chave'] as $palavra) {
                    if (strpos($coluna, $palavra) !== false) {
                        $config['peso']++;
                    }
                }
            }
        }

        // Retornar o tipo com maior peso
        $tipoDetectado = 'novo_formato';
        $maiorPeso = 0;

        foreach ($indicadores as $tipo => $config) {
            if ($config['peso'] > $maiorPeso) {
                $maiorPeso = $config['peso'];
                $tipoDetectado = $tipo;
            }
        }

        return $tipoDetectado;
    }

    public function editLeitura(Estacao $estacao, LeituraEstacao $leitura)
    {
        return view('estacoes.leituras.edit', compact('estacao', 'leitura'));
    }

    public function updateLeitura(Request $request, Estacao $estacao, LeituraEstacao $leitura)
    {
        $rules = [
            'data_leitura' => 'required|date',
            'observacoes' => 'nullable|string',
        ];

        if ($estacao->tipo === 'hidrometrica') {
            $rules = array_merge($rules, [
                'hora_6h' => 'nullable|date_format:H:i',
                'hora_12h' => 'nullable|date_format:H:i',
                'hora_18h' => 'nullable|date_format:H:i',
                'nivel_6h' => 'nullable|numeric|min:0',
                'nivel_12h' => 'nullable|numeric|min:0',
                'nivel_18h' => 'nullable|numeric|min:0',
            ]);

            // Se a estação trabalha com cotas, adicionar validação
            if ($estacao->trabalhaComCotas()) {
                $rules = array_merge($rules, [
                    'cota_6h' => 'nullable|numeric',
                    'cota_12h' => 'nullable|numeric',
                    'cota_18h' => 'nullable|numeric',
                    'cota_hidrometrica' => 'nullable|numeric',
                ]);
            }
        }

        if (in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica', 'evaporimetrica'])) {
            $rules = array_merge($rules, [
                'precipitacao_mm' => 'nullable|numeric|min:0',
                'temperatura_max' => 'nullable|numeric',
                'temperatura_min' => 'nullable|numeric',
                'humidade_relativa' => 'nullable|numeric|min:0|max:100',
                'evaporacao' => 'nullable|numeric|min:0',
            ]);
        }

        $validated = $request->validate($rules);
        $leitura->update($validated);

        return redirect()->route('estacoes.leituras', $estacao)
            ->with('success', 'Leitura atualizada com sucesso!');
    }

    public function destroyLeitura(Estacao $estacao, LeituraEstacao $leitura)
    {
        try {
            $leitura->delete();

            return redirect()->route('estacoes.leituras', $estacao)
                ->with('success', 'Leitura removida com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('estacoes.leituras', $estacao)
                ->with('error', 'Erro ao remover leitura: ' . $e->getMessage());
        }
    }

}