<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraEstacao;
use App\Models\LeituraBarragem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\EstatisticasExport;
use Barryvdh\DomPDF\Facade\Pdf;

class EstatisticasController extends Controller
{
    /**
     * Exibir página de estatísticas com gráficos integrados
     */
    public function index(Request $request)
    {
        // Determinar qual tipo de estatística mostrar
        $tipo = $request->get('tipo', 'precipitacao');
        $periodo = $request->get('periodo', 'mensal');
        $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(3)->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());

        $barragens = Barragem::orderBy('nome')->get();
        $estacoes = Estacao::orderBy('nome')->get();
        $estacoesPluv = Estacao::whereIn('tipo', ['pluviometrica', 'meteorologica', 'climatologica'])->orderBy('nome')->get();
        $estacoesHidro = Estacao::where('tipo', 'hidrometrica')->orderBy('nome')->get();

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];
        $selectedId = null;

        // Carregar dados conforme o tipo selecionado
        switch ($tipo) {
            case 'precipitacao':
                $selectedId = $request->get('estacao_id');
                if ($selectedId) {
                    $result = $this->calcularPrecipitacao($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'cotas':
                $selectedId = $request->get('barragem_id');
                if ($selectedId) {
                    $result = $this->calcularCotas($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'niveis':
                $selectedId = $request->get('estacao_id');
                if ($selectedId) {
                    $result = $this->calcularNiveis($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;
        }

        return view('estatisticas.index', compact(
            'barragens',
            'estacoes',
            'estacoesPluv',
            'estacoesHidro',
            'tipo',
            'periodo',
            'dataInicio',
            'dataFim',
            'selectedId',
            'medias',
            'graficoLabels',
            'graficoDados'
        ));
    }

    /**
     * Calcular médias de precipitação
     */
    private function calcularPrecipitacao($estacaoId, $periodo, $dataInicio, $dataFim)
    {
        $query = LeituraEstacao::where('estacao_id', $estacaoId)
            ->whereBetween('data_leitura', [$dataInicio, $dataFim])
            ->whereNotNull('precipitacao_mm')
            ->where('precipitacao_mm', '>', 0);

        $medias = $this->agruparPorPeriodo($query, $periodo, 'precipitacao');

        $labels = [];
        $dados = [];
        foreach ($medias as $media) {
            $labels[] = $this->formatarLabel($media, $periodo);
            $dados[] = round($media->media, 2);
        }

        return ['medias' => $medias, 'labels' => $labels, 'dados' => $dados];
    }

    /**
     * Calcular médias de cotas
     */
    private function calcularCotas($barragemId, $periodo, $dataInicio, $dataFim)
    {
        $query = LeituraBarragem::where('barragem_id', $barragemId)
            ->whereBetween('data_leitura', [$dataInicio, $dataFim])
            ->whereNotNull('cota_actual');

        $medias = $this->agruparPorPeriodo($query, $periodo, 'cotas');

        $labels = [];
        $dados = [];
        foreach ($medias as $media) {
            $labels[] = $this->formatarLabel($media, $periodo);
            $dados[] = round($media->media, 3);
        }

        return ['medias' => $medias, 'labels' => $labels, 'dados' => $dados];
    }

    /**
     * Calcular médias de níveis
     */
    private function calcularNiveis($estacaoId, $periodo, $dataInicio, $dataFim)
    {
        $nivelFormula = '(COALESCE(nivel_6h, 0) + COALESCE(nivel_9h, 0) + COALESCE(nivel_12h, 0) + COALESCE(nivel_15h, 0) + COALESCE(nivel_18h, 0)) / ((CASE WHEN nivel_6h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_9h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_12h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_15h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_18h IS NOT NULL THEN 1 ELSE 0 END))';

        $query = LeituraEstacao::where('estacao_id', $estacaoId)
            ->whereBetween('data_leitura', [$dataInicio, $dataFim])
            ->where(function($q) {
                $q->whereNotNull('nivel_6h')
                  ->orWhereNotNull('nivel_9h')
                  ->orWhereNotNull('nivel_12h')
                  ->orWhereNotNull('nivel_15h')
                  ->orWhereNotNull('nivel_18h');
            });

        $medias = $this->agruparPorPeriodo($query, $periodo, 'niveis', $nivelFormula);

        $labels = [];
        $dados = [];
        foreach ($medias as $media) {
            $labels[] = $this->formatarLabel($media, $periodo);
            $dados[] = round($media->media, 3);
        }

        return ['medias' => $medias, 'labels' => $labels, 'dados' => $dados];
    }

    /**
     * Agrupar dados por período
     */
    private function agruparPorPeriodo($query, $periodo, $tipo, $customFormula = null)
    {
        $campo = $tipo === 'precipitacao' ? 'precipitacao_mm' :
                ($tipo === 'cotas' ? 'cota_actual' : $customFormula);

        switch ($periodo) {
            case 'diario':
                return $query->select(
                    DB::raw('DATE(data_leitura) as data'),
                    DB::raw("AVG($campo) as media"),
                    $tipo === 'precipitacao' ? DB::raw("SUM($campo) as total") : DB::raw("0 as total"),
                    DB::raw("MIN($campo) as minimo"),
                    DB::raw("MAX($campo) as maximo"),
                    DB::raw('COUNT(*) as leituras')
                )
                ->groupBy('data')
                ->orderBy('data')
                ->get();

            case 'semanal':
                return $query->select(
                    DB::raw('YEAR(data_leitura) as ano'),
                    DB::raw('WEEK(data_leitura) as semana'),
                    DB::raw("AVG($campo) as media"),
                    $tipo === 'precipitacao' ? DB::raw("SUM($campo) as total") : DB::raw("0 as total"),
                    DB::raw("MIN($campo) as minimo"),
                    DB::raw("MAX($campo) as maximo"),
                    DB::raw('COUNT(*) as leituras')
                )
                ->groupBy('ano', 'semana')
                ->orderBy('ano')
                ->orderBy('semana')
                ->get();

            case 'mensal':
                return $query->select(
                    DB::raw('YEAR(data_leitura) as ano'),
                    DB::raw('MONTH(data_leitura) as mes'),
                    DB::raw("AVG($campo) as media"),
                    $tipo === 'precipitacao' ? DB::raw("SUM($campo) as total") : DB::raw("0 as total"),
                    DB::raw("MIN($campo) as minimo"),
                    DB::raw("MAX($campo) as maximo"),
                    DB::raw('COUNT(*) as leituras')
                )
                ->groupBy('ano', 'mes')
                ->orderBy('ano')
                ->orderBy('mes')
                ->get();

            case 'anual':
                return $query->select(
                    DB::raw('YEAR(data_leitura) as ano'),
                    DB::raw("AVG($campo) as media"),
                    $tipo === 'precipitacao' ? DB::raw("SUM($campo) as total") : DB::raw("0 as total"),
                    DB::raw("MIN($campo) as minimo"),
                    DB::raw("MAX($campo) as maximo"),
                    DB::raw('COUNT(*) as leituras')
                )
                ->groupBy('ano')
                ->orderBy('ano')
                ->get();

            default:
                return collect([]);
        }
    }

    /**
     * Formatar label do gráfico
     */
    private function formatarLabel($media, $periodo)
    {
        switch ($periodo) {
            case 'diario':
                return Carbon::parse($media->data)->format('d/m/Y');
            case 'semanal':
                return "Sem {$media->semana}/{$media->ano}";
            case 'mensal':
                return Carbon::create($media->ano, $media->mes)->format('M/Y');
            case 'anual':
                return $media->ano;
            default:
                return '';
        }
    }

    /**
     * Exportar para Excel
     */
    public function exportarExcel(Request $request)
    {
        $tipo = $request->get('tipo', 'precipitacao');
        $periodo = $request->get('periodo', 'mensal');
        $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(3)->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());
        $selectedId = $request->get('estacao_id') ?? $request->get('barragem_id');

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];
        $entidadeNome = '';

        switch ($tipo) {
            case 'precipitacao':
                if ($selectedId) {
                    $estacao = Estacao::find($selectedId);
                    $entidadeNome = $estacao ? $estacao->nome : '';
                    $result = $this->calcularPrecipitacao($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'cotas':
                if ($selectedId) {
                    $barragem = Barragem::find($selectedId);
                    $entidadeNome = $barragem ? $barragem->nome : '';
                    $result = $this->calcularCotas($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'niveis':
                if ($selectedId) {
                    $estacao = Estacao::find($selectedId);
                    $entidadeNome = $estacao ? $estacao->nome : '';
                    $result = $this->calcularNiveis($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;
        }

        $fileName = "estatisticas_{$tipo}_" . Carbon::now()->format('Y-m-d_His') . '.xlsx';

        return Excel::download(
            new EstatisticasExport($medias, $tipo, $periodo, $entidadeNome, $graficoLabels, $graficoDados),
            $fileName
        );
    }

    /**
     * Exportar para PDF
     */
    public function exportarPdf(Request $request)
    {
        $tipo = $request->get('tipo', 'precipitacao');
        $periodo = $request->get('periodo', 'mensal');
        $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(3)->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());
        $selectedId = $request->get('estacao_id') ?? $request->get('barragem_id');

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];
        $entidadeNome = '';

        switch ($tipo) {
            case 'precipitacao':
                if ($selectedId) {
                    $estacao = Estacao::find($selectedId);
                    $entidadeNome = $estacao ? $estacao->nome : '';
                    $result = $this->calcularPrecipitacao($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'cotas':
                if ($selectedId) {
                    $barragem = Barragem::find($selectedId);
                    $entidadeNome = $barragem ? $barragem->nome : '';
                    $result = $this->calcularCotas($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;

            case 'niveis':
                if ($selectedId) {
                    $estacao = Estacao::find($selectedId);
                    $entidadeNome = $estacao ? $estacao->nome : '';
                    $result = $this->calcularNiveis($selectedId, $periodo, $dataInicio, $dataFim);
                    $medias = $result['medias'];
                    $graficoLabels = $result['labels'];
                    $graficoDados = $result['dados'];
                }
                break;
        }

        $pdf = Pdf::loadView('estatisticas.pdf-grafico', compact(
            'medias',
            'tipo',
            'periodo',
            'entidadeNome',
            'dataInicio',
            'dataFim',
            'graficoLabels',
            'graficoDados'
        ))
        ->setPaper('a4', 'landscape'); // Definir orientação horizontal

        $fileName = "estatisticas_{$tipo}_" . Carbon::now()->format('Y-m-d_His') . '.pdf';

        return $pdf->download($fileName);
    }
}
