<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\NotificacaoEnviada;
use App\Services\TwilioSmsService;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class NotificationConfigController extends Controller
{
    protected $twilioService;
    protected $notificationService;

    public function __construct(TwilioSmsService $twilioService, NotificationService $notificationService)
    {
        $this->twilioService = $twilioService;
        $this->notificationService = $notificationService;
    }

    /**
     * Mostrar configurações de notificação do usuário
     */
    public function index()
    {
        $user = Auth::user();
        $stats = NotificacaoEnviada::getStats('mes');
        $userStats = $user->getEstatisticasNotificacoes(30);
        
        return view('notifications.config', compact('user', 'stats', 'userStats'));
    }

    /**
     * Atualizar configurações de notificação
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'telefone' => 'nullable|string|max:20',
            'notificacao_sms' => 'boolean',
            'notificacao_email' => 'boolean',
            'notificacao_emergencia' => 'boolean',
            'niveis_notificacao' => 'nullable|array',
            'niveis_notificacao.*' => 'in:informacao,atencao,alerta,emergencia',
            'receber_relatorio_diario' => 'boolean',
            'horario_relatorio' => 'nullable|date_format:H:i',
        ]);

        // Validar telefone se fornecido
        if ($validated['telefone']) {
            $telefoneFormatado = $this->twilioService->formatPhoneNumber($validated['telefone']);
            if (!$telefoneFormatado) {
                return redirect()->back()
                    ->withErrors(['telefone' => 'Número de telefone inválido. Use formato: +258XXXXXXXXX ou 8XXXXXXXX'])
                    ->withInput();
            }
            $validated['telefone'] = $telefoneFormatado;
        }

        $user->update($validated);

        return redirect()->back()
            ->with('success', 'Configurações de notificação atualizadas com sucesso!');
    }

    /**
     * Testar SMS
     */
    public function testSms(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->telefone) {
            return redirect()->back()
                ->withErrors(['teste' => 'Configure um número de telefone primeiro.']);
        }

        if (!$user->temTelefoneValido()) {
            return redirect()->back()
                ->withErrors(['teste' => 'Número de telefone inválido.']);
        }

        $resultado = $this->notificationService->testarConfiguracao($user->telefone);

        if ($resultado['success']) {
            return redirect()->back()
                ->with('success', 'SMS de teste enviado com sucesso!');
        } else {
            return redirect()->back()
                ->withErrors(['teste' => 'Falha ao enviar SMS: ' . $resultado['error']]);
        }
    }

    /**
     * Dashboard de notificações (admin)
     */
    public function dashboard()
    {
        // $this->authorize('admin'); // Assumindo que existe uma política - comentado temporariamente

        $stats = [
            'hoje' => NotificacaoEnviada::getStats('hoje'),
            'semana' => NotificacaoEnviada::getStats('semana'),
            'mes' => NotificacaoEnviada::getStats('mes'),
        ];

        $notificacoesRecentes = NotificacaoEnviada::with(['user', 'alerta'])
            ->orderBy('created_at', 'desc')
            ->take(20)
            ->get();

        $usuariosComNotificacao = User::where('notificacao_sms', true)
            ->orWhere('notificacao_email', true)
            ->count();

        $taxaEntregaGeral = $this->calcularTaxaEntregaGeral();

        return view('notifications.dashboard', compact(
            'stats', 
            'notificacoesRecentes', 
            'usuariosComNotificacao',
            'taxaEntregaGeral'
        ));
    }

    /**
     * Histórico de notificações do usuário
     */
    public function historico()
    {
        $user = Auth::user();
        
        $notificacoes = $user->notificacoesEnviadas()
            ->with('alerta')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('notifications.historico', compact('notificacoes'));
    }

    /**
     * Reenviar notificação falhada
     */
    public function reenviar(NotificacaoEnviada $notificacao)
    {
        // $this->authorize('update', $notificacao); // Política de autorização - comentado temporariamente

        if ($notificacao->status !== 'falhado') {
            return redirect()->back()
                ->withErrors(['reenvio' => 'Apenas notificações falhadas podem ser reenviadas.']);
        }

        try {
            if ($notificacao->tipo === 'sms') {
                $resultado = $this->twilioService->sendSms(
                    $notificacao->destinatario, 
                    $notificacao->conteudo
                );

                if ($resultado['success']) {
                    $notificacao->marcarComoEnviado($resultado['message_sid']);
                    return redirect()->back()
                        ->with('success', 'Notificação reenviada com sucesso!');
                } else {
                    return redirect()->back()
                        ->withErrors(['reenvio' => 'Falha ao reenviar: ' . $resultado['error']]);
                }
            }
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['reenvio' => 'Erro interno: ' . $e->getMessage()]);
        }
    }

    /**
     * Configurações globais do sistema (admin)
     */
    public function configuracoes()
    {
        // $this->authorize('admin'); // Comentado temporariamente

        $configs = [
            'emergency_sms_enabled' => config('services.notifications.emergency_sms_enabled'),
            'alert_email_enabled' => config('services.notifications.alert_email_enabled'),
            'escalation_enabled' => config('services.notifications.escalation_enabled'),
            'escalation_time_minutes' => config('services.notifications.escalation_time_minutes'),
            'daily_report_time' => config('services.notifications.daily_report_time'),
        ];

        return view('notifications.configuracoes', compact('configs'));
    }

    /**
     * Enviar relatório diário manualmente (admin)
     */
    public function enviarRelatorioDiario()
    {
        // $this->authorize('admin'); // Comentado temporariamente

        try {
            $this->notificationService->enviarRelatorioDiario();
            
            return redirect()->back()
                ->with('success', 'Relatório diário enviado com sucesso!');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withErrors(['relatorio' => 'Erro ao enviar relatório: ' . $e->getMessage()]);
        }
    }

    /**
     * Calcular taxa de entrega geral
     */
    private function calcularTaxaEntregaGeral($dias = 30)
    {
        $total = NotificacaoEnviada::where('created_at', '>=', now()->subDays($dias))->count();
        $entregues = NotificacaoEnviada::where('created_at', '>=', now()->subDays($dias))
            ->where('status', 'entregue')
            ->count();

        return $total > 0 ? round(($entregues / $total) * 100, 2) : 0;
    }

    /**
     * API para estatísticas em tempo real
     */
    public function apiStats()
    {
        return response()->json([
            'notificacoes_hoje' => NotificacaoEnviada::hoje()->count(),
            'alertas_ativos' => \App\Models\Alerta::where('estado', 'activo')->count(),
            'usuarios_notificados' => User::where('notificacao_sms', true)
                ->orWhere('notificacao_email', true)
                ->count(),
            'taxa_entrega' => $this->calcularTaxaEntregaGeral(7),
        ]);
    }
}