<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\RegionalAccessService;
use Symfony\Component\HttpFoundation\Response;

/**
 * Middleware para verificar acesso regional a recursos
 */
class CheckRegionalAccess
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     * @param  string|null  $resourceType  Tipo de recurso (barragem, estacao, etc.)
     * @param  string|null  $permission    Permissao a verificar (opcional)
     */
    public function handle(Request $request, Closure $next, ?string $resourceType = null, ?string $permission = null): Response
    {
        $user = $request->user();

        if (!$user) {
            abort(401, 'Autenticacao necessaria.');
        }

        $accessService = new RegionalAccessService($user);

        // Para rotas com recurso especifico
        if ($resourceType && $request->route($resourceType)) {
            $resource = $request->route($resourceType);

            $canAccess = match($resourceType) {
                'barragem' => $accessService->canAccessBarragem($resource),
                'estacao' => $accessService->canAccessEstacao($resource),
                default => true
            };

            if (!$canAccess) {
                abort(403, 'Nao tem permissao para aceder a este recurso. Ele nao pertence a sua divisao.');
            }

            // Verificar permissao especifica se fornecida
            if ($permission) {
                $divisaoId = $resource->divisao_administrativa_id ?? null;
                if (!$accessService->canPerformAction($permission, $divisaoId)) {
                    abort(403, 'Nao tem permissao para realizar esta accao nesta divisao.');
                }
            }
        }

        // Partilhar servico com views
        view()->share('regionalAccess', $accessService);

        // Disponibilizar no request para controllers
        $request->attributes->set('regionalAccess', $accessService);

        return $next($request);
    }
}
