<?php

namespace App\Jobs;

use App\Models\Alerta;
use App\Services\NotificationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessarEscalacaoAlerta implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $alertaId;

    /**
     * Create a new job instance.
     */
    public function __construct($alertaId)
    {
        $this->alertaId = $alertaId;
    }

    /**
     * Execute the job.
     */
    public function handle(NotificationService $notificationService): void
    {
        try {
            $alerta = Alerta::find($this->alertaId);
            
            if (!$alerta) {
                Log::warning('Alerta não encontrado para escalação', ['alerta_id' => $this->alertaId]);
                return;
            }

            // Verificar se alerta ainda está ativo
            if (!$alerta->isAtivo()) {
                Log::info('Alerta já foi resolvido, cancelando escalação', ['alerta_id' => $this->alertaId]);
                return;
            }

            // Processar escalação
            $notificationService->processarEscalacao($alerta);
            
            Log::info('Escalação processada com sucesso', ['alerta_id' => $this->alertaId]);
            
        } catch (\Exception $e) {
            Log::error('Erro ao processar escalação de alerta', [
                'alerta_id' => $this->alertaId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-tentar o job em caso de erro
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Job de escalação falhou definitivamente', [
            'alerta_id' => $this->alertaId,
            'error' => $exception->getMessage()
        ]);
    }
}