<?php
// app/Models/Alerta.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Alerta extends Model
{
    use HasFactory;

    protected $fillable = [
        'tipo',
        'nivel',
        'barragem_id',
        'estacao_id',
        'titulo',
        'descricao',
        'valor_registado',
        'valor_limite',
        'data_inicio',
        'data_fim',
        'estado',
        'acoes_tomadas',
    ];

    protected $casts = [
        'valor_registado' => 'decimal:3',
        'valor_limite' => 'decimal:3',
        'data_inicio' => 'datetime',
        'data_fim' => 'datetime',
    ];

    public function barragem(): BelongsTo
    {
        return $this->belongsTo(Barragem::class);
    }

    public function estacao(): BelongsTo
    {
        return $this->belongsTo(Estacao::class);
    }

    public function getInfrastrutura()
    {
        return $this->barragem ?? $this->estacao;
    }

    public function getNomeInfrastrutura()
    {
        $infraestrutura = $this->getInfrastrutura();
        return $infraestrutura ? $infraestrutura->nome : 'N/A';
    }

    public function getTipoFormatado()
    {
        $tipos = [
            'nivel_critico' => 'Nível Crítico',
            'descarga' => 'Descarga',
            'precipitacao' => 'Precipitação',
            'emergencia' => 'Emergência',
        ];

        return $tipos[$this->tipo] ?? $this->tipo;
    }

    public function getNivelFormatado()
    {
        $niveis = [
            'informacao' => 'Informação',
            'atencao' => 'Atenção',
            'alerta' => 'Alerta',
            'emergencia' => 'Emergência',
        ];

        return $niveis[$this->nivel] ?? $this->nivel;
    }

    public function getCorNivel()
    {
        $cores = [
            'informacao' => 'blue',
            'atencao' => 'yellow',
            'alerta' => 'orange',
            'emergencia' => 'red',
        ];

        return $cores[$this->nivel] ?? 'gray';
    }

    public function getDuracaoFormatada()
    {
        if (!$this->data_fim) {
            return 'Em curso (' . $this->data_inicio->diffForHumans() . ')';
        }

        return $this->data_inicio->diffForHumans($this->data_fim, true);
    }

    public function isAtivo()
    {
        return $this->estado === 'activo';
    }

    public function resolver($acoes_tomadas = null)
    {
        $this->update([
            'estado' => 'resolvido',
            'data_fim' => now(),
            'acoes_tomadas' => $acoes_tomadas,
        ]);
    }

    public function cancelar($motivo = null)
    {
        $this->update([
            'estado' => 'cancelado',
            'data_fim' => now(),
            'acoes_tomadas' => $motivo,
        ]);
    }

    public static function criarAlertaBarragem($barragem, $leitura)
    {
        // Verificar se cota_actual está disponível
        if (!$leitura->cota_actual) {
            return null;
        }

        $status = $barragem->getStatusNivel($leitura->cota_actual);

        // Mapear status para valores válidos do enum: 'informacao','atencao','alerta','emergencia'
        $nivelMap = [
            'nmc_excedido' => 'emergencia',
            'proximo_nmc' => 'alerta',
            'npa_atingido' => 'atencao',
            'abaixo_nme' => 'atencao',
            // Status que não geram alerta:
            'operacional' => null,
            'nme_atingido' => null,
            'normal' => null,
        ];

        $nivelAlerta = $nivelMap[$status] ?? null;

        // Só criar alerta se tiver um nível válido
        if ($nivelAlerta) {
            // Formatar hora de forma segura
            $horaFormatada = $leitura->hora_leitura
                ? (is_string($leitura->hora_leitura) ? $leitura->hora_leitura : $leitura->hora_leitura->format('H:i'))
                : '06:00';

            $alerta = static::create([
                'tipo' => 'nivel_critico',
                'nivel' => $nivelAlerta,
                'barragem_id' => $barragem->id,
                'titulo' => "Nível {$status} na barragem {$barragem->nome}",
                'descricao' => "A barragem {$barragem->nome} registou cota de {$leitura->cota_actual}m às {$horaFormatada} do dia {$leitura->data_leitura->format('d/m/Y')}.",
                'valor_registado' => $leitura->cota_actual,
                'valor_limite' => $nivelAlerta === 'emergencia' ? $barragem->cota_nmc : $barragem->cota_npa,
                'data_inicio' => now(),
                'estado' => 'activo',
            ]);

            // Processar notificações
            app(\App\Services\NotificationService::class)->processarNovoAlerta($alerta);

            return $alerta;
        }

        return null;
    }

    public static function criarAlertaEstacao($estacao, $leitura)
    {
        if ($estacao->nivel_alerta && $leitura->nivel_hidrometrico >= $estacao->nivel_alerta) {
            $alerta = static::create([
                'tipo' => 'nivel_critico',
                'nivel' => 'alerta',
                'estacao_id' => $estacao->id,
                'titulo' => "Nível de alerta na estação {$estacao->nome}",
                'descricao' => "A estação {$estacao->nome} registou nível hidrométrico de {$leitura->nivel_hidrometrico}m no dia {$leitura->data_leitura->format('d/m/Y')}.",
                'valor_registado' => $leitura->nivel_hidrometrico,
                'valor_limite' => $estacao->nivel_alerta,
                'data_inicio' => now(),
                'estado' => 'activo',
            ]);

            // Processar notificações
            app(\App\Services\NotificationService::class)->processarNovoAlerta($alerta);
            
            return $alerta;
        }
    }

    public function notificacoesEnviadas()
    {
        return $this->hasMany(NotificacaoEnviada::class);
    }
}