<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CurvaVolume extends Model
{
    use HasFactory;

    protected $table = 'curvas_volume';

    protected $fillable = [
        'barragem_id',
        'cota',
        'volume',
        'area',
        'observacoes',
        'criado_por',
    ];

    protected $casts = [
        'cota' => 'decimal:3',
        'volume' => 'decimal:3',
        'area' => 'decimal:3',
    ];

    public function barragem(): BelongsTo
    {
        return $this->belongsTo(Barragem::class);
    }

    public function criador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'criado_por');
    }

    /**
     * Calcular volume interpolado para uma cota específica
     */
    public static function interpolateVolume($barragem, $cota)
    {
        $pontoInferior = $barragem->curvas_volume()
            ->where('cota', '<=', $cota)
            ->orderBy('cota', 'desc')
            ->first();

        $pontoSuperior = $barragem->curvas_volume()
            ->where('cota', '>', $cota)
            ->orderBy('cota', 'asc')
            ->first();

        if (!$pontoInferior) {
            return $pontoSuperior ? 0 : null;
        }

        if (!$pontoSuperior) {
            return $pontoInferior->volume;
        }

        // Interpolação linear
        $deltaCota = $pontoSuperior->cota - $pontoInferior->cota;
        $deltaVolume = $pontoSuperior->volume - $pontoInferior->volume;
        $proporcao = ($cota - $pontoInferior->cota) / $deltaCota;

        return $pontoInferior->volume + ($deltaVolume * $proporcao);
    }

    /**
     * Calcular área interpolada para uma cota específica
     */
    public static function interpolateArea($barragem, $cota)
    {
        $pontoInferior = $barragem->curvas_volume()
            ->where('cota', '<=', $cota)
            ->whereNotNull('area')
            ->orderBy('cota', 'desc')
            ->first();

        $pontoSuperior = $barragem->curvas_volume()
            ->where('cota', '>', $cota)
            ->whereNotNull('area')
            ->orderBy('cota', 'asc')
            ->first();

        if (!$pontoInferior) {
            return $pontoSuperior ? 0 : null;
        }

        if (!$pontoSuperior) {
            return $pontoInferior->area;
        }

        // Interpolação linear
        $deltaCota = $pontoSuperior->cota - $pontoInferior->cota;
        $deltaArea = $pontoSuperior->area - $pontoInferior->area;
        $proporcao = ($cota - $pontoInferior->cota) / $deltaCota;

        return $pontoInferior->area + ($deltaArea * $proporcao);
    }
}