<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

/**
 * Modelo para permissoes customizadas por divisao/role
 * Permite conceder ou revogar permissoes especificas numa divisao
 */
class DivisaoRolePermission extends Model
{
    protected $table = 'divisao_role_permissions';

    protected $fillable = [
        'divisao_administrativa_id',
        'role_id',
        'permission_id',
        'tipo',
    ];

    /**
     * Divisao administrativa
     */
    public function divisao(): BelongsTo
    {
        return $this->belongsTo(DivisaoAdministrativa::class, 'divisao_administrativa_id');
    }

    /**
     * Role do Spatie
     */
    public function role(): BelongsTo
    {
        return $this->belongsTo(Role::class);
    }

    /**
     * Permissao do Spatie
     */
    public function permission(): BelongsTo
    {
        return $this->belongsTo(Permission::class);
    }

    /**
     * Verificar se e uma concessao
     */
    public function isGrant(): bool
    {
        return $this->tipo === 'grant';
    }

    /**
     * Verificar se e uma revogacao
     */
    public function isRevoke(): bool
    {
        return $this->tipo === 'revoke';
    }

    /**
     * Scope para concessoes
     */
    public function scopeGrants($query)
    {
        return $query->where('tipo', 'grant');
    }

    /**
     * Scope para revogacoes
     */
    public function scopeRevokes($query)
    {
        return $query->where('tipo', 'revoke');
    }

    /**
     * Scope por divisao
     */
    public function scopeParaDivisao($query, int $divisaoId)
    {
        return $query->where('divisao_administrativa_id', $divisaoId);
    }

    /**
     * Scope por role
     */
    public function scopeParaRole($query, int $roleId)
    {
        return $query->where('role_id', $roleId);
    }

    /**
     * Obter descricao formatada
     */
    public function getDescricaoAttribute(): string
    {
        $accao = $this->isGrant() ? 'Concede' : 'Revoga';
        $permissao = $this->permission?->name ?? 'N/A';
        $role = $this->role?->name ?? 'N/A';
        $divisao = $this->divisao?->nome ?? 'N/A';

        return "{$accao} '{$permissao}' para '{$role}' em '{$divisao}'";
    }
}
