<?php
// app/Models/LeituraBarragem.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LeituraBarragem extends Model
{
    use HasFactory;

    protected $table = 'leituras_barragens';

    protected $fillable = [
        'barragem_id',
        'data_leitura',
        'hora_leitura',
        'nivel_hidrometrico', // Leitura na escala/régua (valor de entrada principal)
        'cota_actual',
        'volume_actual',
        'volume_morto',
        'percentagem_enchimento',
        'percentagem_total',
        'volume_armazenado',
        'caudal_captacao',
        'caudal_afluente',
        'caudal_efluente',
        'descarga_fundo',
        'descarga_superficie',
        'caudal_ecologico',
        'caudal_turbinado',
        'caudal_descarregador',
        'evaporacao',
        'infiltracao',
        'variacao_volume',
        'comparacao_ano_anterior',
        'descarga_total',
        'precipitacao_p1',
        'precipitacao_p2',
        'precipitacao_p3',
        'precipitacao_p4',
        'precipitacoes',
        'cota_npa',
        'nivel_minimo',
        'nivel_maximo',
        'data_nivel_minimo',
        'data_nivel_maximo',
        'operador_id',
        'metodo_leitura',
        'observacoes',
        'validado',
        'validado_por',
        'data_validacao',
    ];

    protected $casts = [
        'data_leitura' => 'date',
        'hora_leitura' => 'datetime',
        'nivel_hidrometrico' => 'decimal:2',
        'cota_actual' => 'decimal:2',
        'volume_actual' => 'decimal:2',
        'percentagem_enchimento' => 'decimal:2',
        'caudal_afluente' => 'decimal:3',
        'caudal_efluente' => 'decimal:3',
        'descarga_fundo' => 'decimal:3',
        'descarga_superficie' => 'decimal:3',
        'caudal_ecologico' => 'decimal:3',
        'caudal_turbinado' => 'decimal:3',
        'caudal_descarregador' => 'decimal:3',
        'evaporacao' => 'decimal:3',
        'infiltracao' => 'decimal:3',
        'variacao_volume' => 'decimal:2',
        'comparacao_ano_anterior' => 'decimal:2',
        'descarga_total' => 'decimal:3',
        'precipitacao_p1' => 'decimal:2',
        'precipitacao_p2' => 'decimal:2',
        'precipitacao_p3' => 'decimal:2',
        'precipitacao_p4' => 'decimal:2',
        'precipitacoes' => 'array',
        'validado' => 'boolean',
        'data_validacao' => 'datetime',
    ];

    public function barragem(): BelongsTo
    {
        return $this->belongsTo(Barragem::class);
    }

    public function operador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'operador_id');
    }

    public function validador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'validado_por');
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });

        static::updating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });
    }

    /**
     * Calcular valores automáticos baseados nas fórmulas do Excel
     * Baseado em: ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm
     *
     * Fluxo:
     * 1. Se nivel_hidrometrico fornecido E escala_npa configurada → calcula cota
     * 2. Senao, usa cota_actual diretamente
     * 3. Da cota, calcula volume, percentagens, etc.
     */
    public function calcularValoresAutomaticos()
    {
        if (!$this->barragem) {
            return;
        }

        // 1. CALCULAR COTA a partir do Nivel Hidrometrico (se aplicavel)
        // Formula Excel: Cota = NPA - (escala_npa - nivel_hidrometrico)
        if ($this->nivel_hidrometrico && $this->barragem->escala_npa && $this->barragem->cota_npa) {
            $this->cota_actual = $this->barragem->cota_npa - ($this->barragem->escala_npa - $this->nivel_hidrometrico);
        }

        // Verificar se temos cota para continuar
        if (!$this->cota_actual) {
            return;
        }

        // 2. Calcular volume atual usando curva cota-volume da barragem
        $this->volume_actual = $this->barragem->calcularVolumeAtual($this->cota_actual);

        if ($this->volume_actual && $this->barragem->volume_maximo) {
            // 3. Calcular PERCENTAGEM TOTAL
            // FORMULA EXCEL: Q4 = (M4/3.9)*100
            // Onde: M4 = Volume Atual, 3.9 = Volume Maximo da barragem
            $this->percentagem_total = ($this->volume_actual / $this->barragem->volume_maximo) * 100;

            // 4. Calcular VOLUME UTIL
            // FORMULA EXCEL: N4 = M4 - E4
            // Onde: M4 = Volume Total, E4 = Volume Morto
            $volume_util = $this->volume_actual - ($this->barragem->volume_morto ?? 0);

            // 5. Calcular PERCENTAGEM DE ENCHIMENTO (% UTIL)
            // FORMULA EXCEL CORRETA: R4 = (N4 / Volume_Util_Maximo) * 100
            $volume_util_maximo = $this->barragem->volume_maximo - ($this->barragem->volume_morto ?? 0);

            if ($volume_util_maximo > 0) {
                $this->percentagem_enchimento = max(0, ($volume_util / $volume_util_maximo) * 100);
            } else {
                $this->percentagem_enchimento = 0;
            }

            // Limitar percentagens entre 0 e 100
            $this->percentagem_total = max(0, min(100, $this->percentagem_total));
            $this->percentagem_enchimento = max(0, min(100, $this->percentagem_enchimento));
        }

        // 6. Calcular variacao de volume (necessario para descarga)
        $this->calcularVariacaoVolume();

        // 7. Calcular descarga total baseado na formula do Excel
        $this->calcularDescargaTotal();

        // 8. Calcular comparacao com ano anterior
        $this->calcularComparacaoAnoAnterior();
    }

    private function calcularDescargaTotal()
    {
        // BASEADO NO EXCEL ORIGINAL - Fórmula R4: =N4/3.25
        // Onde N4 = variação de volume
        if ($this->variacao_volume !== null) {
            $this->descarga_total = $this->variacao_volume / 3.25;
        } else {
            // Fallback: soma dos componentes se existirem
            $this->descarga_total =
                ($this->caudal_efluente ?? 0) +
                ($this->descarga_fundo ?? 0) +
                ($this->descarga_superficie ?? 0) +
                ($this->caudal_ecologico ?? 0) +
                ($this->caudal_turbinado ?? 0) +
                ($this->caudal_descarregador ?? 0);
        }
    }

    private function calcularVariacaoVolume()
    {
        $leitura_anterior = static::where('barragem_id', $this->barragem_id)
            ->where('data_leitura', '<', $this->data_leitura)
            ->orderBy('data_leitura', 'desc')
            ->first();

        if ($leitura_anterior && $this->volume_actual && $leitura_anterior->volume_actual) {
            $this->variacao_volume = $this->volume_actual - $leitura_anterior->volume_actual;
        }
    }

    private function calcularComparacaoAnoAnterior()
    {
        $data_ano_anterior = $this->data_leitura->copy()->subYear();
        
        $leitura_ano_anterior = static::where('barragem_id', $this->barragem_id)
            ->whereDate('data_leitura', $data_ano_anterior)
            ->first();

        if ($leitura_ano_anterior && $this->percentagem_enchimento && $leitura_ano_anterior->percentagem_enchimento) {
            $this->comparacao_ano_anterior = $this->percentagem_enchimento - $leitura_ano_anterior->percentagem_enchimento;
        }
    }

    public function getStatusValidacao()
    {
        if ($this->validado) {
            return 'validado';
        }
        
        if ($this->created_at->diffInHours(now()) > 24) {
            return 'pendente';
        }
        
        return 'recente';
    }
}