<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class NotificacaoEnviada extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'alerta_id',
        'tipo',
        'categoria',
        'destinatario',
        'conteudo',
        'status',
        'external_id',
        'resposta_servico',
        'erro_detalhes',
        'enviado_em',
        'entregue_em',
    ];

    protected $casts = [
        'resposta_servico' => 'array',
        'enviado_em' => 'datetime',
        'entregue_em' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function alerta(): BelongsTo
    {
        return $this->belongsTo(Alerta::class);
    }

    /**
     * Marcar como enviado
     */
    public function marcarComoEnviado($messageId, $resposta = null)
    {
        $this->update([
            'status' => 'enviado',
            'external_id' => $messageId,
            'resposta_servico' => $resposta,
            'enviado_em' => now(),
        ]);
    }

    /**
     * Marcar como entregue
     */
    public function marcarComoEntregue()
    {
        $this->update([
            'status' => 'entregue',
            'entregue_em' => now(),
        ]);
    }

    /**
     * Marcar como falhado
     */
    public function marcarComoFalhado($erro)
    {
        $this->update([
            'status' => 'falhado',
            'erro_detalhes' => $erro,
        ]);
    }

    /**
     * Scope para notificações pendentes
     */
    public function scopePendentes($query)
    {
        return $query->where('status', 'pendente');
    }

    /**
     * Scope para notificações enviadas
     */
    public function scopeEnviadas($query)
    {
        return $query->whereIn('status', ['enviado', 'entregue']);
    }

    /**
     * Scope para notificações falhadas
     */
    public function scopeFalhadas($query)
    {
        return $query->where('status', 'falhado');
    }

    /**
     * Scope por tipo de notificação
     */
    public function scopeTipo($query, $tipo)
    {
        return $query->where('tipo', $tipo);
    }

    /**
     * Scope por categoria
     */
    public function scopeCategoria($query, $categoria)
    {
        return $query->where('categoria', $categoria);
    }

    /**
     * Scope para notificações de hoje
     */
    public function scopeHoje($query)
    {
        return $query->whereDate('created_at', today());
    }

    /**
     * Estatísticas de notificações
     */
    public static function getStats($periodo = 'hoje')
    {
        $query = static::query();

        if ($periodo === 'hoje') {
            $query->whereDate('created_at', today());
        } elseif ($periodo === 'semana') {
            $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
        } elseif ($periodo === 'mes') {
            $query->whereMonth('created_at', now()->month);
        }

        return [
            'total' => $query->count(),
            'enviadas' => $query->clone()->whereIn('status', ['enviado', 'entregue'])->count(),
            'pendentes' => $query->clone()->where('status', 'pendente')->count(),
            'falhadas' => $query->clone()->where('status', 'falhado')->count(),
            'sms' => $query->clone()->where('tipo', 'sms')->count(),
            'email' => $query->clone()->where('tipo', 'email')->count(),
            'emergencias' => $query->clone()->where('categoria', 'emergencia')->count(),
            'escalacoes' => $query->clone()->where('categoria', 'escalacao')->count(),
        ];
    }

    /**
     * Taxa de entrega para um usuário
     */
    public static function getTaxaEntrega($userId, $dias = 30)
    {
        $total = static::where('user_id', $userId)
            ->where('created_at', '>=', now()->subDays($dias))
            ->count();

        $entregues = static::where('user_id', $userId)
            ->where('created_at', '>=', now()->subDays($dias))
            ->where('status', 'entregue')
            ->count();

        return $total > 0 ? round(($entregues / $total) * 100, 2) : 0;
    }

    /**
     * Formatação do status
     */
    public function getStatusFormatado()
    {
        $status = [
            'pendente' => 'Pendente',
            'enviado' => 'Enviado',
            'entregue' => 'Entregue',
            'falhado' => 'Falhado',
        ];

        return $status[$this->status] ?? $this->status;
    }

    /**
     * Formatação da categoria
     */
    public function getCategoriaFormatada()
    {
        $categorias = [
            'alerta' => 'Alerta',
            'emergencia' => 'Emergência',
            'escalacao' => 'Escalação',
            'resolucao' => 'Resolução',
            'relatorio' => 'Relatório',
        ];

        return $categorias[$this->categoria] ?? $this->categoria;
    }

    /**
     * Cor do status
     */
    public function getCorStatus()
    {
        $cores = [
            'pendente' => 'yellow',
            'enviado' => 'blue',
            'entregue' => 'green',
            'falhado' => 'red',
        ];

        return $cores[$this->status] ?? 'gray';
    }
}