<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

/**
 * Modelo para atribuicoes de divisoes a utilizadores
 * Funciona como tabela pivot com dados adicionais
 */
class UserDivisao extends Model
{
    protected $table = 'user_divisoes';

    protected $fillable = [
        'user_id',
        'divisao_administrativa_id',
        'permissoes_customizadas',
        'nivel_acesso',
        'is_primary',
        'valido_de',
        'valido_ate',
        'ativo',
    ];

    protected $casts = [
        'permissoes_customizadas' => 'array',
        'is_primary' => 'boolean',
        'valido_de' => 'date',
        'valido_ate' => 'date',
        'ativo' => 'boolean',
    ];

    /**
     * Utilizador desta atribuicao
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Divisao administrativa atribuida
     */
    public function divisao(): BelongsTo
    {
        return $this->belongsTo(DivisaoAdministrativa::class, 'divisao_administrativa_id');
    }

    /**
     * Verificar se a atribuicao esta actualmente valida
     */
    public function isValida(): bool
    {
        if (!$this->ativo) {
            return false;
        }

        $hoje = Carbon::today();

        // Verificar data de inicio
        if ($this->valido_de && $hoje->lt($this->valido_de)) {
            return false;
        }

        // Verificar data de fim
        if ($this->valido_ate && $hoje->gt($this->valido_ate)) {
            return false;
        }

        return true;
    }

    /**
     * Verificar se tem permissao customizada concedida
     */
    public function temPermissaoConcedida(string $permissao): bool
    {
        if (!$this->permissoes_customizadas) {
            return false;
        }

        $grants = $this->permissoes_customizadas['grant'] ?? [];
        return in_array($permissao, $grants);
    }

    /**
     * Verificar se tem permissao customizada revogada
     */
    public function temPermissaoRevogada(string $permissao): bool
    {
        if (!$this->permissoes_customizadas) {
            return false;
        }

        $revokes = $this->permissoes_customizadas['revoke'] ?? [];
        return in_array($permissao, $revokes);
    }

    /**
     * Scope para atribuicoes activas e validas
     */
    public function scopeValidas($query)
    {
        return $query->where('ativo', true)
            ->where(function ($q) {
                $q->whereNull('valido_de')
                  ->orWhere('valido_de', '<=', Carbon::today());
            })
            ->where(function ($q) {
                $q->whereNull('valido_ate')
                  ->orWhere('valido_ate', '>=', Carbon::today());
            });
    }

    /**
     * Scope para atribuicoes primarias
     */
    public function scopePrimarias($query)
    {
        return $query->where('is_primary', true);
    }

    /**
     * Obter nome da divisao formatado
     */
    public function getNomeDivisaoAttribute(): string
    {
        return $this->divisao?->nome_completo ?? $this->divisao?->nome ?? 'N/A';
    }

    /**
     * Obter status formatado
     */
    public function getStatusAttribute(): string
    {
        if (!$this->ativo) {
            return 'Inactivo';
        }

        $hoje = Carbon::today();

        if ($this->valido_de && $hoje->lt($this->valido_de)) {
            return 'Pendente';
        }

        if ($this->valido_ate && $hoje->gt($this->valido_ate)) {
            return 'Expirado';
        }

        return 'Activo';
    }
}
