<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Estacao;
use App\Services\RegionalAccessService;
use Illuminate\Auth\Access\HandlesAuthorization;

/**
 * Policy para controle de acesso a Estacoes
 */
class EstacaoPolicy
{
    use HandlesAuthorization;

    protected function getAccessService(User $user): RegionalAccessService
    {
        return new RegionalAccessService($user);
    }

    /**
     * Verificar se pode ver lista de estacoes
     */
    public function viewAny(User $user): bool
    {
        return $user->can('ver-dashboard');
    }

    /**
     * Verificar se pode ver uma estacao especifica
     */
    public function view(User $user, Estacao $estacao): bool
    {
        $accessService = $this->getAccessService($user);
        return $accessService->canAccessEstacao($estacao);
    }

    /**
     * Verificar se pode criar estacoes
     */
    public function create(User $user): bool
    {
        return $user->can('criar-estacoes');
    }

    /**
     * Verificar se pode editar uma estacao
     */
    public function update(User $user, Estacao $estacao): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessEstacao($estacao)) {
            return false;
        }

        return $accessService->canPerformAction(
            'editar-estacoes',
            $estacao->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode eliminar uma estacao
     */
    public function delete(User $user, Estacao $estacao): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessEstacao($estacao)) {
            return false;
        }

        return $accessService->canPerformAction(
            'eliminar-estacoes',
            $estacao->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode gerir leituras
     */
    public function manageLeituras(User $user, Estacao $estacao): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessEstacao($estacao)) {
            return false;
        }

        return $accessService->canPerformAction(
            'gerir-leituras-estacoes',
            $estacao->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode validar leituras
     */
    public function validateLeituras(User $user, Estacao $estacao): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessEstacao($estacao)) {
            return false;
        }

        return $accessService->canPerformAction(
            'validar-leituras-estacoes',
            $estacao->divisao_administrativa_id
        );
    }
}
