<?php

namespace App\Services;

use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class AnaliseHidrologicaService
{
    /**
     * Gerar análise completa para boletim hidrológico
     */
    public function gerarAnaliseCompleta(): array
    {
        return [
            'barragens' => $this->analisarBarragens(),
            'estacoes' => $this->analisarEstacoes(),
            'resumo_geral' => $this->gerarResumoGeral(),
            'previsao' => $this->gerarPrevisao(),
            'alertas' => $this->identificarAlertas(),
        ];
    }

    /**
     * Analisar todas as barragens ativas
     */
    public function analisarBarragens(): array
    {
        $barragens = Barragem::where('estado', 'activa')
            ->with('ultima_leitura')
            ->get();

        $analises = [];

        foreach ($barragens as $barragem) {
            $analises[] = $this->analisarBarragem($barragem);
        }

        return $analises;
    }

    /**
     * Análise detalhada de uma barragem
     */
    public function analisarBarragem(Barragem $barragem): array
    {
        $leituras7dias = $this->getLeiturasPeriodo($barragem, 7);
        $leituras30dias = $this->getLeiturasPeriodo($barragem, 30);
        $leituraAnoAnterior = $this->getLeituraAnoAnterior($barragem);

        $tendencia = $this->calcularTendencia($leituras7dias, 'percentagem_enchimento');
        $variacaoVolume = $this->calcularVariacao($leituras7dias, 'volume_actual');
        $comparacaoAnual = $this->compararComAnoAnterior($barragem->ultima_leitura, $leituraAnoAnterior);

        return [
            'barragem_id' => $barragem->id,
            'nome' => $barragem->nome,
            'enchimento_atual' => $barragem->ultima_leitura->percentagem_enchimento ?? 0,
            'volume_atual' => $barragem->ultima_leitura->volume_actual ?? 0,
            'cota_atual' => $barragem->ultima_leitura->cota_actual ?? 0,
            'tendencia' => $tendencia,
            'variacao_7dias' => $variacaoVolume,
            'comparacao_ano_anterior' => $comparacaoAnual,
            'status' => $this->determinarStatusBarragem($barragem),
            'descricao' => $this->gerarDescricaoBarragem($barragem, $tendencia, $comparacaoAnual),
        ];
    }

    /**
     * Calcular tendência (subida, descida, estável)
     */
    private function calcularTendencia(Collection $leituras, string $campo): array
    {
        if ($leituras->count() < 2) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        $valores = $leituras->pluck($campo)->filter()->values();

        if ($valores->count() < 2) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        $primeiro = $valores->last(); // Mais antigo
        $ultimo = $valores->first();  // Mais recente

        if ($primeiro == 0) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        $variacao = (($ultimo - $primeiro) / $primeiro) * 100;

        if ($variacao > 2) {
            return [
                'tipo' => 'subida',
                'valor' => round($variacao, 2),
                'texto' => 'tendência de subida'
            ];
        } elseif ($variacao < -2) {
            return [
                'tipo' => 'descida',
                'valor' => round($variacao, 2),
                'texto' => 'tendência de descida'
            ];
        } else {
            return [
                'tipo' => 'estavel',
                'valor' => round($variacao, 2),
                'texto' => 'nível estacionário'
            ];
        }
    }

    /**
     * Calcular variação absoluta no período
     */
    private function calcularVariacao(Collection $leituras, string $campo): array
    {
        if ($leituras->count() < 2) {
            return ['valor' => 0, 'percentual' => 0];
        }

        $valores = $leituras->pluck($campo)->filter()->values();

        if ($valores->count() < 2) {
            return ['valor' => 0, 'percentual' => 0];
        }

        $primeiro = $valores->last();
        $ultimo = $valores->first();
        $diferenca = $ultimo - $primeiro;
        $percentual = $primeiro > 0 ? (($diferenca) / $primeiro) * 100 : 0;

        return [
            'valor' => round($diferenca, 3),
            'percentual' => round($percentual, 2)
        ];
    }

    /**
     * Comparar com o mesmo período do ano anterior
     */
    private function compararComAnoAnterior($leituraAtual, $leituraAnoAnterior): array
    {
        if (!$leituraAtual || !$leituraAnoAnterior) {
            return [
                'disponivel' => false,
                'diferenca' => 0,
                'texto' => 'Sem dados do ano anterior'
            ];
        }

        $enchimentoAtual = $leituraAtual->percentagem_enchimento ?? 0;
        $enchimentoAnterior = $leituraAnoAnterior->percentagem_enchimento ?? 0;
        $diferenca = $enchimentoAtual - $enchimentoAnterior;

        $texto = $diferenca > 0
            ? sprintf('%.1f%% acima do ano anterior', abs($diferenca))
            : ($diferenca < 0
                ? sprintf('%.1f%% abaixo do ano anterior', abs($diferenca))
                : 'igual ao ano anterior');

        return [
            'disponivel' => true,
            'diferenca' => round($diferenca, 2),
            'texto' => $texto
        ];
    }

    /**
     * Determinar status da barragem
     */
    private function determinarStatusBarragem(Barragem $barragem): string
    {
        $leitura = $barragem->ultima_leitura;

        if (!$leitura) {
            return 'sem_dados';
        }

        $cota = $leitura->cota_actual ?? 0;
        $nivelEmergencia = $barragem->nivel_emergencia ?? PHP_INT_MAX;
        $nivelAlerta = $barragem->nivel_alerta ?? PHP_INT_MAX;

        if ($cota >= $nivelEmergencia) {
            return 'emergencia';
        } elseif ($cota >= $nivelAlerta) {
            return 'alerta';
        } else {
            return 'normal';
        }
    }

    /**
     * Gerar descrição contextual da barragem
     */
    private function gerarDescricaoBarragem(Barragem $barragem, array $tendencia, array $comparacaoAnual): string
    {
        $leitura = $barragem->ultima_leitura;

        if (!$leitura) {
            return "Sem dados recentes disponíveis para a Barragem de {$barragem->nome}.";
        }

        $enchimento = number_format($leitura->percentagem_enchimento ?? 0, 2);
        $volume = number_format($leitura->volume_actual ?? 0, 3);

        // Construir descrição base
        $descricao = "A Barragem de {$barragem->nome} apresenta enchimento de {$enchimento}% ({$volume} Mm³)";

        // Adicionar tendência
        if ($tendencia['tipo'] !== 'sem_dados') {
            $descricao .= ", com {$tendencia['texto']}";
            if ($tendencia['valor'] != 0) {
                $descricao .= " (" . ($tendencia['valor'] > 0 ? '+' : '') . "{$tendencia['valor']}%)";
            }
        }

        // Adicionar comparação anual
        if ($comparacaoAnual['disponivel']) {
            $descricao .= ", {$comparacaoAnual['texto']}";
        }

        $descricao .= ".";

        return $descricao;
    }

    /**
     * Analisar estações hidrométricas
     */
    public function analisarEstacoes(): array
    {
        $estacoes = Estacao::where('estado', 'activa')
            ->whereIn('tipo', ['hidrometrica', 'hidro'])
            ->with('ultima_leitura')
            ->get();

        $analises = [];

        foreach ($estacoes as $estacao) {
            $leituras7dias = $this->getLeiturasEstacaoPeriodo($estacao, 7);
            $tendencia = $this->calcularTendenciaEstacao($leituras7dias);

            $analises[] = [
                'estacao_id' => $estacao->id,
                'nome' => $estacao->nome,
                'bacia' => $estacao->bacia_hidrografica->nome ?? 'N/D',
                'nivel_atual' => $estacao->ultima_leitura->nivel_hidrometrico ?? 0,
                'tendencia' => $tendencia,
                'nivel_alerta' => $estacao->nivel_alerta,
                'status' => $this->determinarStatusEstacao($estacao),
            ];
        }

        return $analises;
    }

    /**
     * Calcular tendência para estação
     */
    private function calcularTendenciaEstacao(Collection $leituras): array
    {
        if ($leituras->count() < 2) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        // Usar nível hidrométrico ou média dos níveis
        $niveis = $leituras->map(function ($l) {
            return $l->nivel_hidrometrico ?? (
                ($l->nivel_6h + $l->nivel_12h + $l->nivel_18h) / 3
            );
        })->filter()->values();

        if ($niveis->count() < 2) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        $primeiro = $niveis->last();
        $ultimo = $niveis->first();

        if ($primeiro == 0) {
            return ['tipo' => 'sem_dados', 'valor' => 0, 'texto' => 'Dados insuficientes'];
        }

        $variacao = (($ultimo - $primeiro) / $primeiro) * 100;

        if ($variacao > 5) {
            return ['tipo' => 'subida', 'valor' => round($variacao, 2), 'texto' => 'em subida'];
        } elseif ($variacao < -5) {
            return ['tipo' => 'descida', 'valor' => round($variacao, 2), 'texto' => 'em descida'];
        } else {
            return ['tipo' => 'estavel', 'valor' => round($variacao, 2), 'texto' => 'estacionário'];
        }
    }

    /**
     * Determinar status da estação
     */
    private function determinarStatusEstacao(Estacao $estacao): string
    {
        $leitura = $estacao->ultima_leitura;

        if (!$leitura) {
            return 'sem_dados';
        }

        $nivel = $leitura->nivel_hidrometrico ?? 0;
        $nivelAlerta = $estacao->nivel_alerta ?? PHP_INT_MAX;

        return $nivel >= $nivelAlerta ? 'alerta' : 'normal';
    }

    /**
     * Gerar resumo geral da situação hidrológica
     */
    public function gerarResumoGeral(): string
    {
        $barragens = $this->analisarBarragens();
        $estacoes = $this->analisarEstacoes();

        // Contar tendências das barragens
        $tendenciasBarragens = collect($barragens)->groupBy('tendencia.tipo');
        $subindo = $tendenciasBarragens->get('subida', collect())->count();
        $descendo = $tendenciasBarragens->get('descida', collect())->count();
        $estaveis = $tendenciasBarragens->get('estavel', collect())->count();

        // Contar tendências das estações
        $tendenciasEstacoes = collect($estacoes)->groupBy('tendencia.tipo');
        $estacoesSubindo = $tendenciasEstacoes->get('subida', collect())->count();
        $estacoesDescendo = $tendenciasEstacoes->get('descida', collect())->count();

        // Construir resumo
        $resumo = "Nas últimas 24 horas, observou-se ";

        // Análise das barragens
        if ($subindo > $descendo && $subindo > $estaveis) {
            $resumo .= "tendência geral de subida dos níveis nas albufeiras da região";
        } elseif ($descendo > $subindo && $descendo > $estaveis) {
            $resumo .= "tendência geral de descida dos níveis nas albufeiras da região";
        } else {
            $resumo .= "níveis relativamente estáveis nas albufeiras da região";
        }

        // Análise das estações
        if ($estacoesSubindo > $estacoesDescendo) {
            $resumo .= ", com os níveis hidrométricos nos rios a apresentar ligeira subida";
        } elseif ($estacoesDescendo > $estacoesSubindo) {
            $resumo .= ", com os níveis hidrométricos nos rios a baixar progressivamente";
        } else {
            $resumo .= ", com os níveis hidrométricos nos rios mantendo-se estacionários";
        }

        $resumo .= ".";

        // Calcular média de enchimento
        $mediaEnchimento = collect($barragens)->avg('enchimento_atual');
        if ($mediaEnchimento > 0) {
            $resumo .= sprintf(" A média de enchimento das albufeiras situa-se em %.1f%%.", $mediaEnchimento);
        }

        return $resumo;
    }

    /**
     * Gerar previsão hidrológica para as próximas 24h
     */
    public function gerarPrevisao(): string
    {
        $barragens = $this->analisarBarragens();

        // Analisar tendências recentes para prever comportamento
        $tendencias = collect($barragens)->pluck('tendencia.tipo');
        $predominante = $tendencias->countBy()->sortDesc()->keys()->first();

        $previsoes = [
            'subida' => [
                "Para as próximas 24 horas, prevê-se continuação da tendência de subida dos níveis hidrométricos, particularmente nas bacias que registaram precipitação nas últimas horas.",
                "Espera-se manutenção da tendência de subida dos níveis nas albufeiras, recomendando-se vigilância nas áreas de maior acumulação.",
            ],
            'descida' => [
                "Para as próximas 24 horas, prevê-se manutenção da tendência de descida gradual dos níveis hidrométricos na região.",
                "Não se prevêem alterações significativas do actual cenário hidrológico, com os níveis a manterem tendência de descida progressiva.",
            ],
            'estavel' => [
                "Para as próximas 24 horas, não se prevêem alterações significativas do actual cenário hidrológico.",
                "Espera-se manutenção dos níveis actuais, sem variações expressivas nas próximas 24 horas.",
            ],
            'sem_dados' => [
                "Dados insuficientes para elaboração de previsão detalhada. Recomenda-se acompanhamento contínuo.",
            ],
        ];

        $opcoes = $previsoes[$predominante] ?? $previsoes['estavel'];
        return $opcoes[array_rand($opcoes)];
    }

    /**
     * Identificar situações de alerta
     */
    public function identificarAlertas(): array
    {
        $alertas = [];

        // Verificar barragens
        $barragens = Barragem::where('estado', 'activa')->with('ultima_leitura')->get();

        foreach ($barragens as $barragem) {
            $leitura = $barragem->ultima_leitura;
            if (!$leitura) continue;

            $cota = $leitura->cota_actual ?? 0;

            if ($barragem->nivel_emergencia && $cota >= $barragem->nivel_emergencia) {
                $alertas[] = [
                    'tipo' => 'emergencia',
                    'entidade' => 'barragem',
                    'nome' => $barragem->nome,
                    'mensagem' => "Nível de EMERGÊNCIA atingido na Barragem de {$barragem->nome}",
                ];
            } elseif ($barragem->nivel_alerta && $cota >= $barragem->nivel_alerta) {
                $alertas[] = [
                    'tipo' => 'alerta',
                    'entidade' => 'barragem',
                    'nome' => $barragem->nome,
                    'mensagem' => "Nível de ALERTA atingido na Barragem de {$barragem->nome}",
                ];
            }
        }

        // Verificar estações
        $estacoes = Estacao::where('estado', 'activa')->with('ultima_leitura')->get();

        foreach ($estacoes as $estacao) {
            $leitura = $estacao->ultima_leitura;
            if (!$leitura || !$estacao->nivel_alerta) continue;

            $nivel = $leitura->nivel_hidrometrico ?? 0;

            if ($nivel >= $estacao->nivel_alerta) {
                $alertas[] = [
                    'tipo' => 'alerta',
                    'entidade' => 'estacao',
                    'nome' => $estacao->nome,
                    'mensagem' => "Nível de ALERTA atingido na Estação {$estacao->nome}",
                ];
            }
        }

        return $alertas;
    }

    /**
     * Gerar texto de alertas e precauções
     */
    public function gerarTextoPrecaucoes(): string
    {
        $alertas = $this->identificarAlertas();

        $texto = "A ARA-Norte, IP, apela à sociedade para a observância de medidas de precaução";

        if (count($alertas) > 0) {
            $emergencias = collect($alertas)->where('tipo', 'emergencia');

            if ($emergencias->count() > 0) {
                $texto .= ", especialmente nas proximidades de: " .
                    $emergencias->pluck('nome')->implode(', ') .
                    ". Recomenda-se evitar actividades junto aos cursos de água nestas áreas";
            }
        }

        $texto .= ". Evitar a travessia de cursos de água em locais não sinalizados. ";
        $texto .= "Apela-se igualmente ao acompanhamento da informação hidrológica disseminada pela ARA Norte, IP, e outras entidades competentes.";

        return $texto;
    }

    /**
     * Gerar análise de níveis hidrométricos para o boletim
     */
    public function gerarAnaliseNiveis(): string
    {
        $estacoes = $this->analisarEstacoes();

        if (empty($estacoes)) {
            return "Não há dados disponíveis de níveis hidrométricos para análise.";
        }

        $tendencias = collect($estacoes)->groupBy('tendencia.tipo');
        $subindo = $tendencias->get('subida', collect());
        $descendo = $tendencias->get('descida', collect());
        $estaveis = $tendencias->get('estavel', collect());

        $texto = "• Nas últimas 24h observou-se ";

        if ($subindo->count() > $descendo->count()) {
            $texto .= "tendência de subida dos níveis hidrométricos nas bacias hidrográficas da região";
            if ($subindo->count() > 0) {
                $bacias = $subindo->pluck('bacia')->unique()->filter()->implode(', ');
                if ($bacias) {
                    $texto .= ", nomeadamente nas bacias do {$bacias}";
                }
            }
        } elseif ($descendo->count() > $subindo->count()) {
            $texto .= "tendência de descida dos níveis hidrométricos nas bacias hidrográficas da região";
        } else {
            $texto .= "níveis hidrométricos relativamente estáveis nas bacias hidrográficas da região";
        }

        $texto .= ".";

        return $texto;
    }

    // ========== Métodos auxiliares ==========

    private function getLeiturasPeriodo(Barragem $barragem, int $dias): Collection
    {
        return LeituraBarragem::where('barragem_id', $barragem->id)
            ->where('data_leitura', '>=', Carbon::now()->subDays($dias))
            ->orderBy('data_leitura', 'desc')
            ->get();
    }

    private function getLeiturasEstacaoPeriodo(Estacao $estacao, int $dias): Collection
    {
        return LeituraEstacao::where('estacao_id', $estacao->id)
            ->where('data_leitura', '>=', Carbon::now()->subDays($dias))
            ->orderBy('data_leitura', 'desc')
            ->get();
    }

    private function getLeituraAnoAnterior(Barragem $barragem)
    {
        $dataAnoAnterior = Carbon::now()->subYear();

        return LeituraBarragem::where('barragem_id', $barragem->id)
            ->whereDate('data_leitura', $dataAnoAnterior->toDateString())
            ->first();
    }
}
