<?php

namespace App\Services;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Chart\Chart;
use PhpOffice\PhpSpreadsheet\Chart\DataSeries;
use PhpOffice\PhpSpreadsheet\Chart\DataSeriesValues;
use PhpOffice\PhpSpreadsheet\Chart\Legend;
use PhpOffice\PhpSpreadsheet\Chart\PlotArea;
use PhpOffice\PhpSpreadsheet\Chart\Title;

class ExcelChartService
{
    /**
     * Gerar gráfico de barragem estilo Excel
     */
    public function gerarGraficoBarragem(array $config): string
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Dados');

        // Dados do gráfico
        $labels = $config['labels'] ?? ['O', 'N', 'D', 'J', 'F', 'M', 'A', 'M', 'J', 'J', 'A', 'S'];
        $datasets = $config['datasets'] ?? [];
        $precipitacao = $config['precipitacao'] ?? [];
        $title = $config['title'] ?? 'Barragem';

        // Inserir labels na linha 1 (coluna B em diante)
        $col = 'B';
        foreach ($labels as $label) {
            $sheet->setCellValue($col . '1', $label);
            $col++;
        }

        // Inserir dados das séries (linhas)
        $row = 2;
        foreach ($datasets as $dataset) {
            $sheet->setCellValue('A' . $row, $dataset['label']);
            $col = 'B';
            foreach ($dataset['data'] as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }
            $row++;
        }

        // Linha para precipitação
        $precipRow = $row;
        if (!empty($precipitacao)) {
            $sheet->setCellValue('A' . $precipRow, 'Precip.');
            $col = 'B';
            foreach ($precipitacao as $value) {
                $sheet->setCellValue($col . $precipRow, $value);
                $col++;
            }
        }

        // Calcular última coluna
        $lastCol = chr(ord('A') + count($labels));

        // === CRIAR GRÁFICO DE LINHAS (Cota) ===
        $lineLabels = [];
        $lineCategories = [];
        $lineValues = [];

        // Categorias (meses)
        $lineCategories[] = new DataSeriesValues(
            DataSeriesValues::DATASERIES_TYPE_STRING,
            'Dados!$B$1:$' . $lastCol . '$1',
            null,
            count($labels)
        );

        // Séries de dados (linhas)
        $row = 2;
        foreach ($datasets as $dataset) {
            $lineLabels[] = new DataSeriesValues(
                DataSeriesValues::DATASERIES_TYPE_STRING,
                'Dados!$A$' . $row,
                null,
                1
            );
            $lineValues[] = new DataSeriesValues(
                DataSeriesValues::DATASERIES_TYPE_NUMBER,
                'Dados!$B$' . $row . ':$' . $lastCol . '$' . $row,
                null,
                count($labels)
            );
            $row++;
        }

        // DataSeries para linhas
        $lineSeries = new DataSeries(
            DataSeries::TYPE_LINECHART,
            DataSeries::GROUPING_STANDARD,
            range(0, count($datasets) - 1),
            $lineLabels,
            $lineCategories,
            $lineValues
        );
        $lineSeries->setPlotDirection(DataSeries::DIRECTION_COL);

        // === CRIAR GRÁFICO DE BARRAS (Precipitação) - se houver dados ===
        $allSeries = [$lineSeries];

        if (!empty($precipitacao)) {
            $barLabels = [
                new DataSeriesValues(
                    DataSeriesValues::DATASERIES_TYPE_STRING,
                    'Dados!$A$' . $precipRow,
                    null,
                    1
                )
            ];
            $barCategories = [
                new DataSeriesValues(
                    DataSeriesValues::DATASERIES_TYPE_STRING,
                    'Dados!$B$1:$' . $lastCol . '$1',
                    null,
                    count($labels)
                )
            ];
            $barValues = [
                new DataSeriesValues(
                    DataSeriesValues::DATASERIES_TYPE_NUMBER,
                    'Dados!$B$' . $precipRow . ':$' . $lastCol . '$' . $precipRow,
                    null,
                    count($labels)
                )
            ];

            $barSeries = new DataSeries(
                DataSeries::TYPE_BARCHART,
                DataSeries::GROUPING_STANDARD,
                [0],
                $barLabels,
                $barCategories,
                $barValues
            );
            $barSeries->setPlotDirection(DataSeries::DIRECTION_COL);

            $allSeries[] = $barSeries;
        }

        // Plot Area com todas as séries
        $plotArea = new PlotArea(null, $allSeries);

        // Legenda
        $legend = new Legend(Legend::POSITION_BOTTOM, null, false);

        // Título
        $chartTitle = new Title($title);

        // Criar gráfico
        $chart = new Chart(
            'chart1',
            $chartTitle,
            $legend,
            $plotArea,
            true,
            DataSeries::EMPTY_AS_GAP,
            null,
            null
        );

        // Posicionar gráfico (ocupa toda a área visível)
        $chart->setTopLeftPosition('A' . ($precipRow + 2));
        $chart->setBottomRightPosition('O' . ($precipRow + 22));

        // Adicionar gráfico à planilha
        $sheet->addChart($chart);

        // Ocultar linhas de dados (só mostrar gráfico)
        for ($i = 1; $i <= $precipRow; $i++) {
            $sheet->getRowDimension($i)->setVisible(false);
        }

        // Salvar ficheiro
        $filename = storage_path('app/chart_' . uniqid() . '.xlsx');
        $writer = new Xlsx($spreadsheet);
        $writer->setIncludeCharts(true);
        $writer->save($filename);

        return $filename;
    }

    /**
     * Converter Excel para PNG usando LibreOffice
     */
    public function excelToPng(string $excelFile): ?string
    {
        $outputDir = dirname($excelFile);
        $basename = pathinfo($excelFile, PATHINFO_FILENAME);
        $pngFile = $outputDir . '/' . $basename . '.png';

        // Usar LibreOffice para converter
        $command = "/opt/homebrew/bin/soffice --headless --convert-to png --outdir " .
                   escapeshellarg($outputDir) . " " . escapeshellarg($excelFile) . " 2>&1";
        exec($command, $output, $returnCode);

        if (file_exists($pngFile)) {
            return $pngFile;
        }

        return null;
    }

    /**
     * Gerar gráfico completo e retornar como base64
     */
    public function gerarGraficoBase64(array $config): ?string
    {
        // Gerar Excel
        $excelFile = $this->gerarGraficoBarragem($config);

        if (!$excelFile || !file_exists($excelFile)) {
            return null;
        }

        // Converter para PNG
        $pngFile = $this->excelToPng($excelFile);

        if (!$pngFile || !file_exists($pngFile)) {
            @unlink($excelFile);
            return null;
        }

        // Ler e converter para base64
        $imageData = file_get_contents($pngFile);
        $base64 = 'data:image/png;base64,' . base64_encode($imageData);

        // Limpar ficheiros temporários
        @unlink($excelFile);
        @unlink($pngFile);

        return $base64;
    }
}
