<?php

namespace App\Services;

class IQACalculatorService
{
    /**
     * Calcular IQA usando método NSF-WQI
     */
    public function calcular($parametros)
    {
        $qi = [];

        // Calcular qi para cada parâmetro
        $qi['od'] = $this->calcularQi_OD($parametros->oxigenio_dissolvido ?? 0);
        $qi['cf'] = $this->calcularQi_ColiformesFecais($parametros->coliformes_fecais ?? 0);
        $qi['ph'] = $this->calcularQi_pH($parametros->ph ?? 7);
        $qi['dbo'] = $this->calcularQi_DBO($parametros->dbo ?? 0);
        $qi['n'] = $this->calcularQi_NitrogenioTotal($parametros->nitrogenio_total ?? 0);
        $qi['p'] = $this->calcularQi_FosfatoTotal($parametros->fosfato_total ?? 0);
        $qi['turb'] = $this->calcularQi_Turbidez($parametros->turbidez ?? 0);
        $qi['sol'] = $this->calcularQi_SolidosTotais($parametros->solidos_totais ?? 0);
        $qi['temp'] = $this->calcularQi_Temperatura($parametros->temperatura ?? 25);

        // Pesos dos parâmetros (NSF-WQI)
        $pesos = [
            'od' => 0.17,
            'cf' => 0.15,
            'ph' => 0.12,
            'dbo' => 0.10,
            'n' => 0.10,
            'p' => 0.10,
            'turb' => 0.08,
            'sol' => 0.08,
            'temp' => 0.10,
        ];

        // Cálculo do IQA (produtório)
        $iqa = 1;
        foreach ($qi as $parametro => $valor) {
            $iqa *= pow($valor, $pesos[$parametro]);
        }

        $classificacao = $this->classificarIQA($iqa);

        return [
            'iqa' => round($iqa, 2),
            'classificacao' => $classificacao,
            'detalhes' => $qi,
        ];
    }

    private function calcularQi_OD($od_percent)
    {
        if ($od_percent >= 140) return 47;
        if ($od_percent >= 100) return 93 + ($od_percent - 100) * (-46/40);
        if ($od_percent >= 85) return 85 + ($od_percent - 85) * (8/15);
        if ($od_percent >= 50) return 47 + ($od_percent - 50) * (38/35);
        if ($od_percent >= 10) return 5 + ($od_percent - 10) * (42/40);
        return 2;
    }

    private function calcularQi_ColiformesFecais($cf_nmp)
    {
        if ($cf_nmp <= 1) return 98;
        if ($cf_nmp <= 10) return 84 - ($cf_nmp - 1) * (14/9);
        if ($cf_nmp <= 100) return 37 - ($cf_nmp - 10) * (47/90);
        if ($cf_nmp <= 1000) return 17 - ($cf_nmp - 100) * (20/900);
        if ($cf_nmp <= 10000) return 5 - ($cf_nmp - 1000) * (12/9000);
        return 2;
    }

    private function calcularQi_pH($ph)
    {
        if ($ph >= 2 && $ph <= 2.5) return 2 + ($ph - 2) * (6/0.5);
        if ($ph > 2.5 && $ph <= 3) return 8 + ($ph - 2.5) * (7/0.5);
        if ($ph > 3 && $ph <= 4.5) return 15 + ($ph - 3) * (25/1.5);
        if ($ph > 4.5 && $ph <= 6.5) return 40 + ($ph - 4.5) * (39/2);
        if ($ph > 6.5 && $ph <= 7.5) return 79 + ($ph - 6.5) * (17/1);
        if ($ph > 7.5 && $ph <= 8) return 96 - ($ph - 7.5) * (4/0.5);
        if ($ph > 8 && $ph <= 8.5) return 92 - ($ph - 8) * (11/0.5);
        if ($ph > 8.5 && $ph <= 9) return 81 - ($ph - 8.5) * (28/0.5);
        if ($ph > 9 && $ph <= 10) return 53 - ($ph - 9) * (26/1);
        if ($ph > 10 && $ph <= 11) return 27 - ($ph - 10) * (15/1);
        if ($ph > 11 && $ph <= 12) return 12 - ($ph - 11) * (10/1);
        return 2;
    }

    private function calcularQi_DBO($dbo)
    {
        if ($dbo <= 1) return 100;
        if ($dbo <= 2) return 93 - ($dbo - 1) * (7/1);
        if ($dbo <= 3) return 86 - ($dbo - 2) * (7/1);
        if ($dbo <= 5) return 74 - ($dbo - 3) * (12/2);
        if ($dbo <= 10) return 46 - ($dbo - 5) * (28/5);
        if ($dbo <= 15) return 18 - ($dbo - 10) * (10/5);
        if ($dbo <= 20) return 8 - ($dbo - 15) * (6/5);
        return 2;
    }

    private function calcularQi_NitrogenioTotal($n_total)
    {
        if ($n_total <= 0.5) return 98;
        if ($n_total <= 1) return 88 - ($n_total - 0.5) * (10/0.5);
        if ($n_total <= 2) return 72 - ($n_total - 1) * (16/1);
        if ($n_total <= 5) return 42 - ($n_total - 2) * (30/3);
        if ($n_total <= 10) return 12 - ($n_total - 5) * (10/5);
        return 2;
    }

    private function calcularQi_FosfatoTotal($p_total)
    {
        if ($p_total <= 0.01) return 98;
        if ($p_total <= 0.05) return 84 - ($p_total - 0.01) * (14/0.04);
        if ($p_total <= 0.1) return 65 - ($p_total - 0.05) * (19/0.05);
        if ($p_total <= 0.5) return 25 - ($p_total - 0.1) * (40/0.4);
        if ($p_total <= 1) return 5 - ($p_total - 0.5) * (3/0.5);
        return 2;
    }

    private function calcularQi_Turbidez($turbidez)
    {
        if ($turbidez <= 5) return 100 - $turbidez * (10/5);
        if ($turbidez <= 25) return 90 - ($turbidez - 5) * (35/20);
        if ($turbidez <= 50) return 55 - ($turbidez - 25) * (30/25);
        if ($turbidez <= 100) return 25 - ($turbidez - 50) * (20/50);
        return 5;
    }

    private function calcularQi_SolidosTotais($solidos)
    {
        if ($solidos <= 50) return 95;
        if ($solidos <= 100) return 80 - ($solidos - 50) * (15/50);
        if ($solidos <= 250) return 50 - ($solidos - 100) * (30/150);
        if ($solidos <= 500) return 20 - ($solidos - 250) * (15/250);
        return 5;
    }

    private function calcularQi_Temperatura($temp)
    {
        $delta = abs($temp - 25);
        if ($delta <= 1) return 95;
        if ($delta <= 5) return 85 - ($delta - 1) * (10/4);
        if ($delta <= 10) return 65 - ($delta - 5) * (20/5);
        if ($delta <= 15) return 45 - ($delta - 10) * (20/5);
        return 25;
    }

    private function classificarIQA($iqa)
    {
        if ($iqa >= 91) return 'Excelente';
        if ($iqa >= 71) return 'Boa';
        if ($iqa >= 51) return 'Moderada';
        if ($iqa >= 26) return 'Má';
        return 'Muito Má';
    }
}
