<?php

namespace App\Services;

use App\Models\Alerta;
use App\Models\User;
use App\Models\NotificacaoEnviada;
use App\Services\TwilioSmsService;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Queue;
use Exception;

class NotificationService
{
    protected $twilioService;

    public function __construct(TwilioSmsService $twilioService)
    {
        $this->twilioService = $twilioService;
    }

    /**
     * Processar notificações para um novo alerta
     */
    public function processarNovoAlerta(Alerta $alerta)
    {
        Log::info('Processando notificações para novo alerta', ['alerta_id' => $alerta->id]);

        // Determinar usuários que devem ser notificados
        $usuarios = $this->determinarUsuariosParaNotificar($alerta);

        foreach ($usuarios as $usuario) {
            $this->enviarNotificacaoAlerta($usuario, $alerta);
        }

        // Agendar escalação se necessário
        if (config('services.notifications.escalation_enabled')) {
            $this->agendarEscalacao($alerta);
        }
    }

    /**
     * Processar notificação de resolução de alerta
     */
    public function processarResolucaoAlerta(Alerta $alerta)
    {
        Log::info('Processando notificações para resolução de alerta', ['alerta_id' => $alerta->id]);

        // Encontrar usuários que foram notificados sobre este alerta
        $usuariosNotificados = $this->getUsuariosNotificadosDoAlerta($alerta);

        foreach ($usuariosNotificados as $usuario) {
            $this->enviarNotificacaoResolucao($usuario, $alerta);
        }

        // Cancelar escalação se estava agendada
        $this->cancelarEscalacao($alerta);
    }

    /**
     * Enviar notificação de alerta para um usuário
     */
    public function enviarNotificacaoAlerta(User $usuario, Alerta $alerta)
    {
        // Verificar se usuário deve receber notificação deste nível
        if (!$this->usuarioDeveReceberNotificacao($usuario, $alerta)) {
            return;
        }

        $categoria = $alerta->nivel === 'emergencia' ? 'emergencia' : 'alerta';

        // Enviar SMS se habilitado
        if ($usuario->notificacao_sms && $usuario->telefone) {
            $this->enviarSms($usuario, $alerta, $categoria);
        }

        // Enviar email se habilitado
        if ($usuario->notificacao_email) {
            $this->enviarEmail($usuario, $alerta, $categoria);
        }
    }

    /**
     * Enviar notificação de resolução
     */
    public function enviarNotificacaoResolucao(User $usuario, Alerta $alerta)
    {
        // SMS de resolução
        if ($usuario->notificacao_sms && $usuario->telefone) {
            $this->enviarSms($usuario, $alerta, 'resolucao');
        }

        // Email de resolução
        if ($usuario->notificacao_email) {
            $this->enviarEmail($usuario, $alerta, 'resolucao');
        }
    }

    /**
     * Enviar SMS
     */
    protected function enviarSms(User $usuario, Alerta $alerta, string $categoria)
    {
        $telefone = $this->twilioService->formatPhoneNumber($usuario->telefone);
        
        if (!$telefone) {
            Log::warning('Número de telefone inválido', [
                'user_id' => $usuario->id,
                'telefone' => $usuario->telefone
            ]);
            return;
        }

        // Criar registro de notificação
        $notificacao = NotificacaoEnviada::create([
            'user_id' => $usuario->id,
            'alerta_id' => $alerta->id,
            'tipo' => 'sms',
            'categoria' => $categoria,
            'destinatario' => $telefone,
            'conteudo' => $this->gerarConteudoSms($alerta, $categoria),
            'status' => 'pendente',
        ]);

        try {
            // Enviar SMS baseado na categoria
            switch ($categoria) {
                case 'emergencia':
                    $resultado = $this->twilioService->sendEmergencyAlert($telefone, $alerta);
                    break;
                case 'resolucao':
                    $resultado = $this->twilioService->sendAlertResolved($telefone, $alerta);
                    break;
                case 'escalacao':
                    $minutos = now()->diffInMinutes($alerta->created_at);
                    $resultado = $this->twilioService->sendAlertEscalation($telefone, $alerta, $minutos);
                    break;
                default:
                    $resultado = $this->twilioService->sendAlert($telefone, $alerta);
            }

            if ($resultado['success']) {
                $notificacao->marcarComoEnviado($resultado['message_sid'], $resultado);
                Log::info('SMS enviado com sucesso', [
                    'user_id' => $usuario->id,
                    'alerta_id' => $alerta->id,
                    'message_sid' => $resultado['message_sid']
                ]);
            } else {
                $notificacao->marcarComoFalhado($resultado['error']);
                Log::error('Falha ao enviar SMS', [
                    'user_id' => $usuario->id,
                    'alerta_id' => $alerta->id,
                    'erro' => $resultado['error']
                ]);
            }
        } catch (Exception $e) {
            $notificacao->marcarComoFalhado($e->getMessage());
            Log::error('Exceção ao enviar SMS', [
                'user_id' => $usuario->id,
                'alerta_id' => $alerta->id,
                'erro' => $e->getMessage()
            ]);
        }
    }

    /**
     * Enviar email (placeholder - implementar conforme necessário)
     */
    protected function enviarEmail(User $usuario, Alerta $alerta, string $categoria)
    {
        if (!config('services.notifications.alert_email_enabled')) {
            return;
        }

        // Criar registro de notificação
        $notificacao = NotificacaoEnviada::create([
            'user_id' => $usuario->id,
            'alerta_id' => $alerta->id,
            'tipo' => 'email',
            'categoria' => $categoria,
            'destinatario' => $usuario->email,
            'conteudo' => $this->gerarConteudoEmail($alerta, $categoria),
            'status' => 'pendente',
        ]);

        try {
            // Aqui implementar envio de email com Mailable
            // Mail::to($usuario->email)->send(new AlertEmail($alerta, $categoria));
            
            $notificacao->marcarComoEnviado('email_' . time());
            Log::info('Email enviado com sucesso', [
                'user_id' => $usuario->id,
                'alerta_id' => $alerta->id
            ]);
        } catch (Exception $e) {
            $notificacao->marcarComoFalhado($e->getMessage());
            Log::error('Falha ao enviar email', [
                'user_id' => $usuario->id,
                'alerta_id' => $alerta->id,
                'erro' => $e->getMessage()
            ]);
        }
    }

    /**
     * Determinar usuários que devem receber notificação
     */
    protected function determinarUsuariosParaNotificar(Alerta $alerta)
    {
        $query = User::query();

        // Usuários com notificações habilitadas
        $query->where(function($q) {
            $q->where('notificacao_sms', true)
              ->orWhere('notificacao_email', true);
        });

        // Para emergências, incluir usuários com notificação de emergência
        if ($alerta->nivel === 'emergencia') {
            $query->where('notificacao_emergencia', true);
        }

        // Filtrar por níveis de notificação configurados
        $query->where(function($q) use ($alerta) {
            $q->whereNull('niveis_notificacao')
              ->orWhereRaw("JSON_CONTAINS(niveis_notificacao, '\"" . $alerta->nivel . "\"')");
        });

        return $query->get();
    }

    /**
     * Verificar se usuário deve receber notificação deste nível
     */
    protected function usuarioDeveReceberNotificacao(User $usuario, Alerta $alerta)
    {
        // Emergências sempre notificam se habilitado
        if ($alerta->nivel === 'emergencia' && $usuario->notificacao_emergencia) {
            return true;
        }

        // Verificar níveis configurados
        if ($usuario->niveis_notificacao) {
            $niveisConfigurados = json_decode($usuario->niveis_notificacao, true);
            return in_array($alerta->nivel, $niveisConfigurados);
        }

        // Se não há configuração específica, notificar alertas e acima
        $hierarquia = ['informacao', 'atencao', 'alerta', 'emergencia'];
        $nivelIndice = array_search($alerta->nivel, $hierarquia);
        
        return $nivelIndice >= 2; // alerta e emergencia
    }

    /**
     * Agendar escalação de alerta
     */
    public function agendarEscalacao(Alerta $alerta)
    {
        $tempoEscalacao = config('services.notifications.escalation_time_minutes', 30);
        
        // Agendar job para verificar se alerta ainda está ativo
        Queue::later(
            now()->addMinutes($tempoEscalacao),
            new \App\Jobs\ProcessarEscalacaoAlerta($alerta->id)
        );
    }

    /**
     * Processar escalação de alerta
     */
    public function processarEscalacao(Alerta $alerta)
    {
        // Verificar se alerta ainda está ativo
        if (!$alerta->isAtivo()) {
            return;
        }

        Log::info('Processando escalação de alerta', ['alerta_id' => $alerta->id]);

        // Encontrar supervisores e diretores
        $supervisores = User::whereIn('nivel_escalacao', ['supervisor', 'director'])
            ->where('notificacao_emergencia', true)
            ->get();

        foreach ($supervisores as $supervisor) {
            $this->enviarNotificacaoEscalacao($supervisor, $alerta);
        }
    }

    /**
     * Enviar notificação de escalação
     */
    protected function enviarNotificacaoEscalacao(User $usuario, Alerta $alerta)
    {
        if ($usuario->notificacao_sms && $usuario->telefone) {
            $this->enviarSms($usuario, $alerta, 'escalacao');
        }

        if ($usuario->notificacao_email) {
            $this->enviarEmail($usuario, $alerta, 'escalacao');
        }
    }

    /**
     * Cancelar escalação de alerta
     */
    protected function cancelarEscalacao(Alerta $alerta)
    {
        // Implementar cancelamento do job de escalação se necessário
        Log::info('Cancelando escalação de alerta', ['alerta_id' => $alerta->id]);
    }

    /**
     * Obter usuários notificados sobre um alerta
     */
    protected function getUsuariosNotificadosDoAlerta(Alerta $alerta)
    {
        return User::whereHas('notificacoesEnviadas', function($query) use ($alerta) {
            $query->where('alerta_id', $alerta->id)
                  ->whereIn('status', ['enviado', 'entregue']);
        })->get();
    }

    /**
     * Gerar conteúdo para SMS
     */
    protected function gerarConteudoSms(Alerta $alerta, string $categoria)
    {
        switch ($categoria) {
            case 'emergencia':
                return "🚨 EMERGÊNCIA: {$alerta->titulo} em {$alerta->getNomeInfrastrutura()}";
            case 'resolucao':
                return "✅ RESOLVIDO: {$alerta->titulo}";
            case 'escalacao':
                return "⚠️ ESCALAÇÃO: Alerta sem resposta - {$alerta->titulo}";
            default:
                return "🔔 ALERTA: {$alerta->titulo} em {$alerta->getNomeInfrastrutura()}";
        }
    }

    /**
     * Gerar conteúdo para email
     */
    protected function gerarConteudoEmail(Alerta $alerta, string $categoria)
    {
        return json_encode([
            'subject' => $this->gerarAssuntoEmail($alerta, $categoria),
            'body' => $this->gerarCorpoEmail($alerta, $categoria)
        ]);
    }

    /**
     * Gerar assunto do email
     */
    protected function gerarAssuntoEmail(Alerta $alerta, string $categoria)
    {
        $prefixos = [
            'emergencia' => '🚨 EMERGÊNCIA',
            'alerta' => '🔔 ALERTA',
            'resolucao' => '✅ RESOLVIDO',
            'escalacao' => '⚠️ ESCALAÇÃO'
        ];

        $prefixo = $prefixos[$categoria] ?? '📢 NOTIFICAÇÃO';
        
        return "{$prefixo} - {$alerta->titulo} - Sistema Hidrológico";
    }

    /**
     * Gerar corpo do email
     */
    protected function gerarCorpoEmail(Alerta $alerta, string $categoria)
    {
        return [
            'alerta' => $alerta,
            'categoria' => $categoria,
            'sistema' => 'Sistema Hidrológico - Moçambique'
        ];
    }

    /**
     * Enviar relatório diário
     */
    public function enviarRelatorioDiario()
    {
        $usuarios = User::where('receber_relatorio_diario', true)->get();
        
        foreach ($usuarios as $usuario) {
            $stats = $this->gerarEstatisticasDiarias();
            
            if ($usuario->notificacao_sms && $usuario->telefone) {
                $this->twilioService->sendDailyReport($usuario->telefone, $stats);
            }
        }
    }

    /**
     * Gerar estatísticas diárias
     */
    protected function gerarEstatisticasDiarias()
    {
        return [
            'alertas_hoje' => Alerta::whereDate('created_at', today())->count(),
            'alertas_ativos' => Alerta::where('estado', 'activo')->count(),
            'alertas_resolvidos' => Alerta::whereDate('data_fim', today())->count(),
            'emergencias' => Alerta::where('estado', 'activo')->where('nivel', 'emergencia')->count(),
        ];
    }

    /**
     * Testar configuração de notificações
     */
    public function testarConfiguracao($telefone)
    {
        return $this->twilioService->testConnection($telefone);
    }
}