<?php

namespace App\Services;

use App\Models\Estacao;
use App\Models\LeituraEstacao;
use App\Models\SebaIntegracaoLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Exception;

class SebaHydrometrieService
{
    protected $apiUrl;
    protected $username;
    protected $password;
    protected $apiKey;
    protected $timeout;

    public function __construct()
    {
        $this->apiUrl = config('services.seba.api_url');
        $this->username = config('services.seba.username');
        $this->password = config('services.seba.password');
        $this->apiKey = config('services.seba.api_key');
        $this->timeout = config('services.seba.timeout', 30);
    }

    /**
     * Importar dados de uma estação específica
     */
    public function importarDadosEstacao(Estacao $estacao, Carbon $dataInicio = null, Carbon $dataFim = null): array
    {
        try {
            $log = $this->iniciarLog($estacao, 'importacao_estacao');

            // Se não especificadas, usar últimas 24 horas
            $dataInicio = $dataInicio ?? now()->subDay();
            $dataFim = $dataFim ?? now();

            Log::info('Iniciando importação SEBA', [
                'estacao_id' => $estacao->id,
                'estacao_seba_id' => $estacao->seba_station_id,
                'periodo' => $dataInicio->format('Y-m-d H:i') . ' - ' . $dataFim->format('Y-m-d H:i')
            ]);

            if (!$estacao->seba_station_id) {
                throw new Exception("Estação {$estacao->nome} não possui ID SEBA configurado");
            }

            $dados = $this->buscarDadosEstacao($estacao->seba_station_id, $dataInicio, $dataFim);

            if (empty($dados)) {
                $this->finalizarLog($log, 'sucesso', 'Nenhum dado novo encontrado', 0, 0);
                return ['success' => true, 'message' => 'Nenhum dado novo encontrado', 'imported' => 0];
            }

            $resultado = $this->processarDados($estacao, $dados);

            $this->finalizarLog($log, 'sucesso', $resultado['message'], $resultado['imported'], $resultado['errors']);

            return $resultado;

        } catch (Exception $e) {
            $this->finalizarLog($log ?? null, 'erro', $e->getMessage(), 0, 1);
            Log::error('Erro na importação SEBA', [
                'estacao_id' => $estacao->id,
                'erro' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Erro na importação: ' . $e->getMessage(),
                'imported' => 0
            ];
        }
    }

    /**
     * Importar dados de todas as estações configuradas
     */
    public function importarTodasEstacoes(Carbon $dataInicio = null, Carbon $dataFim = null): array
    {
        $estacoes = Estacao::whereNotNull('seba_station_id')
                          ->where('ativa', true)
                          ->get();

        if ($estacoes->isEmpty()) {
            return [
                'success' => false,
                'message' => 'Nenhuma estação com integração SEBA encontrada',
                'results' => []
            ];
        }

        $resultados = [];
        $totalImportado = 0;
        $totalErros = 0;

        Log::info('Iniciando importação em massa SEBA', [
            'total_estacoes' => $estacoes->count()
        ]);

        foreach ($estacoes as $estacao) {
            $resultado = $this->importarDadosEstacao($estacao, $dataInicio, $dataFim);
            $resultados[] = [
                'estacao' => $estacao->nome,
                'resultado' => $resultado
            ];

            if ($resultado['success']) {
                $totalImportado += $resultado['imported'];
            } else {
                $totalErros++;
            }

            // Pausa entre requisições para não sobrecarregar o servidor SEBA
            usleep(500000); // 0.5 segundos
        }

        return [
            'success' => $totalErros === 0,
            'message' => "Importação concluída: {$totalImportado} leituras importadas, {$totalErros} erros",
            'total_imported' => $totalImportado,
            'total_errors' => $totalErros,
            'results' => $resultados
        ];
    }

    /**
     * Buscar dados de uma estação no SEBA
     */
    protected function buscarDadosEstacao(string $sebaStationId, Carbon $dataInicio, Carbon $dataFim): array
    {
        // Tentar diferentes métodos de integração
        $metodos = [
            'rest_api' => function() use ($sebaStationId, $dataInicio, $dataFim) {
                return $this->buscarViaRestApi($sebaStationId, $dataInicio, $dataFim);
            },
            'sensorthings_api' => function() use ($sebaStationId, $dataInicio, $dataFim) {
                return $this->buscarViaSensorThingsApi($sebaStationId, $dataInicio, $dataFim);
            },
            'csv_download' => function() use ($sebaStationId, $dataInicio, $dataFim) {
                return $this->buscarViaCsvDownload($sebaStationId, $dataInicio, $dataFim);
            }
        ];

        foreach ($metodos as $metodo => $funcao) {
            try {
                Log::info("Tentando método: {$metodo}", ['station_id' => $sebaStationId]);
                $dados = $funcao();
                if (!empty($dados)) {
                    Log::info("Sucesso com método: {$metodo}", [
                        'station_id' => $sebaStationId,
                        'registros' => count($dados)
                    ]);
                    return $dados;
                }
            } catch (Exception $e) {
                Log::warning("Falhou método: {$metodo}", [
                    'station_id' => $sebaStationId,
                    'erro' => $e->getMessage()
                ]);
                continue;
            }
        }

        return [];
    }

    /**
     * Buscar dados via REST API do SEBA Hydrocenter
     */
    protected function buscarViaRestApi(string $sebaStationId, Carbon $dataInicio, Carbon $dataFim): array
    {
        if (!$this->apiUrl) {
            throw new Exception('URL da API SEBA não configurada');
        }

        $response = Http::timeout($this->timeout)
            ->withBasicAuth($this->username, $this->password)
            ->withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json'
            ])
            ->get("{$this->apiUrl}/stations/{$sebaStationId}/data", [
                'start_time' => $dataInicio->toISOString(),
                'end_time' => $dataFim->toISOString(),
                'format' => 'json'
            ]);

        if (!$response->successful()) {
            throw new Exception("Erro na API SEBA: {$response->status()} - {$response->body()}");
        }

        return $this->processarRespostaJson($response->json());
    }

    /**
     * Buscar dados via SensorThings API
     */
    protected function buscarViaSensorThingsApi(string $sebaStationId, Carbon $dataInicio, Carbon $dataFim): array
    {
        $sensorThingsUrl = config('services.seba.sensorthings_url');
        if (!$sensorThingsUrl) {
            throw new Exception('URL SensorThings não configurada');
        }

        // Construir query OData para SensorThings API
        $filter = "phenomenonTime ge {$dataInicio->toISOString()} and phenomenonTime le {$dataFim->toISOString()}";

        $response = Http::timeout($this->timeout)
            ->get("{$sensorThingsUrl}/Datastreams({$sebaStationId})/Observations", [
                '$filter' => $filter,
                '$orderby' => 'phenomenonTime asc',
                '$top' => 1000
            ]);

        if (!$response->successful()) {
            throw new Exception("Erro na SensorThings API: {$response->status()}");
        }

        return $this->processarRespostaSensorThings($response->json());
    }

    /**
     * Buscar dados via download de CSV
     */
    protected function buscarViaCsvDownload(string $sebaStationId, Carbon $dataInicio, Carbon $dataFim): array
    {
        $csvUrl = config('services.seba.csv_download_url');
        if (!$csvUrl) {
            throw new Exception('URL de download CSV não configurada');
        }

        $response = Http::timeout($this->timeout)
            ->withBasicAuth($this->username, $this->password)
            ->get($csvUrl, [
                'station_id' => $sebaStationId,
                'start_date' => $dataInicio->format('Y-m-d'),
                'end_date' => $dataFim->format('Y-m-d'),
                'format' => 'csv'
            ]);

        if (!$response->successful()) {
            throw new Exception("Erro no download CSV: {$response->status()}");
        }

        return $this->processarCsv($response->body());
    }

    /**
     * Processar resposta JSON da API REST
     */
    protected function processarRespostaJson(array $data): array
    {
        $leituras = [];

        if (isset($data['measurements']) && is_array($data['measurements'])) {
            foreach ($data['measurements'] as $measurement) {
                $leituras[] = [
                    'data_leitura' => Carbon::parse($measurement['timestamp'])->format('Y-m-d'),
                    'hora_leitura' => Carbon::parse($measurement['timestamp'])->format('H:i:s'),
                    'nivel_hidrometrico' => $measurement['water_level'] ?? null,
                    'precipitacao' => $measurement['precipitation'] ?? null,
                    'temperatura' => $measurement['temperature'] ?? null,
                    'origem' => 'seba_rest_api'
                ];
            }
        }

        return $leituras;
    }

    /**
     * Processar resposta da SensorThings API
     */
    protected function processarRespostaSensorThings(array $data): array
    {
        $leituras = [];

        if (isset($data['value']) && is_array($data['value'])) {
            foreach ($data['value'] as $observation) {
                $timestamp = Carbon::parse($observation['phenomenonTime']);

                $leituras[] = [
                    'data_leitura' => $timestamp->format('Y-m-d'),
                    'hora_leitura' => $timestamp->format('H:i:s'),
                    'nivel_hidrometrico' => $observation['result'] ?? null,
                    'origem' => 'seba_sensorthings'
                ];
            }
        }

        return $leituras;
    }

    /**
     * Processar dados CSV
     */
    protected function processarCsv(string $csvContent): array
    {
        $leituras = [];
        $linhas = str_getcsv($csvContent, "\n");

        // Assumir que primeira linha é cabeçalho
        if (count($linhas) < 2) {
            return [];
        }

        $cabecalho = str_getcsv($linhas[0]);

        for ($i = 1; $i < count($linhas); $i++) {
            $valores = str_getcsv($linhas[$i]);

            if (count($valores) !== count($cabecalho)) {
                continue;
            }

            $linha = array_combine($cabecalho, $valores);

            // Mapear campos baseado nos nomes de colunas típicos do SEBA
            $leituras[] = [
                'data_leitura' => $this->extrairData($linha),
                'hora_leitura' => $this->extrairHora($linha),
                'nivel_hidrometrico' => $this->extrairValor($linha, ['water_level', 'level', 'niveau', 'nivel']),
                'precipitacao' => $this->extrairValor($linha, ['precipitation', 'rain', 'pluie', 'precipitacao']),
                'temperatura' => $this->extrairValor($linha, ['temperature', 'temp', 'temperatura']),
                'origem' => 'seba_csv'
            ];
        }

        return $leituras;
    }

    /**
     * Processar e salvar dados no banco
     */
    protected function processarDados(Estacao $estacao, array $dados): array
    {
        $importados = 0;
        $erros = 0;
        $duplicados = 0;

        foreach ($dados as $dado) {
            try {
                // Verificar se já existe
                $existe = LeituraEstacao::where('estacao_id', $estacao->id)
                    ->where('data_leitura', $dado['data_leitura'])
                    ->where('hora_leitura', $dado['hora_leitura'])
                    ->exists();

                if ($existe) {
                    $duplicados++;
                    continue;
                }

                // Criar nova leitura
                LeituraEstacao::create([
                    'estacao_id' => $estacao->id,
                    'data_leitura' => $dado['data_leitura'],
                    'hora_leitura' => $dado['hora_leitura'],
                    'nivel_hidrometrico' => $dado['nivel_hidrometrico'],
                    'precipitacao' => $dado['precipitacao'],
                    'temperatura' => $dado['temperatura'],
                    'origem' => $dado['origem'] ?? 'seba_import',
                    'importado_automaticamente' => true,
                    'validado' => false // Requer validação manual
                ]);

                $importados++;

                // Verificar alertas se aplicável
                if ($estacao->tipo === 'hidrometrica' && $dado['nivel_hidrometrico']) {
                    \App\Models\Alerta::criarAlertaEstacao($estacao,
                        LeituraEstacao::where('estacao_id', $estacao->id)
                                      ->latest()
                                      ->first()
                    );
                }

            } catch (Exception $e) {
                $erros++;
                Log::error('Erro ao processar leitura SEBA', [
                    'estacao_id' => $estacao->id,
                    'dado' => $dado,
                    'erro' => $e->getMessage()
                ]);
            }
        }

        $message = "Importação concluída: {$importados} leituras importadas";
        if ($duplicados > 0) {
            $message .= ", {$duplicados} duplicados ignorados";
        }
        if ($erros > 0) {
            $message .= ", {$erros} erros";
        }

        return [
            'success' => true,
            'message' => $message,
            'imported' => $importados,
            'duplicates' => $duplicados,
            'errors' => $erros
        ];
    }

    /**
     * Testar conectividade com SEBA
     */
    public function testarConectividade(): array
    {
        $testes = [];

        // Teste REST API
        if ($this->apiUrl && $this->username && $this->password) {
            try {
                $response = Http::timeout(10)
                    ->withBasicAuth($this->username, $this->password)
                    ->get($this->apiUrl . '/status');

                $testes['rest_api'] = [
                    'status' => $response->successful() ? 'sucesso' : 'erro',
                    'message' => $response->successful() ? 'API REST acessível' : 'Erro HTTP ' . $response->status(),
                    'response_time' => $response->transferStats?->getTransferTime() ?? 0
                ];
            } catch (Exception $e) {
                $testes['rest_api'] = [
                    'status' => 'erro',
                    'message' => $e->getMessage(),
                    'response_time' => 0
                ];
            }
        }

        // Teste SensorThings API
        if (config('services.seba.sensorthings_url')) {
            try {
                $response = Http::timeout(10)
                    ->get(config('services.seba.sensorthings_url'));

                $testes['sensorthings'] = [
                    'status' => $response->successful() ? 'sucesso' : 'erro',
                    'message' => $response->successful() ? 'SensorThings API acessível' : 'Erro HTTP ' . $response->status()
                ];
            } catch (Exception $e) {
                $testes['sensorthings'] = [
                    'status' => 'erro',
                    'message' => $e->getMessage()
                ];
            }
        }

        return $testes;
    }

    // Métodos auxiliares
    protected function extrairData(array $linha): string
    {
        $campos = ['date', 'timestamp', 'datetime', 'data', 'fecha'];
        foreach ($campos as $campo) {
            if (isset($linha[$campo])) {
                return Carbon::parse($linha[$campo])->format('Y-m-d');
            }
        }
        return now()->format('Y-m-d');
    }

    protected function extrairHora(array $linha): string
    {
        $campos = ['time', 'timestamp', 'datetime', 'hora', 'heure'];
        foreach ($campos as $campo) {
            if (isset($linha[$campo])) {
                return Carbon::parse($linha[$campo])->format('H:i:s');
            }
        }
        return now()->format('H:i:s');
    }

    protected function extrairValor(array $linha, array $possiveisCampos): ?float
    {
        foreach ($possiveisCampos as $campo) {
            if (isset($linha[$campo]) && is_numeric($linha[$campo])) {
                return (float) $linha[$campo];
            }
        }
        return null;
    }

    protected function iniciarLog(Estacao $estacao, string $tipo): SebaIntegracaoLog
    {
        return SebaIntegracaoLog::create([
            'estacao_id' => $estacao->id,
            'tipo' => $tipo,
            'status' => 'executando',
            'iniciado_em' => now()
        ]);
    }

    protected function finalizarLog(?SebaIntegracaoLog $log, string $status, string $mensagem, int $importados, int $erros): void
    {
        if ($log) {
            $log->update([
                'status' => $status,
                'mensagem' => $mensagem,
                'registros_importados' => $importados,
                'registros_erro' => $erros,
                'finalizado_em' => now()
            ]);
        }
    }
}