<?php

namespace App\Console\Commands;

use App\Models\Estacao;
use App\Services\SebaHydrometrieService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SebaImportarTodasEstacoes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'seba:importar-todas
                           {--inicio= : Data de início (Y-m-d H:i)}
                           {--fim= : Data de fim (Y-m-d H:i)}
                           {--force : Forçar importação mesmo se desabilitada}
                           {--paralelo=1 : Número de estações em paralelo (máx 5)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Importar dados de todas as estações configuradas no SEBA-HYDROMETRIE';

    protected $sebaService;

    public function __construct(SebaHydrometrieService $sebaService)
    {
        parent::__construct();
        $this->sebaService = $sebaService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Iniciando importação em massa do SEBA-HYDROMETRIE');

        // Buscar estações elegíveis
        $query = Estacao::whereNotNull('seba_station_id')->where('ativa', true);

        if (!$this->option('force')) {
            $query->where('auto_importacao_ativa', true);
        }

        $estacoes = $query->get();

        if ($estacoes->isEmpty()) {
            $this->warn('⚠️ Nenhuma estação configurada para integração SEBA encontrada.');
            $this->info('Configure as estações com seba_station_id e auto_importacao_ativa=true');
            return Command::SUCCESS;
        }

        $this->info("📊 Encontradas {$estacoes->count()} estações para importação:");

        // Mostrar lista de estações
        $this->table(
            ['ID', 'Nome', 'SEBA ID', 'Última Sync', 'Intervalo (min)'],
            $estacoes->map(function($estacao) {
                return [
                    $estacao->id,
                    $estacao->nome,
                    $estacao->seba_station_id,
                    $estacao->ultima_sincronizacao_seba?->format('d/m/Y H:i') ?? 'Nunca',
                    $estacao->intervalo_importacao_minutos
                ];
            })->toArray()
        );

        if (!$this->confirm('Continuar com a importação?', true)) {
            $this->info('❌ Importação cancelada pelo usuário.');
            return Command::SUCCESS;
        }

        // Determinar período
        $dataInicio = $this->option('inicio') ? Carbon::createFromFormat('Y-m-d H:i', $this->option('inicio')) : null;
        $dataFim = $this->option('fim') ? Carbon::createFromFormat('Y-m-d H:i', $this->option('fim')) : null;

        $periodo = ($dataInicio && $dataFim)
            ? "Período: {$dataInicio->format('d/m/Y H:i')} - {$dataFim->format('d/m/Y H:i')}"
            : "Período: Últimas 24 horas";

        $this->info($periodo);

        // Barra de progresso
        $progressBar = $this->output->createProgressBar($estacoes->count());
        $progressBar->setFormat('verbose');

        $resultados = [];
        $totalImportado = 0;
        $totalErros = 0;

        foreach ($estacoes as $estacao) {
            $progressBar->setMessage("Processando: {$estacao->nome}");
            $progressBar->advance();

            try {
                $resultado = $this->sebaService->importarDadosEstacao($estacao, $dataInicio, $dataFim);

                $resultados[] = [
                    'estacao' => $estacao->nome,
                    'seba_id' => $estacao->seba_station_id,
                    'status' => $resultado['success'] ? '✅ Sucesso' : '❌ Erro',
                    'importados' => $resultado['imported'] ?? 0,
                    'duplicados' => $resultado['duplicates'] ?? 0,
                    'erros' => $resultado['errors'] ?? 0,
                    'mensagem' => substr($resultado['message'], 0, 50) . (strlen($resultado['message']) > 50 ? '...' : '')
                ];

                if ($resultado['success']) {
                    $totalImportado += $resultado['imported'];

                    // Atualizar última sincronização
                    if ($resultado['imported'] > 0) {
                        $estacao->update(['ultima_sincronizacao_seba' => now()]);
                    }
                } else {
                    $totalErros++;
                }

                // Pausa entre requisições para não sobrecarregar
                usleep(500000); // 0.5 segundos

            } catch (\Exception $e) {
                $totalErros++;
                $resultados[] = [
                    'estacao' => $estacao->nome,
                    'seba_id' => $estacao->seba_station_id,
                    'status' => '❌ Erro',
                    'importados' => 0,
                    'duplicados' => 0,
                    'erros' => 1,
                    'mensagem' => substr($e->getMessage(), 0, 50) . '...'
                ];

                Log::error('Erro na importação em massa SEBA', [
                    'estacao_id' => $estacao->id,
                    'erro' => $e->getMessage()
                ]);
            }
        }

        $progressBar->finish();
        $this->newLine(2);

        // Mostrar resultados
        $this->info('📋 Resultados da Importação:');

        $this->table(
            ['Estação', 'SEBA ID', 'Status', 'Importados', 'Duplicados', 'Erros', 'Mensagem'],
            $resultados
        );

        // Resumo final
        $this->info('📊 Resumo Final:');
        $this->table(
            ['Métrica', 'Valor'],
            [
                ['Estações processadas', $estacoes->count()],
                ['Total de registros importados', $totalImportado],
                ['Estações com erro', $totalErros],
                ['Taxa de sucesso', round((($estacoes->count() - $totalErros) / $estacoes->count()) * 100, 1) . '%'],
                ['Duração', $this->getExecutionTime()]
            ]
        );

        if ($totalErros === 0) {
            $this->info('🎉 Importação concluída com sucesso!');
            return Command::SUCCESS;
        } else {
            $this->warn("⚠️ Importação concluída com {$totalErros} erro(s). Verifique os logs para detalhes.");
            return Command::SUCCESS; // Não falha completamente se algumas importações funcionaram
        }
    }

    private function getExecutionTime(): string
    {
        $endTime = microtime(true);
        $executionTime = $endTime - LARAVEL_START;

        if ($executionTime < 60) {
            return round($executionTime, 2) . 's';
        } else {
            return round($executionTime / 60, 2) . 'min';
        }
    }
}
