<?php

namespace App\Console\Commands;

use App\Services\SebaHydrometrieService;
use Illuminate\Console\Command;

class SebaTestarConectividade extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'seba:testar-conectividade';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Testar conectividade com os serviços do SEBA-HYDROMETRIE';

    protected $sebaService;

    public function __construct(SebaHydrometrieService $sebaService)
    {
        parent::__construct();
        $this->sebaService = $sebaService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Testando conectividade com SEBA-HYDROMETRIE...');
        $this->newLine();

        // Verificar configurações
        $this->info('📋 Verificando configurações:');
        $configs = [
            'API URL' => config('services.seba.api_url', 'Não configurado'),
            'Username' => config('services.seba.username') ? '✅ Configurado' : '❌ Não configurado',
            'Password' => config('services.seba.password') ? '✅ Configurado' : '❌ Não configurado',
            'SensorThings URL' => config('services.seba.sensorthings_url', 'Não configurado'),
            'CSV Download URL' => config('services.seba.csv_download_url', 'Não configurado'),
            'Timeout' => config('services.seba.timeout', 30) . ' segundos',
        ];

        $this->table(['Configuração', 'Status'], array_map(function($key, $value) {
            return [$key, $value];
        }, array_keys($configs), $configs));

        $this->newLine();

        // Testar conectividade
        $this->info('🌐 Testando conectividade...');

        $testes = $this->sebaService->testarConectividade();

        if (empty($testes)) {
            $this->error('❌ Nenhum método de conectividade configurado!');
            $this->info('💡 Configure pelo menos um dos seguintes:');
            $this->info('   - SEBA_API_URL + SEBA_USERNAME + SEBA_PASSWORD');
            $this->info('   - SEBA_SENSORTHINGS_URL');
            $this->info('   - SEBA_CSV_DOWNLOAD_URL');
            return Command::FAILURE;
        }

        $resultados = [];
        $sucessos = 0;

        foreach ($testes as $metodo => $resultado) {
            $status = $resultado['status'] === 'sucesso' ? '✅ Sucesso' : '❌ Erro';
            $tempo = isset($resultado['response_time']) ? round($resultado['response_time'] * 1000, 0) . 'ms' : '-';

            $resultados[] = [
                'Método' => ucfirst(str_replace('_', ' ', $metodo)),
                'Status' => $status,
                'Mensagem' => $resultado['message'],
                'Tempo' => $tempo
            ];

            if ($resultado['status'] === 'sucesso') {
                $sucessos++;
            }
        }

        $this->table(['Método', 'Status', 'Mensagem', 'Tempo Resposta'], $resultados);

        $this->newLine();

        // Resumo
        $total = count($testes);
        $taxaSucesso = round(($sucessos / $total) * 100, 1);

        if ($sucessos === $total) {
            $this->info("🎉 Todos os testes passaram! ({$sucessos}/{$total} - {$taxaSucesso}%)");
            $this->info('✅ Sistema pronto para importação automática');

            // Mostrar próximos passos
            $this->newLine();
            $this->info('📚 Próximos passos:');
            $this->info('1. Configure as estações com IDs SEBA: php artisan seba:configurar-estacoes');
            $this->info('2. Teste importação: php artisan seba:importar-estacao {id}');
            $this->info('3. Configure cron para automação: */15 * * * * php artisan seba:importar-todas');

            return Command::SUCCESS;

        } else if ($sucessos > 0) {
            $this->warn("⚠️ Conectividade parcial: {$sucessos}/{$total} testes passaram ({$taxaSucesso}%)");
            $this->info('💡 Pelo menos um método funciona - sistema operacional');

            return Command::SUCCESS;

        } else {
            $this->error("❌ Todos os testes falharam! (0/{$total})");
            $this->info('🔧 Verifique as configurações e conectividade de rede');

            // Dicas de troubleshooting
            $this->newLine();
            $this->info('🔍 Troubleshooting:');
            $this->info('• Verifique se as URLs estão corretas');
            $this->info('• Confirme username/password do SEBA Hydrocenter');
            $this->info('• Teste conectividade de rede: ping seba-server.com');
            $this->info('• Verifique firewall e proxy corporativo');

            return Command::FAILURE;
        }
    }
}
