<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Barragem;
use App\Models\LeituraBarragem;
use Illuminate\Http\Request;
use Carbon\Carbon;

class GraficosBarragemController extends Controller
{
    public function getDados(Request $request)
    {
        $barragem_id = $request->get('barragem_id');
        $periodo = $request->get('periodo', '30d');
        $tipo = $request->get('tipo', 'todos');
        $comparacao = $request->get('comparacao', 'atual');

        // Determinar intervalo de datas
        $startDate = $this->getStartDate($periodo);

        if ($barragem_id && $barragem_id !== 'todas') {
            // Dados de uma barragem específica
            $barragem = Barragem::findOrFail($barragem_id);
            return $this->getDadosBarragem($barragem, $startDate, $periodo, $tipo, $comparacao);
        } else {
            // Dados de todas as barragens
            return $this->getDadosTodasBarragens($startDate, $periodo, $tipo, $comparacao);
        }
    }

    private function getStartDate($periodo)
    {
        return match($periodo) {
            '7d' => Carbon::now()->subDays(7),
            '30d' => Carbon::now()->subDays(30),
            '90d' => Carbon::now()->subDays(90),
            '1y' => Carbon::now()->subYear(),
            default => Carbon::now()->subDays(30)
        };
    }

    private function getDadosBarragem(Barragem $barragem, $startDate, $periodo, $tipo, $comparacao)
    {
        $leituras = $barragem->leituras()
            ->where('data_leitura', '>=', $startDate)
            ->orderBy('data_leitura')
            ->get();

        if ($leituras->isEmpty()) {
            return response()->json([
                'labels' => [],
                'datasets' => [],
                'barragem' => $barragem->nome,
                'message' => 'Nenhuma leitura encontrada para o período selecionado'
            ]);
        }

        // Agrupar dados por período
        $groupedData = $this->groupLeituras($leituras, $periodo);

        $labels = [];
        $datasets = [];

        // Preparar labels
        foreach ($groupedData as $date => $dayLeituras) {
            if ($periodo === '1y') {
                $labels[] = Carbon::parse($date)->format('M/Y');
            } else {
                $labels[] = Carbon::parse($date)->format('d/m');
            }
        }

        // Preparar datasets baseado no tipo solicitado
        if ($tipo === 'todos' || $tipo === 'precipitacao') {
            $precipitacaoData = [];
            foreach ($groupedData as $date => $dayLeituras) {
                $leitura = $dayLeituras->last();
                $precipitacaoData[] = $leitura->precipitacao ?? 0;
            }

            $datasets[] = [
                'label' => 'Precipitação (mm)',
                'type' => 'bar',
                'data' => $precipitacaoData,
                'backgroundColor' => 'rgba(59, 130, 246, 0.6)',
                'borderColor' => 'rgba(59, 130, 246, 1)',
                'borderWidth' => 1,
                'yAxisID' => 'y1'
            ];
        }

        if ($tipo === 'todos' || $tipo === 'cotas') {
            $cotasData = [];
            foreach ($groupedData as $date => $dayLeituras) {
                $leitura = $dayLeituras->last();
                $cotasData[] = $leitura->cota_actual ?? 0;
            }

            $datasets[] = [
                'label' => 'Cota (m)',
                'type' => 'line',
                'data' => $cotasData,
                'borderColor' => 'rgba(34, 197, 94, 1)',
                'backgroundColor' => 'rgba(34, 197, 94, 0.1)',
                'borderWidth' => 3,
                'tension' => 0.4,
                'yAxisID' => 'y'
            ];

            // Adicionar linhas de referência se existirem
            if ($barragem->cota_npa) {
                $datasets[] = [
                    'label' => 'NPA',
                    'type' => 'line',
                    'data' => array_fill(0, count($labels), $barragem->cota_npa),
                    'borderColor' => 'rgba(239, 68, 68, 1)',
                    'backgroundColor' => 'rgba(239, 68, 68, 0.1)',
                    'borderDash' => [5, 5],
                    'borderWidth' => 2,
                    'pointRadius' => 0,
                    'yAxisID' => 'y'
                ];
            }

            if ($barragem->cota_nme) {
                $datasets[] = [
                    'label' => 'NME',
                    'type' => 'line',
                    'data' => array_fill(0, count($labels), $barragem->cota_nme),
                    'borderColor' => 'rgba(251, 191, 36, 1)',
                    'backgroundColor' => 'rgba(251, 191, 36, 0.1)',
                    'borderDash' => [5, 5],
                    'borderWidth' => 2,
                    'pointRadius' => 0,
                    'yAxisID' => 'y'
                ];
            }
        }

        if ($tipo === 'todos' || $tipo === 'volumes') {
            $volumesData = [];
            $enchimentoData = [];

            foreach ($groupedData as $date => $dayLeituras) {
                $leitura = $dayLeituras->last();
                $volumesData[] = $leitura->volume_actual ?? 0;
                $enchimentoData[] = $leitura->percentagem_enchimento ?? 0;
            }

            $datasets[] = [
                'label' => 'Volume (Mm³)',
                'type' => 'line',
                'data' => $volumesData,
                'borderColor' => 'rgba(147, 51, 234, 1)',
                'backgroundColor' => 'rgba(147, 51, 234, 0.1)',
                'borderWidth' => 3,
                'tension' => 0.4,
                'yAxisID' => 'y2'
            ];

            $datasets[] = [
                'label' => 'Enchimento (%)',
                'type' => 'line',
                'data' => $enchimentoData,
                'borderColor' => 'rgba(245, 158, 11, 1)',
                'backgroundColor' => 'rgba(245, 158, 11, 0.1)',
                'borderWidth' => 3,
                'tension' => 0.4,
                'yAxisID' => 'y3'
            ];
        }

        // Adicionar comparação se solicitada
        if ($comparacao === 'ano-anterior') {
            $this->addComparacaoAnoAnterior($barragem, $datasets, $labels, $startDate, $tipo);
        }

        return response()->json([
            'labels' => $labels,
            'datasets' => $datasets,
            'barragem' => $barragem->nome,
            'tipo' => $tipo,
            'periodo' => $periodo,
            'comparacao' => $comparacao
        ]);
    }

    private function getDadosTodasBarragens($startDate, $periodo, $tipo, $comparacao)
    {
        $barragens = Barragem::with(['leituras' => function($query) use ($startDate) {
            $query->where('data_leitura', '>=', $startDate)
                  ->orderBy('data_leitura');
        }])->get();

        $result = [];

        foreach ($barragens as $barragem) {
            if ($barragem->leituras->count() > 0) {
                $ultimaLeitura = $barragem->leituras->last();

                $dadosBarragem = [
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'total_leituras' => $barragem->leituras->count()
                ];

                if ($tipo === 'todos' || $tipo === 'precipitacao') {
                    $dadosBarragem['precipitacao_total'] = $barragem->leituras->sum('precipitacao');
                    $dadosBarragem['precipitacao_media'] = $barragem->leituras->avg('precipitacao');
                }

                if ($tipo === 'todos' || $tipo === 'cotas') {
                    $dadosBarragem['cota_atual'] = $ultimaLeitura->cota_actual;
                    $dadosBarragem['cota_maxima'] = $barragem->leituras->max('cota_actual');
                    $dadosBarragem['cota_minima'] = $barragem->leituras->min('cota_actual');
                }

                if ($tipo === 'todos' || $tipo === 'volumes') {
                    $dadosBarragem['volume_atual'] = $ultimaLeitura->volume_actual;
                    $dadosBarragem['enchimento_atual'] = $ultimaLeitura->percentagem_enchimento;
                    $dadosBarragem['volume_maximo_capacidade'] = $barragem->volume_maximo;
                }

                $result[] = $dadosBarragem;
            }
        }

        return response()->json([
            'barragens' => $result,
            'resumo' => [
                'total_barragens' => count($result),
                'media_enchimento' => collect($result)->avg('enchimento_atual'),
                'total_precipitacao' => collect($result)->sum('precipitacao_total')
            ],
            'tipo' => $tipo,
            'periodo' => $periodo
        ]);
    }

    private function groupLeituras($leituras, $periodo)
    {
        if ($periodo === '1y') {
            return $leituras->groupBy(function ($leitura) {
                return $leitura->data_leitura->format('Y-m');
            });
        } else {
            return $leituras->groupBy(function ($leitura) {
                return $leitura->data_leitura->format('Y-m-d');
            });
        }
    }

    private function addComparacaoAnoAnterior($barragem, &$datasets, $labels, $startDate, $tipo)
    {
        $anoAnteriorStart = $startDate->copy()->subYear();
        $anoAnteriorEnd = Carbon::now()->subYear();

        $leiturasAnoAnterior = $barragem->leituras()
            ->where('data_leitura', '>=', $anoAnteriorStart)
            ->where('data_leitura', '<=', $anoAnteriorEnd)
            ->orderBy('data_leitura')
            ->get();

        if ($leiturasAnoAnterior->count() > 0) {
            $groupedAnoAnterior = $this->groupLeituras($leiturasAnoAnterior, '30d');

            if ($tipo === 'todos' || $tipo === 'cotas') {
                $cotasAnoAnterior = [];
                foreach ($labels as $index => $label) {
                    // Tentar encontrar dados correspondentes do ano anterior
                    $cotasAnoAnterior[] = $leiturasAnoAnterior->avg('cota_actual') ?? 0;
                }

                $datasets[] = [
                    'label' => 'Cota Ano Anterior (m)',
                    'type' => 'line',
                    'data' => $cotasAnoAnterior,
                    'borderColor' => 'rgba(156, 163, 175, 1)',
                    'backgroundColor' => 'rgba(156, 163, 175, 0.1)',
                    'borderWidth' => 2,
                    'borderDash' => [10, 5],
                    'tension' => 0.4,
                    'yAxisID' => 'y'
                ];
            }
        }
    }
}