<?php

namespace App\Http\Controllers;

use App\Models\Estacao;
use App\Models\FormulaCaudal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class FormulaCaudalController extends Controller
{
    public function index(Estacao $estacao)
    {
        $formulas = $estacao->formulas_caudal()
            ->with('criador')
            ->orderBy('data_inicio', 'desc')
            ->get();

        $formulaAtiva = $estacao->getFormulaAtiva();

        return view('estacoes.formulas-caudal', compact('estacao', 'formulas', 'formulaAtiva'));
    }

    public function store(Request $request, Estacao $estacao)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'motivo_mudanca' => 'nullable|string',
            'coeficiente_a' => 'required|numeric',
            'coeficiente_b' => 'required|numeric',
            'coeficiente_c' => 'nullable|numeric',
            'data_inicio' => 'required|date',
            'data_fim' => 'nullable|date|after:data_inicio',
            'nivel_min' => 'nullable|numeric|min:0',
            'nivel_max' => 'nullable|numeric|gt:nivel_min',
            'r_quadrado' => 'nullable|numeric|min:0|max:1',
        ]);

        // Desativar fórmulas anteriores se a nova começar hoje
        if ($validated['data_inicio'] <= today()) {
            $estacao->formulas_caudal()
                ->where('ativa', true)
                ->update(['ativa' => false]);
        }

        $validated['estacao_id'] = $estacao->id;
        $validated['criado_por'] = Auth::id();
        $validated['coeficiente_c'] = $validated['coeficiente_c'] ?? 0;
        $validated['ativa'] = true;

        FormulaCaudal::create($validated);

        return redirect()->route('estacoes.formulas-caudal.index', $estacao)
            ->with('success', 'Fórmula criada com sucesso!');
    }

    public function toggle(Estacao $estacao, FormulaCaudal $formula)
    {
        $formula->update(['ativa' => !$formula->ativa]);

        $status = $formula->ativa ? 'ativada' : 'desativada';
        return redirect()->route('estacoes.formulas-caudal.index', $estacao)
            ->with('success', "Fórmula {$status} com sucesso!");
    }

    public function destroy(Estacao $estacao, FormulaCaudal $formula)
    {
        $formula->delete();

        return redirect()->route('estacoes.formulas-caudal.index', $estacao)
            ->with('success', 'Fórmula removida com sucesso!');
    }
}