<?php

namespace App\Jobs;

use App\Models\Estacao;
use App\Services\SebaHydrometrieService;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SebaImportacaoAutomaticaJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $estacao;
    protected $dataInicio;
    protected $dataFim;

    /**
     * The number of times the job may be attempted.
     */
    public $tries = 3;

    /**
     * The maximum number of seconds the job can run.
     */
    public $timeout = 300; // 5 minutos

    /**
     * Create a new job instance.
     */
    public function __construct(Estacao $estacao, Carbon $dataInicio = null, Carbon $dataFim = null)
    {
        $this->estacao = $estacao;
        $this->dataInicio = $dataInicio ?? now()->subHour(); // Última hora por padrão
        $this->dataFim = $dataFim ?? now();
    }

    /**
     * Execute the job.
     */
    public function handle(SebaHydrometrieService $sebaService): void
    {
        Log::info('Iniciando importação automática SEBA via job', [
            'estacao_id' => $this->estacao->id,
            'estacao_nome' => $this->estacao->nome,
            'seba_station_id' => $this->estacao->seba_station_id,
            'periodo_inicio' => $this->dataInicio->toISOString(),
            'periodo_fim' => $this->dataFim->toISOString(),
            'tentativa' => $this->attempts()
        ]);

        try {
            // Verificar se a estação ainda está ativa e configurada
            if (!$this->estacao->ativa || !$this->estacao->seba_station_id || !$this->estacao->auto_importacao_ativa) {
                Log::info('Estação não elegível para importação automática', [
                    'estacao_id' => $this->estacao->id,
                    'ativa' => $this->estacao->ativa,
                    'seba_station_id' => $this->estacao->seba_station_id,
                    'auto_importacao_ativa' => $this->estacao->auto_importacao_ativa
                ]);
                return;
            }

            // Verificar se não foi importado recentemente (evitar duplicatas)
            $ultimaImportacao = $this->estacao->ultima_sincronizacao_seba;
            $intervaloMinimo = now()->subMinutes($this->estacao->intervalo_importacao_minutos * 0.8); // 80% do intervalo

            if ($ultimaImportacao && $ultimaImportacao->gt($intervaloMinimo)) {
                Log::info('Importação SEBA pulada - intervalo ainda não atingido', [
                    'estacao_id' => $this->estacao->id,
                    'ultima_importacao' => $ultimaImportacao->toISOString(),
                    'proximo_permitido' => $intervaloMinimo->toISOString()
                ]);
                return;
            }

            // Executar importação
            $resultado = $sebaService->importarDadosEstacao(
                $this->estacao,
                $this->dataInicio,
                $this->dataFim
            );

            if ($resultado['success']) {
                Log::info('Importação automática SEBA concluída com sucesso', [
                    'estacao_id' => $this->estacao->id,
                    'registros_importados' => $resultado['imported'],
                    'duplicados' => $resultado['duplicates'] ?? 0,
                    'erros' => $resultado['errors'] ?? 0
                ]);

                // Atualizar timestamp de última sincronização
                if ($resultado['imported'] > 0) {
                    $this->estacao->update(['ultima_sincronizacao_seba' => now()]);
                }

                // Agendar próxima importação
                $this->agendarProximaImportacao();

            } else {
                Log::error('Falha na importação automática SEBA', [
                    'estacao_id' => $this->estacao->id,
                    'erro' => $resultado['message']
                ]);

                // Re-lançar exceção para tentar novamente
                throw new \Exception($resultado['message']);
            }

        } catch (\Exception $e) {
            Log::error('Exceção durante importação automática SEBA', [
                'estacao_id' => $this->estacao->id,
                'erro' => $e->getMessage(),
                'tentativa' => $this->attempts(),
                'max_tentativas' => $this->tries
            ]);

            // Se esgotaram as tentativas, agendar nova importação para mais tarde
            if ($this->attempts() >= $this->tries) {
                $this->agendarProximaImportacao(true);
            }

            throw $e; // Re-lançar para o sistema de retry do Laravel
        }
    }

    /**
     * Agendar próxima importação automática
     */
    protected function agendarProximaImportacao(bool $comAtraso = false): void
    {
        if (!config('services.seba.auto_import_enabled', true)) {
            return;
        }

        $intervalo = $this->estacao->intervalo_importacao_minutos;

        // Se houve erro, aguardar mais tempo antes da próxima tentativa
        if ($comAtraso) {
            $intervalo = min($intervalo * 2, 240); // Máximo 4 horas
        }

        $proximaExecucao = now()->addMinutes($intervalo);

        // Criar novo job para execução futura
        static::dispatch($this->estacao)
              ->delay($proximaExecucao);

        Log::info('Próxima importação SEBA agendada', [
            'estacao_id' => $this->estacao->id,
            'proxima_execucao' => $proximaExecucao->toISOString(),
            'intervalo_minutos' => $intervalo,
            'com_atraso' => $comAtraso
        ]);
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('Job de importação SEBA falhou definitivamente', [
            'estacao_id' => $this->estacao->id,
            'erro' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);

        // Agendar nova tentativa com atraso maior
        $this->agendarProximaImportacao(true);
    }

    /**
     * Get the tags that should be assigned to the job.
     */
    public function tags(): array
    {
        return [
            'seba-import',
            'estacao:' . $this->estacao->id,
            'auto-import'
        ];
    }

    /**
     * Calculate the number of seconds to wait before retrying the job.
     */
    public function backoff(): array
    {
        return [30, 120, 300]; // 30s, 2min, 5min
    }
}
