<?php
// app/Models/LeituraEstacao.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LeituraEstacao extends Model
{
    use HasFactory;

    protected $table = 'leituras_estacoes';

    protected $fillable = [
        'estacao_id',
        'data_leitura',
        'hora_6h',
        'hora_9h',
        'hora_12h',
        'hora_15h',
        'hora_18h',
        'nivel_6h',
        'nivel_9h',
        'nivel_12h',
        'nivel_15h',
        'nivel_18h',
        'nivel_hidrometrico',
        'cota_6h',
        'cota_9h',
        'cota_12h',
        'cota_15h',
        'cota_18h',
        'cota_hidrometrica',
        'caudal',
        'caudal_manual',
        'caudal_estimado',
        'campos_sem_dados',
        'precipitacao_mm',
        'temperatura_max',
        'temperatura_min',
        'humidade_relativa',
        'evaporacao',
        'operador_id',
        'observacoes',
        'validado',
        'origem',
        'importado_automaticamente',
        'hora_leitura',
        'precipitacao',
        'temperatura',
    ];

    protected $casts = [
        'data_leitura' => 'date',
        'hora_6h' => 'datetime',
        'hora_9h' => 'datetime',
        'hora_12h' => 'datetime',
        'hora_15h' => 'datetime',
        'hora_18h' => 'datetime',
        'hora_leitura' => 'datetime',
        'nivel_6h' => 'decimal:2',
        'nivel_9h' => 'decimal:2',
        'nivel_12h' => 'decimal:2',
        'nivel_15h' => 'decimal:2',
        'nivel_18h' => 'decimal:2',
        'nivel_hidrometrico' => 'decimal:2',
        'cota_6h' => 'decimal:3',
        'cota_9h' => 'decimal:3',
        'cota_12h' => 'decimal:3',
        'cota_15h' => 'decimal:3',
        'cota_18h' => 'decimal:3',
        'cota_hidrometrica' => 'decimal:3',
        'caudal' => 'decimal:3',
        'caudal_manual' => 'decimal:3',
        'caudal_estimado' => 'boolean',
        'campos_sem_dados' => 'array',
        'precipitacao_mm' => 'decimal:2',
        'precipitacao' => 'decimal:2',
        'temperatura_max' => 'decimal:2',
        'temperatura_min' => 'decimal:2',
        'temperatura' => 'decimal:2',
        'humidade_relativa' => 'decimal:2',
        'evaporacao' => 'decimal:2',
        'validado' => 'boolean',
        'importado_automaticamente' => 'boolean',
    ];

    public function estacao(): BelongsTo
    {
        return $this->belongsTo(Estacao::class);
    }

    public function operador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'operador_id');
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });

        static::updating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });
    }

    public function calcularValoresAutomaticos()
    {
        if ($this->estacao && $this->estacao->isHidrometrica()) {
            $this->calcularNivelHidrometrico();
            $this->calcularCotas();
            $this->calcularCaudal();
        }
    }

    private function calcularNivelHidrometrico()
    {
        $niveis = array_filter([
            $this->nivel_6h,
            $this->nivel_9h,
            $this->nivel_12h,
            $this->nivel_15h,
            $this->nivel_18h
        ]);

        if (!empty($niveis)) {
            $this->nivel_hidrometrico = array_sum($niveis) / count($niveis);
        }
    }

    private function calcularCotas()
    {
        if (!$this->estacao || !$this->estacao->trabalhaComCotas()) {
            return;
        }

        // Se a estação recebe cotas manualmente, não calculamos automaticamente
        if ($this->estacao->recebeCotasManualmente()) {
            return;
        }

        // Calcular cotas individuais a partir dos níveis
        if ($this->nivel_6h) {
            $this->cota_6h = $this->estacao->calcularCotaDeNivel($this->nivel_6h);
        }

        if ($this->nivel_9h) {
            $this->cota_9h = $this->estacao->calcularCotaDeNivel($this->nivel_9h);
        }

        if ($this->nivel_12h) {
            $this->cota_12h = $this->estacao->calcularCotaDeNivel($this->nivel_12h);
        }

        if ($this->nivel_15h) {
            $this->cota_15h = $this->estacao->calcularCotaDeNivel($this->nivel_15h);
        }

        if ($this->nivel_18h) {
            $this->cota_18h = $this->estacao->calcularCotaDeNivel($this->nivel_18h);
        }

        // Calcular cota hidrométrica média
        if ($this->nivel_hidrometrico) {
            $this->cota_hidrometrica = $this->estacao->calcularCotaDeNivel($this->nivel_hidrometrico);
        } else {
            // Se não temos nível hidrométrico, calcular a partir das cotas individuais
            $this->calcularCotaHidrometrica();
        }
    }

    private function calcularCotaHidrometrica()
    {
        $cotas = array_filter([
            $this->cota_6h,
            $this->cota_9h,
            $this->cota_12h,
            $this->cota_15h,
            $this->cota_18h
        ]);

        if (!empty($cotas)) {
            $this->cota_hidrometrica = array_sum($cotas) / count($cotas);
        }
    }

    private function calcularCaudal()
    {
        if ($this->nivel_hidrometrico && $this->estacao) {
            $this->caudal = $this->estacao->calcularCaudal($this->nivel_hidrometrico);
        }
    }

    public function getNiveisArray()
    {
        return [
            '6h' => $this->nivel_6h,
            '9h' => $this->nivel_9h,
            '12h' => $this->nivel_12h,
            '15h' => $this->nivel_15h,
            '18h' => $this->nivel_18h,
        ];
    }

    public function getCotasArray()
    {
        return [
            '6h' => $this->cota_6h,
            '9h' => $this->cota_9h,
            '12h' => $this->cota_12h,
            '15h' => $this->cota_15h,
            '18h' => $this->cota_18h,
        ];
    }

    public function temCotas()
    {
        return !is_null($this->cota_6h) || !is_null($this->cota_9h) || !is_null($this->cota_12h) || !is_null($this->cota_15h) || !is_null($this->cota_18h) || !is_null($this->cota_hidrometrica);
    }

    public function definirCotasManualmente(array $cotas)
    {
        if (isset($cotas['6h'])) {
            $this->cota_6h = $cotas['6h'];
        }

        if (isset($cotas['9h'])) {
            $this->cota_9h = $cotas['9h'];
        }

        if (isset($cotas['12h'])) {
            $this->cota_12h = $cotas['12h'];
        }

        if (isset($cotas['15h'])) {
            $this->cota_15h = $cotas['15h'];
        }

        if (isset($cotas['18h'])) {
            $this->cota_18h = $cotas['18h'];
        }

        if (isset($cotas['hidrometrica'])) {
            $this->cota_hidrometrica = $cotas['hidrometrica'];
        } else {
            $this->calcularCotaHidrometrica();
        }
    }

    public function getStatusValidacao()
    {
        if ($this->validado) {
            return 'validado';
        }
        
        if ($this->created_at->diffInHours(now()) > 24) {
            return 'pendente';
        }
        
        return 'recente';
    }

    public function temDadosCompletos()
    {
        if ($this->estacao->isHidrometrica()) {
            // Se trabalha com cotas, verificar se tem cotas OU níveis
            if ($this->estacao->trabalhaComCotas()) {
                $temNiveis = !is_null($this->nivel_6h) || !is_null($this->nivel_9h) || !is_null($this->nivel_12h) || !is_null($this->nivel_15h) || !is_null($this->nivel_18h);
                $temCotas = !is_null($this->cota_6h) || !is_null($this->cota_9h) || !is_null($this->cota_12h) || !is_null($this->cota_15h) || !is_null($this->cota_18h);
                return $temNiveis || $temCotas;
            }

            return !is_null($this->nivel_6h) || !is_null($this->nivel_9h) || !is_null($this->nivel_12h) || !is_null($this->nivel_15h) || !is_null($this->nivel_18h);
        }

        if ($this->estacao->isPluviometrica()) {
            return !is_null($this->precipitacao_mm) || !is_null($this->precipitacao);
        }

        if ($this->estacao->isMeteorological()) {
            return !is_null($this->temperatura_max) || !is_null($this->temperatura_min) || !is_null($this->temperatura);
        }

        return false;
    }
}