<?php

namespace App\Services;

use App\Models\Boletim;
use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use App\Models\BaciaHidrografica;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BoletimExport;
use App\Models\SystemSetting;

class BoletimService
{
    public function gerarBoletimHidrologicoRegional($data_inicio, $data_fim)
    {
        $dados = $this->coletarDadosHidrologicoRegional($data_inicio, $data_fim);
        
        $boletim = Boletim::create([
            'tipo' => 'hidrologico_regional',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => now()->toDateString(),
            'hora_emissao' => now()->toTimeString(),
            'periodo_inicio' => $data_inicio,
            'periodo_fim' => $data_fim,
            'situacao_prevalecente' => $this->gerarSituacaoPrevalecente($dados),
            'previsao_hidrologica' => $this->gerarPrevisaoHidrologica($dados),
            'alertas_precaucoes' => $this->gerarAlertasPrecaucoes($dados),
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarBoletimHidrologicoProvincial($data_emissao = null)
    {
        $data_emissao = $data_emissao ?: now()->toDateString();
        $dados = $this->coletarDadosHidrologicoProvincial($data_emissao);

        $boletim = Boletim::create([
            'tipo' => 'hidrologico_provincial',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => $data_emissao,
            'hora_emissao' => now()->toTimeString(),
            'situacao_prevalecente' => $this->gerarSituacaoProvincial($dados),
            'previsao_hidrologica' => 'Para as próximas 24 horas, não prevê-se alterações significativas do actual cenário hidrológico',
            'alertas_precaucoes' => 'A ARA-Norte, IP, apela à sociedade para a observância de medidas de precaução, evitar a travessia de cursos de água. Apela-se igualmente ao acompanhamento da informação hidrológica disseminada pela ARA Norte, IP, e outras entidades competentes.',
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarBoletimInformativoBarragens($data_emissao = null)
    {
        $data_emissao = $data_emissao ?: now()->toDateString();
        $dados = $this->coletarDadosBarragens($data_emissao);

        $boletim = Boletim::create([
            'tipo' => 'informativo_barragens',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => $data_emissao,
            'hora_emissao' => now()->toTimeString(),
            'situacao_prevalecente' => $this->gerarSituacaoBarragens($dados),
            'previsao_hidrologica' => 'Nas próximas 24 horas não se prevê alterações significativas dos níveis hidrométricos nas bacias hidrográficas da região.',
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarPDF($boletim)
    {
        $dados = $boletim->dados_completos;

        if ($boletim->tipo === 'hidrologico_regional') {
            $view = 'boletins.templates.regional';
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $view = 'boletins.templates.provincial';
        } elseif ($boletim->tipo === 'informativo_barragens') {
            $view = 'boletins.pdf.informativo_barragens';
        } else {
            throw new \Exception('Tipo de boletim não suportado para geração de PDF.');
        }

        // Preparar dados do template
        $templateData = $this->prepararDadosTemplate($boletim);

        $pdf = Pdf::loadView($view, $templateData)
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'defaultFont' => 'DejaVu Sans',
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
            ]);

        $filename = "boletim_{$boletim->tipo}_{$boletim->numero_edicao}_{$boletim->data_emissao->format('Y-m-d')}.pdf";
        $path = "boletins/{$filename}";

        Storage::disk('local')->put($path, $pdf->output());

        $boletim->update(['arquivo_pdf' => $path]);

        return $path;
    }

    public function exportPDF($boletim)
    {
        $templateData = $this->prepararDadosTemplate($boletim);

        if ($boletim->tipo === 'hidrologico_regional') {
            $view = 'boletins.templates.regional';
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $view = 'boletins.templates.provincial';
        } elseif ($boletim->tipo === 'informativo_barragens') {
            $view = 'boletins.templates.informativo_barragens';
        } else {
            throw new \Exception('Tipo de boletim não suportado para exportação de PDF.');
        }

        $pdf = Pdf::loadView($view, $templateData)
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'defaultFont' => 'DejaVu Sans',
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
            ]);

        $filename = "boletim_{$boletim->tipo}_{$boletim->numero_edicao}_{$boletim->data_emissao->format('Y-m-d')}.pdf";

        return $pdf->download($filename);
    }

    public function exportWord($boletim)
    {
        $templateData = $this->prepararDadosTemplate($boletim);

        $phpWord = new PhpWord();
        $section = $phpWord->addSection([
            'paperSize' => 'A4',
            'marginLeft' => 800,
            'marginRight' => 800,
            'marginTop' => 800,
            'marginBottom' => 800,
        ]);

        // Cabeçalho
        $headerTable = $section->addTable([
            'borderSize' => 0,
            'cellMargin' => 50,
        ]);
        $headerRow = $headerTable->addRow();
        $headerRow->addCell(8000)->addText(
            'ARA Norte,IP',
            ['name' => 'Arial', 'size' => 18, 'bold' => true, 'color' => '0066cc']
        );
        $headerRow->addCell(2000)->addText(
            'Logo Ministério',
            ['name' => 'Arial', 'size' => 10, 'color' => '666666'],
            ['alignment' => 'center']
        );

        $section->addTextBreak(1);

        // Título
        if ($boletim->tipo === 'hidrologico_regional') {
            $titulo = "BOLETIM HIDROLÓGICO REGIONAL: Nº {$templateData['numero_edicao']}";
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $titulo = "BOLETIM HIDROLÓGICO DA PROVÍNCIA DE NAMPULA: Nº {$templateData['numero_edicao']}";
        } else {
            $titulo = "BOLETIM INFORMATIVO DE BARRAGENS: Nº {$templateData['numero_edicao']}";
        }

        $section->addText(
            $titulo,
            ['name' => 'Arial', 'size' => 14, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addText(
            'DIA ' . \Carbon\Carbon::parse($templateData['data_emissao'])->format('d.m.Y'),
            ['name' => 'Arial', 'size' => 12, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addTextBreak(2);

        // Seção 1: Situação Prevalecente
        $section->addText(
            '1. SITUAÇÃO PREVALECENTE',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        if ($boletim->tipo === 'hidrologico_provincial') {
            $section->addText(
                'Precipitação registada nas últimas 24 horas nas bacias hidrográfica da província de Nampula:',
                ['name' => 'Arial', 'size' => 11, 'bold' => true]
            );

            $section->addText(
                'Nas últimas 24 horas houve registo de precipitação na rede de observação da região Norte:',
                ['name' => 'Arial', 'size' => 11]
            );

            $section->addText(
                'BACIA HIDROGRÁFICA DO LÚRIO (P-201 Ribáuè: 2.6 mm; P-1156 Malema: 1.2 mm).',
                ['name' => 'Arial', 'size' => 11, 'italic' => true]
            );
        }

        $section->addTextBreak(1);

        // Seção 2: Enchimento das Albufeiras
        $section->addText(
            $boletim->tipo === 'hidrologico_provincial' ? '2. SITUAÇÃO DE ENCHIMENTO DAS ALBUFEIRAS' : '2. ENCHIMENTO DAS PRINCIPAIS ALBUFEIRAS',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        foreach ($templateData['barragens'] ?? [] as $barragem) {
            $nome = is_array($barragem) ? ($barragem['nome'] ?? 'N/A') : $barragem->nome;
            $enchimento = is_array($barragem) ? ($barragem['percentual_enchimento'] ?? 0) : $barragem->percentual_enchimento;
            $volume = is_array($barragem) ? ($barragem['volume_atual'] ?? 0) : $barragem->volume_atual;
            $caudal = is_array($barragem) ? ($barragem['caudal_efluente'] ?? 0) : $barragem->caudal_efluente;
            $nivel_atual = is_array($barragem) ? ($barragem['nivel_atual'] ?? 0) : $barragem->nivel_atual;
            $nivel_alerta = is_array($barragem) ? ($barragem['nivel_alerta'] ?? 0) : $barragem->nivel_alerta;

            $texto = "Barragem de {$nome}: ({$enchimento}%, {$volume} Mm³), Qef = {$caudal} m³/s; " .
                    ($nivel_atual > $nivel_alerta ? 'Nível a baixar' : 'Nível estacionário') . ".";

            $section->addText(
                "• " . $texto,
                ['name' => 'Arial', 'size' => 11]
            );
        }

        $section->addTextBreak(1);

        // Seção 3: Previsão Hidrológica
        $section->addText(
            '3. PREVISÃO HIDROLÓGICA' . ($boletim->tipo === 'hidrologico_provincial' ? '' : ' PARA AS PRÓXIMAS 24 HORAS'),
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        $section->addText(
            $boletim->previsao_hidrologica,
            ['name' => 'Arial', 'size' => 11]
        );

        $section->addTextBreak(1);

        // Seção 4: Alertas e Precauções
        $section->addText(
            '4. ALERTA E PRECAUÇÕES',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        $section->addText(
            $boletim->alertas_precaucoes,
            ['name' => 'Arial', 'size' => 11]
        );

        if ($boletim->tipo === 'hidrologico_provincial') {
            $section->addTextBreak(2);

            // Legenda
            $section->addText(
                'Legenda:',
                ['name' => 'Arial', 'size' => 11, 'bold' => true]
            );

            $legendas = [
                'NA: Nível de alerta',
                'S/NA: Sem Nível de alerta',
                'SI: Sem informação',
                'Qef: Vazão efluente'
            ];

            foreach ($legendas as $legenda) {
                $section->addText(
                    $legenda,
                    ['name' => 'Arial', 'size' => 10]
                );
            }
        }

        $section->addTextBreak(3);

        // Assinatura
        $section->addText(
            'O chefe do Departamento dos Recursos Hídricos',
            ['name' => 'Arial', 'size' => 11],
            ['alignment' => 'center']
        );

        $section->addTextBreak(3);

        $section->addText(
            '________________________________',
            ['name' => 'Arial', 'size' => 11],
            ['alignment' => 'center']
        );

        $section->addText(
            $templateData['responsavel_nome'],
            ['name' => 'Arial', 'size' => 11, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addText(
            "({$templateData['responsavel_cargo']})",
            ['name' => 'Arial', 'size' => 10],
            ['alignment' => 'center']
        );

        // Salvar o arquivo
        $numeroEdicao = str_replace(['/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $boletim->numero_edicao);
        $filename = "boletim_{$boletim->tipo}_{$numeroEdicao}_{$boletim->data_emissao->format('Y-m-d')}.docx";

        $objWriter = IOFactory::createWriter($phpWord, 'Word2007');

        // Usar output buffer para capturar o conteúdo
        ob_start();
        $objWriter->save('php://output');
        $wordContent = ob_get_clean();

        return response($wordContent)
            ->header('Content-Type', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    public function exportExcel($boletim)
    {
        $numeroEdicao = str_replace(['/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $boletim->numero_edicao);
        $filename = "boletim_{$boletim->tipo}_{$numeroEdicao}_{$boletim->data_emissao}.xlsx";

        return Excel::download(new BoletimExport($boletim), $filename);
    }

    private function coletarDadosHidrologicoRegional($data_inicio, $data_fim)
    {
        $dados = [
            'bacias' => [],
            'estacoes_por_bacia' => [],
            'precipitacao_registrada' => false,
            'precipitacao_estacoes' => [],
            'resumo_niveis' => 'Os níveis hidrométricos nas bacias hidrográficas continuam a baixar.',
        ];

        $bacias = BaciaHidrografica::with(['estacoes' => function($query) {
            $query->where('estado', 'activa')->where('tipo', 'hidrometrica');
        }])->get();

        foreach ($bacias as $bacia) {
            $dados['bacias'][$bacia->nome] = [
                'nome' => $bacia->nome,
                'codigo' => $bacia->codigo,
                'estacoes' => [],
            ];

            foreach ($bacia->estacoes as $estacao) {
                $leituras = LeituraEstacao::where('estacao_id', $estacao->id)
                    ->whereBetween('data_leitura', [$data_inicio, $data_fim])
                    ->orderBy('data_leitura', 'desc')
                    ->get();

                $dados['bacias'][$bacia->nome]['estacoes'][] = [
                    'nome' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'nivel_alerta' => $estacao->nivel_alerta,
                    'leituras' => $leituras->map(function($leitura) {
                        return [
                            'data' => $leitura->data_leitura->format('d/m/y'),
                            'nivel_6h' => $leitura->nivel_6h,
                            'nivel_12h' => $leitura->nivel_12h,
                            'nivel_18h' => $leitura->nivel_18h,
                            'precipitacao' => $leitura->precipitacao_mm,
                        ];
                    })->toArray(),
                ];

                // Verificar se houve precipitação
                if ($leituras->where('precipitacao_mm', '>', 0)->count() > 0) {
                    $dados['precipitacao_registrada'] = true;
                }
            }
        }

        // Coletar dados específicos de precipitação das estações pluviométricas
        $estacoes_precipitacao = Estacao::where('estado', 'activa')
            ->where('tipo', 'pluviometrica')
            ->whereIn('provincia', ['Nampula', 'Cabo Delgado', 'Niassa'])
            ->with(['leituras' => function($query) use ($data_inicio, $data_fim) {
                $query->whereBetween('data_leitura', [$data_inicio, $data_fim])
                    ->where('precipitacao_mm', '>', 0)
                    ->orderBy('data_leitura', 'desc');
            }, 'bacia_hidrografica'])
            ->get();

        foreach ($estacoes_precipitacao as $estacao) {
            foreach ($estacao->leituras as $leitura) {
                $dados['precipitacao_estacoes'][] = [
                    'estacao' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'precipitacao' => $leitura->precipitacao_mm,
                    'data' => $leitura->data_leitura->format('d/m/Y'),
                    'bacia' => $estacao->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['precipitacao_registrada'] = true;
            }
        }

        // Gerar análise dinâmica dos níveis hidrométricos
        $dados['analise_niveis'] = $this->analisarNiveisHidrometricos($dados);

        // Estruturar dados para exibição detalhada
        $dados['estacoes'] = $this->estruturarDadosEstacoes($dados);

        return $dados;
    }

    private function analisarNiveisHidrometricos($dados)
    {
        $analise = [];

        // Verificar se há dados de estações hidroméricas
        $total_estacoes_com_dados = 0;
        $estacoes_com_tendencia_subida = 0;
        $estacoes_com_tendencia_descida = 0;
        $estacoes_sem_dados = [];

        foreach ($dados['bacias'] as $bacia) {
            foreach ($bacia['estacoes'] as $estacao) {
                if (!empty($estacao['leituras'])) {
                    $total_estacoes_com_dados++;

                    // Analisar tendência dos últimos 2 dias se houver dados suficientes
                    $leituras = collect($estacao['leituras']);
                    if ($leituras->count() >= 2) {
                        $nivel_recente = $leituras->first()['nivel_18h'] ?? $leituras->first()['nivel_12h'] ?? $leituras->first()['nivel_6h'];
                        $nivel_anterior = $leituras->get(1)['nivel_18h'] ?? $leituras->get(1)['nivel_12h'] ?? $leituras->get(1)['nivel_6h'];

                        if ($nivel_recente && $nivel_anterior) {
                            if ($nivel_recente > $nivel_anterior) {
                                $estacoes_com_tendencia_subida++;
                            } elseif ($nivel_recente < $nivel_anterior) {
                                $estacoes_com_tendencia_descida++;
                            }
                        }
                    }
                } else {
                    $estacoes_sem_dados[] = $estacao['nome'];
                }
            }
        }

        // Verificar comunicações por província
        $cabo_delgado_estacoes = 0;
        $cabo_delgado_sem_dados = 0;

        // Contar estações de Cabo Delgado sem dados (simplificado)
        foreach ($estacoes_sem_dados as $nome_estacao) {
            if (stripos($nome_estacao, 'pemba') !== false || stripos($nome_estacao, 'palma') !== false || stripos($nome_estacao, 'montaquez') !== false) {
                $cabo_delgado_sem_dados++;
            }
        }

        // Gerar texto dinâmico
        if ($total_estacoes_com_dados == 0) {
            $analise['texto'] = "• Não há dados disponíveis de níveis hidrométricos para as últimas 24 horas.";
        } else {
            if ($estacoes_com_tendencia_descida > $estacoes_com_tendencia_subida) {
                $analise['texto'] = "• Nas últimas 24h observou-se tendência de descida dos níveis hidrométricos nas bacias hidrográficas da região.";
            } elseif ($estacoes_com_tendencia_subida > $estacoes_com_tendencia_descida) {
                $analise['texto'] = "• Nas últimas 24h observou-se tendência de subida dos níveis hidrométricos nas bacias hidrográficas da região.";
            } else {
                $analise['texto'] = "• Nas últimas 24h observou-se estabilidade dos níveis hidrométricos nas bacias hidrográficas da região.";
            }
        }

        // Adicionar nota sobre comunicações se há estações sem dados
        if (count($estacoes_sem_dados) > 0) {
            if ($cabo_delgado_sem_dados > 0) {
                $analise['texto'] .= " Importa referir que para a província de Cabo Delgado enfrenta-se dificuldades na obtenção de dados, dificultando assim o fluxo de informação.";
            } else {
                $analise['texto'] .= " Algumas estações da região não reportaram dados para o período em análise.";
            }
        }

        return $analise;
    }

    private function estruturarDadosEstacoes($dados)
    {
        $estacoes_estruturadas = [];

        // Adicionar estações de precipitação
        foreach ($dados['precipitacao_estacoes'] ?? [] as $estacao) {
            $estacoes_estruturadas[] = [
                'nome' => $estacao['estacao'],
                'codigo' => $estacao['codigo'],
                'tipo' => 'pluviometrica',
                'ultima_leitura' => $estacao['data'],
                'precipitacao_mm' => $estacao['precipitacao'],
                'valor_leitura' => $estacao['precipitacao'] . ' mm',
                'estado' => 'activa',
                'bacia' => $estacao['bacia']
            ];
        }

        // Adicionar estações hidrométricas das bacias
        foreach ($dados['bacias'] ?? [] as $bacia) {
            foreach ($bacia['estacoes'] ?? [] as $estacao) {
                if (!empty($estacao['leituras'])) {
                    $ultima_leitura = collect($estacao['leituras'])->first();
                    $nivel = $ultima_leitura['nivel_18h'] ?? $ultima_leitura['nivel_12h'] ?? $ultima_leitura['nivel_6h'];

                    $estacoes_estruturadas[] = [
                        'nome' => $estacao['nome'],
                        'codigo' => $estacao['codigo'],
                        'tipo' => 'hidrometrica',
                        'ultima_leitura' => $ultima_leitura['data'],
                        'nivel_hidrometrico' => $nivel,
                        'valor_leitura' => $nivel ? $nivel . ' m' : 'N/A',
                        'estado' => 'activa',
                        'bacia' => $bacia['nome']
                    ];
                } else {
                    $estacoes_estruturadas[] = [
                        'nome' => $estacao['nome'],
                        'codigo' => $estacao['codigo'],
                        'tipo' => 'hidrometrica',
                        'ultima_leitura' => 'Sem dados',
                        'nivel_hidrometrico' => null,
                        'valor_leitura' => 'N/A',
                        'estado' => 'sem_dados',
                        'bacia' => $bacia['nome']
                    ];
                }
            }
        }

        return $estacoes_estruturadas;
    }

    private function coletarDadosBarragens($data_emissao)
    {
        $dados = [
            'barragens' => [],
            'data_emissao' => $data_emissao,
            'tem_dados' => false,
        ];

        // Coletar todas as barragens ativas com suas leituras mais recentes
        $barragens = Barragem::where('estado', 'activa')
            ->with(['leituras' => function($query) use ($data_emissao) {
                $query->whereDate('data_leitura', '<=', $data_emissao)
                    ->orderBy('data_leitura', 'desc')
                    ->orderBy('hora_leitura', 'desc')
                    ->take(1);
            }, 'bacia_hidrografica'])
            ->get();

        foreach ($barragens as $barragem) {
            $leitura_recente = $barragem->leituras->first();

            if ($leitura_recente) {
                // Determinar status do nível
                $status_nivel = 'normal';
                if ($leitura_recente->cota_actual >= $barragem->nivel_emergencia) {
                    $status_nivel = 'emergencia';
                } elseif ($leitura_recente->cota_actual >= $barragem->nivel_alerta) {
                    $status_nivel = 'alerta';
                }

                $dados['barragens'][] = [
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'cota' => $leitura_recente->cota_actual,
                    'cota_actual' => $leitura_recente->cota_actual,
                    'enchimento' => $leitura_recente->percentagem_enchimento,
                    'percentagem_enchimento' => $leitura_recente->percentagem_enchimento,
                    'volume_atual' => $leitura_recente->volume_actual,
                    'volume_actual' => $leitura_recente->volume_actual,
                    'comparacao_ano_anterior' => $leitura_recente->comparacao_ano_anterior ?? 'N/D',
                    'data_leitura' => $leitura_recente->data_leitura->format('d/m/Y'),
                    'hora_leitura' => $leitura_recente->hora_leitura->format('H:i'),
                    'caudal_ecologico' => $leitura_recente->caudal_ecologico,
                    'descarga_total' => $leitura_recente->descarga_total,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'nivel_emergencia' => $barragem->nivel_emergencia,
                    'status_nivel' => $status_nivel,
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['tem_dados'] = true;
            } else {
                // Incluir barragem sem dados recentes mas informar status
                $dados['barragens'][] = [
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'cota' => null,
                    'enchimento' => null,
                    'volume_atual' => null,
                    'comparacao_ano_anterior' => 'Sem dados',
                    'data_leitura' => 'N/D',
                    'hora_leitura' => 'N/D',
                    'caudal_ecologico' => null,
                    'descarga_total' => null,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                ];
            }
        }

        return $dados;
    }

    private function gerarSituacaoPrevalecente($dados)
    {
        $situacao = "**Precipitação**\n\n";
        
        if ($dados['precipitacao_registrada']) {
            $situacao .= "Na rede regional de observação hidroclimatológica foram registadas precipitações em algumas estações.\n\n";
        } else {
            $situacao .= "Na rede regional de observação hidroclimatológica não houve registo de precipitação.\n\n";
        }

        $situacao .= "**Níveis Hidrométricos**\n\n";
        $situacao .= $dados['resumo_niveis'];

        return $situacao;
    }

    private function gerarSituacaoBarragens($dados)
    {
        $situacao = "**Níveis Hidrométricos**\n\n";

        if (!empty($dados['barragens']) && $dados['tem_dados']) {
            $situacao .= "Situação atual das principais barragens da região:\n\n";

            foreach ($dados['barragens'] as $barragem) {
                if ($barragem['enchimento'] !== null) {
                    $situacao .= "• **{$barragem['nome']}**: {$barragem['enchimento']}% de enchimento";
                    if ($barragem['volume_atual']) {
                        $situacao .= " ({$barragem['volume_atual']} Mm³)";
                    }
                    if ($barragem['caudal_ecologico']) {
                        $situacao .= ", Qef = {$barragem['caudal_ecologico']} m³/s";
                    }
                    $situacao .= ".\n";
                } else {
                    $situacao .= "• **{$barragem['nome']}**: Sem dados disponíveis para o período.\n";
                }
            }
        } else {
            $situacao .= "Não há dados disponíveis de barragens para o período em análise.";
        }

        return $situacao;
    }

    private function gerarPrevisaoHidrologica($dados)
    {
        $previsao = "Para as próximas 24 horas, ";

        // Analisar dados de precipitação para fazer previsão dinâmica
        if ($dados['precipitacao_registrada']) {
            $previsao .= "devido à precipitação registrada na região, prevê-se ligeiro aumento dos níveis hidrométricos nas bacias afetadas. ";
            $previsao .= "Recomenda-se monitoramento contínuo dos níveis de água nas principais estações da região Norte.";
        } else {
            $previsao .= "não se prevê alterações significativas dos níveis hidrométricos nas bacias hidrográficas da região, ";
            $previsao .= "mantendo-se a tendência atual de estabilidade dos níveis.";
        }

        // Adicionar informação sobre estação climatológica
        $mes_atual = now()->month;
        if ($mes_atual >= 11 || $mes_atual <= 4) {
            $previsao .= "\n\nNota: Estamos na época chuvosa (Novembro-Abril), período de maior atenção hidrológica na região.";
        } else {
            $previsao .= "\n\nNota: Estamos na época seca (Maio-Outubro), período de menor atividade hidrológica na região.";
        }

        return $previsao;
    }

    private function gerarAlertasPrecaucoes($dados)
    {
        $alertas = [];

        // Verificar se há precipitação significativa
        if ($dados['precipitacao_registrada']) {
            $alertas[] = "Devido à precipitação registrada, recomenda-se atenção redobrada na travessia de cursos de água.";
        }

        // Verificar época do ano
        $mes_atual = now()->month;
        if ($mes_atual >= 11 || $mes_atual <= 4) {
            $alertas[] = "Durante a época chuvosa, mantenha-se informado sobre as condições hidrológicas através dos canais oficiais.";
        }

        // Alertas padrão da ARA Norte
        $alertas[] = "A ARA-Norte, IP, apela à sociedade para a observância de medidas de precaução, evitar a travessia de cursos de água.";
        $alertas[] = "Apela-se igualmente ao acompanhamento da informação hidrológica disseminada pela ARA Norte, IP, e outras entidades competentes.";

        if (empty($alertas)) {
            return "Não há alertas ou precauções especiais para o período em análise.";
        }

        return implode("\n\n", $alertas);
    }

    private function coletarDadosHidrologicoProvincial($data_emissao)
    {
        $dados = [
            'precipitacao_estacoes' => [],
            'barragens' => [],
            'data_emissao' => $data_emissao,
            'tem_dados' => false,
        ];

        // Coletar dados reais de precipitação das estações da região
        $estacoes = Estacao::where('estado', 'activa')
            ->where('tipo', 'pluviometrica')
            ->whereIn('provincia', ['Nampula', 'Cabo Delgado', 'Niassa'])
            ->with(['leituras' => function($query) use ($data_emissao) {
                $query->whereDate('data_leitura', $data_emissao)
                    ->where('precipitacao_mm', '>', 0);
            }])
            ->get();

        foreach ($estacoes as $estacao) {
            foreach ($estacao->leituras as $leitura) {
                $dados['precipitacao_estacoes'][] = [
                    'estacao' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'precipitacao' => $leitura->precipitacao_mm,
                    'data' => $leitura->data_leitura->format('d/m/Y'),
                    'bacia' => $estacao->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['tem_dados'] = true;
            }
        }

        // Coletar dados reais das barragens
        $barragens = Barragem::where('estado', 'activa')
            ->with(['ultima_leitura', 'bacia_hidrografica'])
            ->get();

        foreach ($barragens as $barragem) {
            $ultima_leitura = $barragem->ultima_leitura;

            if ($ultima_leitura) {
                $dados['barragens'][] = [
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'percentual_enchimento' => $ultima_leitura->percentagem_enchimento,
                    'volume_atual' => $ultima_leitura->volume_actual,
                    'caudal_efluente' => $ultima_leitura->caudal_ecologico,
                    'caudal_descarga' => $ultima_leitura->descarga_total,
                    'nivel_atual' => $ultima_leitura->cota_actual,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'data_leitura' => $ultima_leitura->data_leitura->format('d/m/Y'),
                    'hora_leitura' => $ultima_leitura->hora_leitura->format('H:i'),
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['tem_dados'] = true;
            }
        }

        return $dados;
    }

    private function gerarSituacaoProvincial($dados)
    {
        $situacao = "**Precipitação registada nas últimas 24 horas nas bacias hidrográficas da região Norte:**\n\n";

        if (!empty($dados['precipitacao_estacoes'])) {
            $situacao .= "Nas últimas 24 horas houve registo de precipitação na rede de observação da região Norte:\n\n";

            // Agrupar por bacia
            $precipitacao_por_bacia = [];
            foreach ($dados['precipitacao_estacoes'] as $registro) {
                $bacia = $registro['bacia'];
                if (!isset($precipitacao_por_bacia[$bacia])) {
                    $precipitacao_por_bacia[$bacia] = [];
                }
                $precipitacao_por_bacia[$bacia][] = $registro;
            }

            foreach ($precipitacao_por_bacia as $bacia => $registros) {
                $situacao .= "**BACIA HIDROGRÁFICA " . strtoupper($bacia) . "** (";
                $estacoes_texto = [];
                foreach ($registros as $registro) {
                    $estacoes_texto[] = "{$registro['codigo']} {$registro['estacao']}: {$registro['precipitacao']} mm";
                }
                $situacao .= implode('; ', $estacoes_texto) . ").\n\n";
            }
        } else {
            $situacao .= "Nas últimas 24 horas não houve registo de precipitação na rede de observação da região Norte.\n\n";
        }

        $situacao .= "**Comportamento dos níveis hidrométricos nas principais bacias da região:**\n\n";

        if (!empty($dados['barragens'])) {
            $situacao .= "Nas últimas 24h observou-se comportamento variado dos níveis nas barragens da região:\n\n";

            foreach ($dados['barragens'] as $barragem) {
                $tendencia = $barragem['nivel_atual'] > $barragem['nivel_alerta'] ? 'nível elevado' : 'nível normal';
                $situacao .= "• **{$barragem['nome']}**: {$barragem['percentual_enchimento']}% de enchimento, {$tendencia}.\n";
            }
        } else {
            $situacao .= "Não há dados disponíveis de níveis hidrométricos para o período em análise.";
        }

        return $situacao;
    }

    private function prepararDadosTemplate($boletim)
    {
        $dados = $boletim->dados_completos;

        // Get system settings
        $settings = SystemSetting::all_settings();

        $templateData = [
            'boletim' => $boletim,
            'numero_edicao' => $boletim->numero_edicao,
            'data_emissao' => $boletim->data_emissao,
            'dados' => $dados,
            'responsavel_nome' => $settings['responsible_person_name'] ?? 'João Alberto Miguel Sitoe',
            'responsavel_cargo' => $settings['responsible_person_title'] ?? 'Técnico Superior N1',
            'settings' => $settings,
        ];

        if ($boletim->tipo === 'hidrologico_regional') {
            $templateData['estacoes'] = $this->gerarListaEstacoes();
            $templateData['barragens'] = $this->prepararBarragensRegional($dados);
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $templateData['barragens'] = $dados['barragens'] ?? [];
        }

        return $templateData;
    }

    private function gerarListaEstacoes()
    {
        return [
            ['nome' => 'Ribane'],
            ['nome' => 'Larde'],
            ['nome' => 'Lichinga'],
            ['nome' => 'Ligonha'],
            ['nome' => 'Luambala'],
            ['nome' => 'Lúrio'],
            ['nome' => 'Malehice'],
            ['nome' => 'Messalo'],
            ['nome' => 'Mecuburi'],
            ['nome' => 'Megaruma'],
            ['nome' => 'Melaja'],
            ['nome' => 'Meluli'],
            ['nome' => 'Muaguide'],
            ['nome' => 'Mnarani'],
            ['nome' => 'Mieze'],
            ['nome' => 'Mogincual'],
            ['nome' => 'Montaquez'],
            ['nome' => 'Mongo'],
            ['nome' => 'Meconta'],
            ['nome' => 'Mogovolas'],
            ['nome' => 'Moma'],
            ['nome' => 'Mossuril'],
            ['nome' => 'Murrupula'],
            ['nome' => 'Muecate'],
            ['nome' => 'Nacala'],
            ['nome' => 'Nacarôa'],
            ['nome' => 'Namige'],
            ['nome' => 'Nampula'],
            ['nome' => 'Rapale'],
            ['nome' => 'Ribaué'],
            ['nome' => 'Rovuma'],
        ];
    }

    private function prepararBarragensRegional($dados)
    {
        // Coletar dados reais das barragens da região
        $barragens = Barragem::where('estado', 'activa')
            ->with(['leituras' => function($query) {
                $query->orderBy('data_leitura', 'desc')
                    ->orderBy('hora_leitura', 'desc')
                    ->take(1);
            }])
            ->get();

        $barragens_preparadas = [];

        foreach ($barragens as $barragem) {
            $leitura_recente = $barragem->leituras->first();

            if ($leitura_recente) {
                $barragens_preparadas[] = (object)[
                    'nome' => $barragem->nome,
                    'percentual_enchimento' => $leitura_recente->percentagem_enchimento,
                    'volume_atual' => $leitura_recente->volume_actual,
                    'caudal_efluente' => $leitura_recente->caudal_ecologico ?? '0',
                    'nivel_atual' => $leitura_recente->cota_actual,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'data_leitura' => $leitura_recente->data_leitura->format('d/m/Y'),
                    'descarga_total' => $leitura_recente->descarga_total ?? 0,
                ];
            } else {
                // Incluir barragem mesmo sem dados recentes
                $barragens_preparadas[] = (object)[
                    'nome' => $barragem->nome,
                    'percentual_enchimento' => 0,
                    'volume_atual' => 0,
                    'caudal_efluente' => '0',
                    'nivel_atual' => 0,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'data_leitura' => 'N/D',
                    'descarga_total' => 0,
                ];
            }
        }

        // Se não há barragens com dados, retornar array vazio para evitar erro no template
        return $barragens_preparadas;
    }
}