<?php

namespace App\Services;

use Twilio\Rest\Client;
use Exception;
use Illuminate\Support\Facades\Log;

class TwilioSmsService
{
    protected $client;
    protected $fromNumber;

    public function __construct()
    {
        $this->client = new Client(
            config('services.twilio.sid'),
            config('services.twilio.auth_token')
        );
        $this->fromNumber = config('services.twilio.phone_number');
    }

    /**
     * Enviar SMS simples
     */
    public function sendSms($to, $message)
    {
        try {
            $message = $this->client->messages->create(
                $to,
                [
                    'from' => $this->fromNumber,
                    'body' => $message
                ]
            );

            Log::info('SMS enviado com sucesso', [
                'to' => $to,
                'message_sid' => $message->sid,
                'status' => $message->status
            ]);

            return [
                'success' => true,
                'message_sid' => $message->sid,
                'status' => $message->status
            ];
        } catch (Exception $e) {
            Log::error('Erro ao enviar SMS', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Enviar SMS de alerta de emergência
     */
    public function sendEmergencyAlert($to, $alerta)
    {
        $message = "🚨 ALERTA DE EMERGÊNCIA 🚨\n\n";
        $message .= "Tipo: {$alerta->getTipoFormatado()}\n";
        $message .= "Local: {$alerta->getNomeInfrastrutura()}\n";
        $message .= "Detalhes: {$alerta->titulo}\n";
        $message .= "Hora: " . $alerta->created_at->format('d/m/Y H:i') . "\n\n";
        $message .= "⚠️ AÇÃO IMEDIATA REQUERIDA ⚠️\n";
        $message .= "Sistema Hidrológico - Moçambique";

        return $this->sendSms($to, $message);
    }

    /**
     * Enviar SMS de alerta normal
     */
    public function sendAlert($to, $alerta)
    {
        $nivelIcon = $this->getNivelIcon($alerta->nivel);
        
        $message = "{$nivelIcon} ALERTA DO SISTEMA\n\n";
        $message .= "Nível: {$alerta->getNivelFormatado()}\n";
        $message .= "Tipo: {$alerta->getTipoFormatado()}\n";
        $message .= "Local: {$alerta->getNomeInfrastrutura()}\n";
        $message .= "Descrição: {$alerta->descricao}\n";
        
        if ($alerta->valor_registado) {
            $message .= "Valor: {$alerta->valor_registado}\n";
        }
        
        if ($alerta->valor_limite) {
            $message .= "Limite: {$alerta->valor_limite}\n";
        }
        
        $message .= "Hora: " . $alerta->created_at->format('d/m/Y H:i') . "\n\n";
        $message .= "Sistema Hidrológico - Moçambique";

        return $this->sendSms($to, $message);
    }

    /**
     * Enviar SMS de resolução de alerta
     */
    public function sendAlertResolved($to, $alerta)
    {
        $message = "✅ ALERTA RESOLVIDO\n\n";
        $message .= "Alerta: {$alerta->titulo}\n";
        $message .= "Local: {$alerta->getNomeInfrastrutura()}\n";
        $message .= "Resolvido em: " . $alerta->data_fim->format('d/m/Y H:i') . "\n";
        $message .= "Duração: {$alerta->getDuracaoFormatada()}\n\n";
        
        if ($alerta->acoes_tomadas) {
            $message .= "Ações: " . substr($alerta->acoes_tomadas, 0, 100) . "...\n\n";
        }
        
        $message .= "Sistema Hidrológico - Moçambique";

        return $this->sendSms($to, $message);
    }

    /**
     * Enviar SMS de escalação de alerta
     */
    public function sendAlertEscalation($to, $alerta, $minutosSemResposta)
    {
        $message = "⚠️ ESCALAÇÃO DE ALERTA\n\n";
        $message .= "Alerta sem resposta há {$minutosSemResposta} minutos\n\n";
        $message .= "Tipo: {$alerta->getTipoFormatado()}\n";
        $message .= "Nível: {$alerta->getNivelFormatado()}\n";
        $message .= "Local: {$alerta->getNomeInfrastrutura()}\n";
        $message .= "Criado: " . $alerta->created_at->format('d/m/Y H:i') . "\n\n";
        $message .= "🔔 INTERVENÇÃO SUPERVISORA NECESSÁRIA\n";
        $message .= "Sistema Hidrológico - Moçambique";

        return $this->sendSms($to, $message);
    }

    /**
     * Enviar relatório diário de alertas
     */
    public function sendDailyReport($to, $stats)
    {
        $message = "📊 RELATÓRIO DIÁRIO DE ALERTAS\n";
        $message .= date('d/m/Y') . "\n\n";
        $message .= "Alertas hoje: {$stats['alertas_hoje']}\n";
        $message .= "Ativos: {$stats['alertas_ativos']}\n";
        $message .= "Resolvidos: {$stats['alertas_resolvidos']}\n";
        $message .= "Emergências: {$stats['emergencias']}\n\n";
        
        if ($stats['alertas_ativos'] > 0) {
            $message .= "⚠️ {$stats['alertas_ativos']} alerta(s) ainda ativo(s)\n";
        } else {
            $message .= "✅ Todos os alertas resolvidos\n";
        }
        
        $message .= "\nSistema Hidrológico - Moçambique";

        return $this->sendSms($to, $message);
    }

    /**
     * Testar configuração do Twilio
     */
    public function testConnection($testPhoneNumber)
    {
        $message = "✅ Teste de conectividade SMS\n\n";
        $message .= "Se recebeu esta mensagem, o sistema de SMS está funcionando corretamente.\n\n";
        $message .= "Hora do teste: " . now()->format('d/m/Y H:i:s') . "\n";
        $message .= "Sistema Hidrológico - Moçambique";

        return $this->sendSms($testPhoneNumber, $message);
    }

    /**
     * Obter ícone baseado no nível do alerta
     */
    private function getNivelIcon($nivel)
    {
        $icons = [
            'informacao' => 'ℹ️',
            'atencao' => '⚠️',
            'alerta' => '🔔',
            'emergencia' => '🚨'
        ];

        return $icons[$nivel] ?? '📢';
    }

    /**
     * Validar número de telefone
     */
    public function validatePhoneNumber($phoneNumber)
    {
        // Remove espaços e caracteres especiais
        $phoneNumber = preg_replace('/[^0-9+]/', '', $phoneNumber);
        
        // Verifica se é um número de Moçambique (+258)
        if (strpos($phoneNumber, '+258') === 0) {
            return strlen($phoneNumber) === 12; // +258 + 9 dígitos
        }
        
        // Se não tem código do país, adiciona +258
        if (strlen($phoneNumber) === 9 && $phoneNumber[0] === '8') {
            return '+258' . $phoneNumber;
        }
        
        return false;
    }

    /**
     * Formatar número de telefone para Moçambique
     */
    public function formatPhoneNumber($phoneNumber)
    {
        $phoneNumber = preg_replace('/[^0-9+]/', '', $phoneNumber);
        
        if (strpos($phoneNumber, '+258') === 0) {
            return $phoneNumber;
        }
        
        if (strlen($phoneNumber) === 9 && $phoneNumber[0] === '8') {
            return '+258' . $phoneNumber;
        }
        
        return null;
    }
}