<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('barragens', function (Blueprint $table) {
            // Características de Volume Avançadas
            $table->decimal('volume_util', 10, 2)->nullable()->comment('Volume útil em hm³');
            $table->decimal('capacidade_total', 10, 2)->nullable()->comment('Capacidade total em hm³');
            
            // Cotas Técnicas Avançadas
            $table->decimal('cota_nme', 8, 2)->nullable()->comment('Nível Máximo Extraordinário em metros');
            $table->decimal('cota_nmc', 8, 2)->nullable()->comment('Nível Máximo de Cheia em metros');
            $table->decimal('nivel_descargador', 8, 2)->nullable()->comment('Nível do descargador em metros');
            $table->decimal('cota_seguranca', 8, 2)->nullable()->comment('Cota de segurança em metros');
            
            // Características Físicas Estruturais
            $table->decimal('area_inundacao', 12, 2)->nullable()->comment('Área de inundação em m²');
            $table->decimal('altura_barragem', 8, 2)->nullable()->comment('Altura da barragem em metros');
            $table->decimal('comprimento_crista', 10, 2)->nullable()->comment('Comprimento da crista em metros');
            
            // Características Hidrológicas
            $table->string('rio_principal', 100)->nullable()->comment('Nome do rio principal');
            $table->text('afluentes')->nullable()->comment('Lista de afluentes (JSON ou texto)');
            $table->text('estacoes_pluviometricas')->nullable()->comment('Estações pluviométricas associadas (JSON)');
            
            // Classificação Técnica
            $table->enum('tipo_barragem', [
                'terra', 'betao', 'enrocamento', 'mista', 'gravidade', 'arco', 'contraforte'
            ])->default('terra')->comment('Tipo construtivo da barragem');
            
            $table->enum('finalidade_principal', [
                'abastecimento', 'irrigacao', 'energia', 'controle_cheias', 'recreacao', 'piscicultura', 'multipla'
            ])->default('abastecimento')->comment('Finalidade principal da barragem');
            
            $table->enum('tipo_descargador', [
                'superficie', 'fundo', 'lateral', 'tunel', 'misto'
            ])->nullable()->comment('Tipo de descargador');
            
            // Características Adicionais de Segurança
            $table->enum('classe_risco', ['baixo', 'medio', 'alto'])->default('medio')->comment('Classe de risco da barragem');
            $table->enum('categoria_dano', ['baixo', 'medio', 'alto'])->default('medio')->comment('Categoria de dano potencial');
            
            // Informações de Projeto
            $table->string('empresa_projetista', 200)->nullable()->comment('Empresa responsável pelo projeto');
            $table->string('empresa_construtora', 200)->nullable()->comment('Empresa responsável pela construção');
            $table->year('ano_projeto')->nullable()->comment('Ano do projeto');
            $table->year('ano_inicio_construcao')->nullable()->comment('Ano de início da construção');
            $table->year('ano_conclusao')->nullable()->comment('Ano de conclusão da obra');
            
            // Características Operacionais
            $table->boolean('sistema_automatizado')->default(false)->comment('Possui sistema automatizado');
            $table->boolean('telemetria')->default(false)->comment('Possui sistema de telemetria');
            $table->string('frequencia_monitoramento', 50)->default('diario')->comment('Frequência de monitoramento');
            
            // Índices para performance
            $table->index(['tipo_barragem', 'finalidade_principal'], 'idx_barragem_tipo_finalidade');
            $table->index(['classe_risco', 'categoria_dano'], 'idx_barragem_seguranca');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('barragens', function (Blueprint $table) {
            $table->dropColumn([
                'volume_util', 'capacidade_total',
                'cota_nme', 'cota_nmc', 'nivel_descargador', 'cota_seguranca',
                'area_inundacao', 'altura_barragem', 'comprimento_crista',
                'rio_principal', 'afluentes', 'estacoes_pluviometricas',
                'tipo_barragem', 'finalidade_principal', 'tipo_descargador',
                'classe_risco', 'categoria_dano',
                'empresa_projetista', 'empresa_construtora', 
                'ano_projeto', 'ano_inicio_construcao', 'ano_conclusao',
                'sistema_automatizado', 'telemetria', 'frequencia_monitoramento'
            ]);
            
            $table->dropIndex('idx_barragem_tipo_finalidade');
            $table->dropIndex('idx_barragem_seguranca');
        });
    }
};