<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     * Tabela pivot para atribuir utilizadores a divisoes administrativas
     * Suporta multiplas divisoes por utilizador com permissoes customizadas
     */
    public function up(): void
    {
        Schema::create('user_divisoes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')
                  ->constrained()
                  ->onDelete('cascade');
            $table->foreignId('divisao_administrativa_id')
                  ->constrained('divisoes_administrativas')
                  ->onDelete('cascade');

            // Permissoes customizadas para esta divisao especifica
            // Formato: {"grant": ["permissao1"], "revoke": ["permissao2"]}
            $table->json('permissoes_customizadas')->nullable();

            // Nivel de acesso:
            // 'full' = acesso a esta divisao e todos descendentes (distritos, municipios)
            // 'direct' = acesso apenas a esta divisao especifica
            $table->enum('nivel_acesso', ['full', 'direct'])->default('full');

            // Se esta e a divisao principal do utilizador
            $table->boolean('is_primary')->default(false);

            // Periodo de validade (para atribuicoes temporarias)
            $table->date('valido_de')->nullable();
            $table->date('valido_ate')->nullable();

            // Status da atribuicao
            $table->boolean('ativo')->default(true);

            $table->timestamps();

            // Cada utilizador so pode ter uma atribuicao por divisao
            $table->unique(['user_id', 'divisao_administrativa_id'], 'user_divisao_unique');

            // Indices para performance
            $table->index(['user_id', 'ativo']);
            $table->index(['divisao_administrativa_id', 'ativo']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('user_divisoes');
    }
};
