<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

class TesteBoletimSeeder extends Seeder
{
    /**
     * Seeder para teste completo do sistema de boletins ARA Norte
     * Limpa dados existentes e insere dados realistas para teste
     */
    public function run(): void
    {
        $this->command->info('Limpando dados existentes...');

        // Desabilitar verificação de chaves estrangeiras
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');

        // Limpar tabelas na ordem correta (verificando se existem)
        $tabelas = [
            'boletins',
            'alertas',
            'notificacoes_enviadas',
            'leituras_barragens',
            'leituras_estacoes',
            'curvas_volume',
            'formulas_caudal',
            'barragens',
            'estacoes',
            'bacias_hidrograficas',
        ];

        foreach ($tabelas as $tabela) {
            if (Schema::hasTable($tabela)) {
                DB::table($tabela)->truncate();
                $this->command->info("  - Tabela {$tabela} limpa");
            } else {
                $this->command->warn("  - Tabela {$tabela} não existe, ignorando...");
            }
        }

        // Reabilitar verificação de chaves estrangeiras
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        $this->command->info('Inserindo bacias hidrográficas...');
        $this->inserirBacias();

        $this->command->info('Inserindo barragens...');
        $this->inserirBarragens();

        $this->command->info('Inserindo estações...');
        $this->inserirEstacoes();

        $this->command->info('Inserindo curvas cota-volume...');
        $this->inserirCurvasVolume();

        $this->command->info('Inserindo leituras de barragens...');
        $this->inserirLeiturasBarragens();

        $this->command->info('Inserindo leituras de estações...');
        $this->inserirLeiturasEstacoes();

        $this->command->info('Dados de teste inseridos com sucesso!');
    }

    private function inserirBacias(): void
    {
        $bacias = [
            [
                'id' => 1,
                'nome' => 'Lúrio',
                'codigo' => 'BH-LUR',
                'area_km2' => 60800,
                'comprimento_rio_principal' => 605,
                'provincias_abrangidas' => json_encode(['Nampula', 'Cabo Delgado', 'Niassa']),
                'descricao' => 'Bacia hidrográfica do Rio Lúrio, uma das principais da região Norte',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 2,
                'nome' => 'Messalo',
                'codigo' => 'BH-MES',
                'area_km2' => 19200,
                'comprimento_rio_principal' => 350,
                'provincias_abrangidas' => json_encode(['Cabo Delgado']),
                'descricao' => 'Bacia hidrográfica do Rio Messalo',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 3,
                'nome' => 'Monapo',
                'codigo' => 'BH-MON',
                'area_km2' => 6500,
                'comprimento_rio_principal' => 180,
                'provincias_abrangidas' => json_encode(['Nampula']),
                'descricao' => 'Bacia hidrográfica do Rio Monapo',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 4,
                'nome' => 'Megaruma',
                'codigo' => 'BH-MEG',
                'area_km2' => 4500,
                'comprimento_rio_principal' => 120,
                'provincias_abrangidas' => json_encode(['Cabo Delgado']),
                'descricao' => 'Bacia hidrográfica do Rio Megaruma',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        DB::table('bacias_hidrograficas')->insert($bacias);
    }

    private function inserirBarragens(): void
    {
        $barragens = [
            [
                'id' => 1,
                'nome' => 'Nacala',
                'codigo' => 'BAR-NAC-001',
                'bacia_hidrografica_id' => 3,
                'provincia' => 'Nampula',
                'distrito' => 'Nacala-Porto',
                'latitude' => -14.5425,
                'longitude' => 40.6728,
                'volume_maximo' => 45.5,
                'volume_morto' => 5.2,
                'volume_util' => 40.3,
                'capacidade_total' => 45.5,
                'cota_maxima' => 128.5,
                'cota_minima' => 112.0,
                'cota_nmc' => 127.0,
                'cota_npa' => 125.0,
                'cota_nme' => 115.0,
                'tipo_barragem' => 'Terra',
                'finalidade_principal' => 'Abastecimento',
                'estado' => 'activa',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 2,
                'nome' => 'Chipembe',
                'codigo' => 'BAR-CHI-001',
                'bacia_hidrografica_id' => 1,
                'provincia' => 'Nampula',
                'distrito' => 'Mecubúri',
                'latitude' => -14.8567,
                'longitude' => 38.2345,
                'volume_maximo' => 120.0,
                'volume_morto' => 15.0,
                'volume_util' => 105.0,
                'capacidade_total' => 120.0,
                'cota_maxima' => 345.0,
                'cota_minima' => 320.0,
                'cota_nmc' => 343.0,
                'cota_npa' => 340.0,
                'cota_nme' => 325.0,
                'tipo_barragem' => 'Betão',
                'finalidade_principal' => 'Irrigação',
                'estado' => 'activa',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 3,
                'nome' => 'Montepuez',
                'codigo' => 'BAR-MTP-001',
                'bacia_hidrografica_id' => 2,
                'provincia' => 'Cabo Delgado',
                'distrito' => 'Montepuez',
                'latitude' => -13.1256,
                'longitude' => 39.0012,
                'volume_maximo' => 85.0,
                'volume_morto' => 10.0,
                'volume_util' => 75.0,
                'capacidade_total' => 85.0,
                'cota_maxima' => 256.0,
                'cota_minima' => 235.0,
                'cota_nmc' => 254.0,
                'cota_npa' => 251.0,
                'cota_nme' => 240.0,
                'tipo_barragem' => 'Enrocamento',
                'finalidade_principal' => 'Abastecimento',
                'estado' => 'activa',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        DB::table('barragens')->insert($barragens);
    }

    private function inserirEstacoes(): void
    {
        // Estações Pluviométricas (inserir separadamente)
        $pluviometricas = [
            [
                'id' => 1,
                'nome' => 'Ribáuè',
                'codigo' => 'P-201',
                'tipo' => 'pluviometrica',
                'bacia_hidrografica_id' => 1,
                'provincia' => 'Nampula',
                'distrito' => 'Ribáuè',
                'latitude' => -14.9833,
                'longitude' => 38.2667,
                'altitude' => 520,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 2,
                'nome' => 'Malema',
                'codigo' => 'P-1156',
                'tipo' => 'pluviometrica',
                'bacia_hidrografica_id' => 1,
                'provincia' => 'Nampula',
                'distrito' => 'Malema',
                'latitude' => -14.9500,
                'longitude' => 37.4167,
                'altitude' => 580,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 3,
                'nome' => 'Pemba',
                'codigo' => 'P-302',
                'tipo' => 'pluviometrica',
                'bacia_hidrografica_id' => 2,
                'provincia' => 'Cabo Delgado',
                'distrito' => 'Pemba',
                'latitude' => -12.9667,
                'longitude' => 40.5167,
                'altitude' => 45,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 4,
                'nome' => 'Monapo',
                'codigo' => 'P-405',
                'tipo' => 'pluviometrica',
                'bacia_hidrografica_id' => 3,
                'provincia' => 'Nampula',
                'distrito' => 'Monapo',
                'latitude' => -15.0167,
                'longitude' => 40.0833,
                'altitude' => 120,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        // Estações Hidrométricas (inserir separadamente)
        $hidrometricas = [
            [
                'id' => 5,
                'nome' => 'Ponte Lúrio',
                'codigo' => 'H-101',
                'tipo' => 'hidrometrica',
                'bacia_hidrografica_id' => 1,
                'provincia' => 'Nampula',
                'distrito' => 'Eráti',
                'latitude' => -14.2500,
                'longitude' => 39.5000,
                'altitude' => 85,
                'nivel_alerta' => 4.5,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 6,
                'nome' => 'Estação Messalo',
                'codigo' => 'H-201',
                'tipo' => 'hidrometrica',
                'bacia_hidrografica_id' => 2,
                'provincia' => 'Cabo Delgado',
                'distrito' => 'Macomia',
                'latitude' => -12.3500,
                'longitude' => 40.1000,
                'altitude' => 120,
                'nivel_alerta' => 3.8,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'id' => 7,
                'nome' => 'Foz Monapo',
                'codigo' => 'H-301',
                'tipo' => 'hidrometrica',
                'bacia_hidrografica_id' => 3,
                'provincia' => 'Nampula',
                'distrito' => 'Monapo',
                'latitude' => -15.0333,
                'longitude' => 40.2500,
                'altitude' => 15,
                'nivel_alerta' => 3.2,
                'estado' => 'activa',
                'ativa' => true,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        DB::table('estacoes')->insert($pluviometricas);
        DB::table('estacoes')->insert($hidrometricas);
    }

    private function inserirCurvasVolume(): void
    {
        // Curvas para Barragem Nacala
        $curvas_nacala = [
            ['barragem_id' => 1, 'cota' => 112.0, 'volume' => 5.2, 'area' => 1.2],
            ['barragem_id' => 1, 'cota' => 115.0, 'volume' => 12.0, 'area' => 2.1],
            ['barragem_id' => 1, 'cota' => 118.0, 'volume' => 22.0, 'area' => 3.2],
            ['barragem_id' => 1, 'cota' => 121.0, 'volume' => 32.0, 'area' => 4.1],
            ['barragem_id' => 1, 'cota' => 124.0, 'volume' => 40.0, 'area' => 4.8],
            ['barragem_id' => 1, 'cota' => 127.0, 'volume' => 44.5, 'area' => 5.2],
            ['barragem_id' => 1, 'cota' => 128.5, 'volume' => 45.5, 'area' => 5.4],
        ];

        // Curvas para Barragem Chipembe
        $curvas_chipembe = [
            ['barragem_id' => 2, 'cota' => 320.0, 'volume' => 15.0, 'area' => 2.5],
            ['barragem_id' => 2, 'cota' => 325.0, 'volume' => 35.0, 'area' => 4.0],
            ['barragem_id' => 2, 'cota' => 330.0, 'volume' => 58.0, 'area' => 5.5],
            ['barragem_id' => 2, 'cota' => 335.0, 'volume' => 82.0, 'area' => 7.0],
            ['barragem_id' => 2, 'cota' => 340.0, 'volume' => 105.0, 'area' => 8.2],
            ['barragem_id' => 2, 'cota' => 345.0, 'volume' => 120.0, 'area' => 9.0],
        ];

        // Curvas para Barragem Montepuez
        $curvas_montepuez = [
            ['barragem_id' => 3, 'cota' => 235.0, 'volume' => 10.0, 'area' => 1.8],
            ['barragem_id' => 3, 'cota' => 240.0, 'volume' => 25.0, 'area' => 3.0],
            ['barragem_id' => 3, 'cota' => 245.0, 'volume' => 45.0, 'area' => 4.2],
            ['barragem_id' => 3, 'cota' => 250.0, 'volume' => 68.0, 'area' => 5.5],
            ['barragem_id' => 3, 'cota' => 254.0, 'volume' => 82.0, 'area' => 6.2],
            ['barragem_id' => 3, 'cota' => 256.0, 'volume' => 85.0, 'area' => 6.5],
        ];

        foreach (array_merge($curvas_nacala, $curvas_chipembe, $curvas_montepuez) as $curva) {
            $curva['criado_por'] = 1;
            $curva['created_at'] = now();
            $curva['updated_at'] = now();
            DB::table('curvas_volume')->insert($curva);
        }
    }

    private function inserirLeiturasBarragens(): void
    {
        $hoje = Carbon::today();

        // Leituras para os últimos 7 dias
        for ($i = 6; $i >= 0; $i--) {
            $data = $hoje->copy()->subDays($i);

            // Barragem Nacala - nível estável
            DB::table('leituras_barragens')->insert([
                'barragem_id' => 1,
                'data_leitura' => $data,
                'hora_leitura' => '08:00:00',
                'cota_actual' => 122.5 + rand(-10, 10) / 100,
                'volume_actual' => 35.2 + rand(-50, 50) / 100,
                'percentagem_enchimento' => 74.5 + rand(-20, 20) / 10,
                'caudal_afluente' => 2.5 + rand(0, 10) / 10,
                'caudal_efluente' => 1.8 + rand(0, 5) / 10,
                'caudal_ecologico' => 0.5,
                'descarga_fundo' => 0.3,
                'descarga_superficie' => 0.0,
                'descarga_total' => 0.3,
                'precipitacao' => $i < 3 ? rand(0, 15) : 0,
                'metodo_leitura' => 'manual',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Barragem Chipembe - nível mais alto
            DB::table('leituras_barragens')->insert([
                'barragem_id' => 2,
                'data_leitura' => $data,
                'hora_leitura' => '08:00:00',
                'cota_actual' => 336.8 + rand(-20, 20) / 100,
                'volume_actual' => 88.5 + rand(-100, 100) / 100,
                'percentagem_enchimento' => 70.0 + rand(-30, 30) / 10,
                'caudal_afluente' => 5.2 + rand(0, 20) / 10,
                'caudal_efluente' => 4.5 + rand(0, 15) / 10,
                'caudal_ecologico' => 1.2,
                'descarga_fundo' => 0.8,
                'descarga_superficie' => 0.5,
                'descarga_total' => 1.3,
                'precipitacao' => $i < 3 ? rand(5, 25) : 0,
                'metodo_leitura' => 'manual',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Barragem Montepuez
            DB::table('leituras_barragens')->insert([
                'barragem_id' => 3,
                'data_leitura' => $data,
                'hora_leitura' => '08:00:00',
                'cota_actual' => 247.2 + rand(-15, 15) / 100,
                'volume_actual' => 52.3 + rand(-80, 80) / 100,
                'percentagem_enchimento' => 56.4 + rand(-25, 25) / 10,
                'caudal_afluente' => 3.1 + rand(0, 15) / 10,
                'caudal_efluente' => 2.8 + rand(0, 10) / 10,
                'caudal_ecologico' => 0.8,
                'descarga_fundo' => 0.5,
                'descarga_superficie' => 0.0,
                'descarga_total' => 0.5,
                'precipitacao' => $i < 3 ? rand(0, 20) : 0,
                'metodo_leitura' => 'manual',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }

    private function inserirLeiturasEstacoes(): void
    {
        $hoje = Carbon::today();

        // Leituras de estações pluviométricas para hoje (com precipitação)
        // P-201 Ribáuè - com precipitação
        DB::table('leituras_estacoes')->insert([
            'estacao_id' => 1,
            'data_leitura' => $hoje,
            'precipitacao_mm' => 2.6,
            'observacoes' => 'Chuva ligeira durante a madrugada',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // P-1156 Malema - com precipitação
        DB::table('leituras_estacoes')->insert([
            'estacao_id' => 2,
            'data_leitura' => $hoje,
            'precipitacao_mm' => 1.2,
            'observacoes' => 'Precipitação fraca',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // P-302 Pemba - sem precipitação
        DB::table('leituras_estacoes')->insert([
            'estacao_id' => 3,
            'data_leitura' => $hoje,
            'precipitacao_mm' => 0.0,
            'observacoes' => 'Tempo seco',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // P-405 Monapo - com precipitação
        DB::table('leituras_estacoes')->insert([
            'estacao_id' => 4,
            'data_leitura' => $hoje,
            'precipitacao_mm' => 3.8,
            'observacoes' => 'Chuva moderada',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Leituras de estações hidrométricas para os últimos 3 dias
        for ($i = 2; $i >= 0; $i--) {
            $data = $hoje->copy()->subDays($i);

            // H-101 Ponte Lúrio
            DB::table('leituras_estacoes')->insert([
                'estacao_id' => 5,
                'data_leitura' => $data,
                'nivel_6h' => 2.35 + rand(-10, 10) / 100,
                'nivel_12h' => 2.40 + rand(-10, 10) / 100,
                'nivel_18h' => 2.38 + rand(-10, 10) / 100,
                'nivel_hidrometrico' => 2.38,
                'precipitacao_mm' => $i == 0 ? 1.5 : 0,
                'observacoes' => 'Nível estável',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // H-201 Estação Messalo
            DB::table('leituras_estacoes')->insert([
                'estacao_id' => 6,
                'data_leitura' => $data,
                'nivel_6h' => 1.85 + rand(-15, 15) / 100,
                'nivel_12h' => 1.90 + rand(-15, 15) / 100,
                'nivel_18h' => 1.88 + rand(-15, 15) / 100,
                'nivel_hidrometrico' => 1.88,
                'precipitacao_mm' => 0,
                'observacoes' => 'Sem alterações significativas',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // H-301 Foz Monapo
            DB::table('leituras_estacoes')->insert([
                'estacao_id' => 7,
                'data_leitura' => $data,
                'nivel_6h' => 1.52 + rand(-8, 8) / 100,
                'nivel_12h' => 1.55 + rand(-8, 8) / 100,
                'nivel_18h' => 1.53 + rand(-8, 8) / 100,
                'nivel_hidrometrico' => 1.53,
                'precipitacao_mm' => $i == 0 ? 2.0 : 0,
                'observacoes' => 'Nível baixando lentamente',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }
}
