#!/usr/bin/env python3
"""
Análise detalhada e específica das barragens
"""

import pandas as pd
import sys
import os
from openpyxl import load_workbook
import warnings
warnings.filterwarnings('ignore')

def analyze_barragem_structure():
    """Analisa a estrutura das folhas das barragens"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print("ANÁLISE DA ESTRUTURA DAS BARRAGENS")
    print("="*80)

    try:
        wb = load_workbook(file_path, data_only=True)

        # Analisa uma barragem específica (NAMPULA) em detalhe
        sheet_name = 'NAMPULA'
        print(f"ANÁLISE DETALHADA DA BARRAGEM: {sheet_name}")
        print("-"*50)

        ws = wb[sheet_name]
        print(f"Dimensões da folha: {ws.max_row} linhas x {ws.max_column} colunas")

        # Analisa as primeiras 20 linhas e 30 colunas para entender a estrutura
        print("\nESTRUTURA DOS CABEÇALHOS E PRIMEIRAS LINHAS:")
        for row in range(1, 21):
            row_data = []
            for col in range(1, 31):
                cell = ws.cell(row=row, column=col)
                value = cell.value
                if value is not None:
                    value_str = str(value)
                    if len(value_str) > 25:
                        value_str = value_str[:25] + "..."
                    row_data.append(value_str)
                else:
                    row_data.append("")

            # Só mostra linhas que têm dados relevantes
            if any(data.strip() for data in row_data):
                non_empty_data = [data for data in row_data if data.strip()]
                if non_empty_data:
                    print(f"Linha {row:2d}: {' | '.join(non_empty_data[:8])}")

        # Procura por padrões específicos
        print(f"\nPROCURANDO PADRÕES DE DADOS ESPECÍFICOS:")

        # Procura por células com valores numéricos que podem ser percentagens
        percentage_cells = []
        volume_cells = []
        cota_cells = []

        for row in range(1, min(100, ws.max_row + 1)):
            for col in range(1, min(50, ws.max_column + 1)):
                cell = ws.cell(row=row, column=col)
                if cell.value:
                    value_str = str(cell.value).lower()
                    coordinate = cell.coordinate

                    # Identifica células com percentagens
                    if isinstance(cell.value, (int, float)):
                        if 0 < cell.value <= 100:
                            # Verifica se há indicadores de percentagem nas células próximas
                            nearby_text = ""
                            for r_offset in [-1, 0, 1]:
                                for c_offset in [-1, 0, 1]:
                                    nearby_cell = ws.cell(row=row+r_offset, column=col+c_offset)
                                    if nearby_cell.value:
                                        nearby_text += str(nearby_cell.value).lower() + " "

                            if any(keyword in nearby_text for keyword in ['%', 'perc', 'total', 'util']):
                                percentage_cells.append({
                                    'cell': coordinate,
                                    'value': cell.value,
                                    'context': nearby_text.strip()[:50]
                                })

                    # Identifica referências a volumes
                    if any(keyword in value_str for keyword in ['volume', 'mm³', 'm³']):
                        volume_cells.append({
                            'cell': coordinate,
                            'value': str(cell.value)[:50]
                        })

                    # Identifica referências a cotas
                    if any(keyword in value_str for keyword in ['cota', 'nivel', 'nme', 'npa']):
                        cota_cells.append({
                            'cell': coordinate,
                            'value': str(cell.value)[:50]
                        })

        if percentage_cells:
            print("Possíveis células de percentagem:")
            for cell_info in percentage_cells[:10]:
                print(f"  {cell_info['cell']}: {cell_info['value']} (contexto: {cell_info['context']})")

        if volume_cells:
            print("\nCélulas relacionadas com volumes:")
            for cell_info in volume_cells[:10]:
                print(f"  {cell_info['cell']}: {cell_info['value']}")

        if cota_cells:
            print("\nCélulas relacionadas com cotas/níveis:")
            for cell_info in cota_cells[:10]:
                print(f"  {cell_info['cell']}: {cell_info['value']}")

    except Exception as e:
        print(f"Erro: {str(e)}")
        import traceback
        traceback.print_exc()

def analyze_formulas():
    """Analisa as fórmulas no ficheiro"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print(f"\n{'='*80}")
    print("ANÁLISE DAS FÓRMULAS")
    print(f"{'='*80}")

    try:
        # Carrega sem data_only para ver as fórmulas
        wb = load_workbook(file_path, data_only=False)

        for sheet_name in ['NAMPULA', 'NACALA']:  # Analisa 2 barragens
            if sheet_name in wb.sheetnames:
                print(f"\nFÓRMULAS NA BARRAGEM: {sheet_name}")
                print("-"*50)

                ws = wb[sheet_name]
                formulas_found = []

                # Procura fórmulas nas primeiras 50 linhas e 50 colunas
                for row in range(1, min(51, ws.max_row + 1)):
                    for col in range(1, min(51, ws.max_column + 1)):
                        cell = ws.cell(row=row, column=col)
                        if cell.data_type == 'f' and cell.value:
                            formulas_found.append({
                                'cell': cell.coordinate,
                                'formula': str(cell.value)
                            })

                if formulas_found:
                    print(f"Encontradas {len(formulas_found)} fórmulas")
                    print("Primeiras fórmulas encontradas:")
                    for formula in formulas_found[:15]:
                        print(f"  {formula['cell']}: {formula['formula']}")
                else:
                    print("Nenhuma fórmula encontrada nas primeiras 50x50 células")

    except Exception as e:
        print(f"Erro ao analisar fórmulas: {str(e)}")

def analyze_with_pandas():
    """Usa pandas para uma análise mais estruturada"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print(f"\n{'='*80}")
    print("ANÁLISE COM PANDAS")
    print(f"{'='*80}")

    try:
        # Tenta diferentes estratégias de leitura
        sheet_name = 'NAMPULA'

        print(f"Tentando ler {sheet_name} com diferentes parâmetros...")

        # Estratégia 1: Ler com header=None
        df1 = pd.read_excel(file_path, sheet_name=sheet_name, header=None, nrows=30, engine='openpyxl')
        print(f"Dimensões (header=None): {df1.shape}")

        # Estratégia 2: Ler pulando algumas linhas
        for skip in [0, 1, 2, 3]:
            try:
                df = pd.read_excel(file_path, sheet_name=sheet_name, skiprows=skip, nrows=20, engine='openpyxl')
                print(f"Com skiprows={skip}: {df.shape} - primeiras colunas: {list(df.columns[:5])}")

                # Procura por colunas com dados numéricos (possíveis percentagens)
                numeric_cols = df.select_dtypes(include=['number']).columns
                if len(numeric_cols) > 0:
                    print(f"  Colunas numéricas: {list(numeric_cols[:5])}")

                    # Verifica se há valores entre 0 e 100 (possíveis percentagens)
                    for col in numeric_cols:
                        values = df[col].dropna()
                        if len(values) > 0:
                            percentage_like = values[(values >= 0) & (values <= 100)]
                            if len(percentage_like) > 0:
                                print(f"    {col}: {len(percentage_like)} valores entre 0-100")

                break
            except Exception as e:
                continue

    except Exception as e:
        print(f"Erro na análise com pandas: {str(e)}")

if __name__ == "__main__":
    analyze_barragem_structure()
    analyze_formulas()
    analyze_with_pandas()