@extends('layouts.app')

@section('title', 'Matriz de Permissões')

@push('styles')
<style>
    .permission-cell {
        cursor: pointer;
        transition: all 0.2s;
    }
    .permission-cell:hover {
        transform: scale(1.1);
    }
    .permission-cell.granted {
        background-color: #10B981;
        color: white;
    }
    .permission-cell.denied {
        background-color: #EF4444;
        color: white;
    }
    .permission-cell.empty {
        background-color: #F3F4F6;
        color: #9CA3AF;
    }
    .sticky-header th {
        position: sticky;
        top: 0;
        z-index: 10;
    }
    .sticky-col {
        position: sticky;
        left: 0;
        z-index: 5;
        background: white;
    }
</style>
@endpush

@section('content')
<div class="py-6">
    <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
        <!-- Header -->
        <div class="flex justify-between items-center mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">Matriz de Permissões</h1>
                <p class="text-sm text-gray-600 mt-1">Clique nas células para conceder ou revogar permissões</p>
            </div>
            <div class="flex gap-2">
                <a href="{{ route('admin.roles.index') }}" class="bg-gray-600 hover:bg-gray-700 text-white font-medium py-2 px-4 rounded-lg flex items-center gap-2">
                    <i class="fas fa-arrow-left"></i>
                    Voltar
                </a>
            </div>
        </div>

        <!-- Legenda -->
        <div class="mb-4 flex gap-4 text-sm">
            <span class="flex items-center gap-2">
                <span class="w-6 h-6 rounded bg-green-500 flex items-center justify-center text-white text-xs">
                    <i class="fas fa-check"></i>
                </span>
                Permissao Concedida
            </span>
            <span class="flex items-center gap-2">
                <span class="w-6 h-6 rounded bg-gray-200 flex items-center justify-center text-gray-400 text-xs">
                    <i class="fas fa-times"></i>
                </span>
                Sem Permissao
            </span>
            <span class="flex items-center gap-2">
                <span class="w-6 h-6 rounded bg-red-100 border border-red-300 flex items-center justify-center text-red-500 text-xs">
                    <i class="fas fa-lock"></i>
                </span>
                Protegido (Admin)
            </span>
        </div>

        <!-- Matriz -->
        <div class="bg-white rounded-lg shadow overflow-auto max-h-[70vh]">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50 sticky-header">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider sticky-col bg-gray-50">
                            Permissao
                        </th>
                        @foreach($roles as $role)
                        <th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider min-w-[100px]
                            @if($role->name == 'Administrador') bg-red-50 @endif">
                            {{ $role->name }}
                        </th>
                        @endforeach
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @foreach($permissoesPorCategoria as $categoria => $perms)
                    <!-- Categoria Header -->
                    <tr class="bg-blue-50">
                        <td colspan="{{ $roles->count() + 1 }}" class="px-4 py-2 text-sm font-semibold text-blue-800 capitalize sticky-col bg-blue-50">
                            <i class="fas fa-folder-open mr-2"></i>{{ $categoria }}
                        </td>
                    </tr>
                    @foreach($perms as $permission)
                    <tr class="hover:bg-gray-50">
                        <td class="px-4 py-2 text-sm text-gray-700 sticky-col">
                            {{ $permission->name }}
                        </td>
                        @foreach($roles as $role)
                        <td class="px-4 py-2 text-center">
                            @if($role->name == 'Administrador')
                            <span class="inline-flex items-center justify-center w-8 h-8 rounded bg-red-100 border border-red-300 text-red-500" title="Admin tem todas as permissoes">
                                <i class="fas fa-crown text-xs"></i>
                            </span>
                            @else
                            <button type="button"
                                    class="permission-cell inline-flex items-center justify-center w-8 h-8 rounded {{ $role->hasPermissionTo($permission->name) ? 'granted' : 'empty' }}"
                                    data-role-id="{{ $role->id }}"
                                    data-permission-id="{{ $permission->id }}"
                                    data-permission-name="{{ $permission->name }}"
                                    data-role-name="{{ $role->name }}"
                                    data-has-permission="{{ $role->hasPermissionTo($permission->name) ? '1' : '0' }}"
                                    onclick="togglePermission(this)">
                                <i class="fas {{ $role->hasPermissionTo($permission->name) ? 'fa-check' : 'fa-times' }} text-xs"></i>
                            </button>
                            @endif
                        </td>
                        @endforeach
                    </tr>
                    @endforeach
                    @endforeach
                </tbody>
            </table>
        </div>

        <!-- Status -->
        <div id="status-message" class="hidden fixed bottom-4 right-4 px-4 py-2 rounded-lg shadow-lg text-white text-sm">
        </div>
    </div>
</div>

@push('scripts')
<script>
function togglePermission(button) {
    const roleId = button.dataset.roleId;
    const permissionId = button.dataset.permissionId;
    const permissionName = button.dataset.permissionName;
    const roleName = button.dataset.roleName;
    const hasPermission = button.dataset.hasPermission === '1';

    const action = hasPermission ? 'revoke' : 'grant';

    // Mostrar loading
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin text-xs"></i>';

    fetch(`{{ url('/admin/roles') }}/${roleId}/toggle-permission`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            permission_id: permissionId,
            action: action
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Actualizar visual
            if (action === 'grant') {
                button.classList.remove('empty');
                button.classList.add('granted');
                button.dataset.hasPermission = '1';
                button.innerHTML = '<i class="fas fa-check text-xs"></i>';
            } else {
                button.classList.remove('granted');
                button.classList.add('empty');
                button.dataset.hasPermission = '0';
                button.innerHTML = '<i class="fas fa-times text-xs"></i>';
            }

            showStatus(data.message, 'success');
        } else {
            showStatus(data.error || 'Erro ao actualizar permissao', 'error');
            // Restaurar estado
            button.innerHTML = hasPermission
                ? '<i class="fas fa-check text-xs"></i>'
                : '<i class="fas fa-times text-xs"></i>';
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showStatus('Erro ao actualizar permissao', 'error');
        button.innerHTML = hasPermission
            ? '<i class="fas fa-check text-xs"></i>'
            : '<i class="fas fa-times text-xs"></i>';
    })
    .finally(() => {
        button.disabled = false;
    });
}

function showStatus(message, type) {
    const statusEl = document.getElementById('status-message');
    statusEl.textContent = message;
    statusEl.className = `fixed bottom-4 right-4 px-4 py-2 rounded-lg shadow-lg text-white text-sm ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;

    setTimeout(() => {
        statusEl.classList.add('hidden');
    }, 3000);
}
</script>
@endpush
@endsection
