@extends('layouts.app')

@section('title', 'Leituras Grid - ' . $barragem->nome)

@push('styles')
<style>
    /* Grid Excel-like - Enhanced & Larger */
    .excel-grid {
        width: 100%;
        overflow-x: auto;
        font-size: 14px;
        max-height: calc(100vh - 350px);
        overflow-y: auto;
        border-radius: 8px;
    }

    .excel-grid table {
        border-collapse: collapse;
        background: white;
        width: 100%;
    }

    .excel-grid thead {
        position: sticky;
        top: 0;
        z-index: 20;
    }

    .excel-grid th {
        background: linear-gradient(135deg, #1e40af 0%, #1e3a8a 100%);
        color: white;
        padding: 14px 12px;
        font-weight: 600;
        text-align: center;
        border: 1px solid #1e3a8a;
        font-size: 13px;
        line-height: 1.4;
    }

    .excel-grid td {
        border: 1px solid #d1d5db;
        padding: 0;
        height: 48px;
    }

    .excel-grid input {
        width: 100%;
        height: 48px;
        border: none;
        padding: 8px 12px;
        font-size: 15px;
        background: transparent;
        text-align: right;
    }

    .excel-grid input:focus {
        outline: 3px solid #3b82f6;
        outline-offset: -3px;
        background: #eff6ff;
        z-index: 5;
        position: relative;
    }

    .excel-grid input.date-input {
        text-align: center;
        font-weight: 600;
        color: #1e40af;
        font-size: 14px;
    }

    .excel-grid input.time-input {
        text-align: center;
        font-size: 14px;
    }

    .excel-grid input[readonly] {
        background: #f8fafc;
        color: #374151;
        cursor: not-allowed;
        font-weight: 600;
    }

    .excel-grid .calculated-cell {
        background: linear-gradient(135deg, #fef9c3 0%, #fef3c7 100%);
    }

    .excel-grid .editable-cell {
        background: white;
    }

    .excel-grid tr:hover td {
        background: #e0f2fe;
    }

    .excel-grid tr:hover td.calculated-cell {
        background: #fde68a;
    }

    .excel-grid tr:hover td input[readonly] {
        background: #e0f2fe;
    }

    .excel-grid tr.has-data {
        background: #f0fdf4;
    }

    .excel-grid tr.modified {
        background: #fef9c3;
    }

    /* Row status indicators - Larger border */
    .excel-grid tr.nivel-critico td:first-child {
        border-left: 6px solid #ef4444;
    }

    .excel-grid tr.nivel-alerta td:first-child {
        border-left: 6px solid #f59e0b;
    }

    .excel-grid tr.nivel-normal td:first-child {
        border-left: 6px solid #22c55e;
    }

    /* Column widths - Larger */
    .col-data { width: 130px; min-width: 130px; }
    .col-hora { width: 90px; min-width: 90px; }
    .col-cota { width: 120px; min-width: 120px; }
    .col-volume { width: 120px; min-width: 120px; }
    .col-percent { width: 110px; min-width: 110px; }
    .col-caudal { width: 110px; min-width: 110px; }
    .col-precip { width: 100px; min-width: 100px; }
    .col-obs { width: 180px; min-width: 180px; }

    .header-group {
        background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%) !important;
        font-size: 13px;
        padding: 10px 8px !important;
    }

    /* Alternate row colors */
    .excel-grid tbody tr:nth-child(even) {
        background-color: #f9fafb;
    }

    .excel-grid tbody tr:nth-child(even).has-data {
        background-color: #ecfdf5;
    }

    /* Stats Cards */
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
        gap: 16px;
        margin-bottom: 16px;
    }

    .stat-card {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 12px;
        padding: 16px;
        display: flex;
        align-items: center;
        gap: 12px;
        transition: all 0.2s;
    }

    .stat-card:hover {
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        transform: translateY(-2px);
    }

    .stat-icon {
        width: 48px;
        height: 48px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
    }

    .stat-icon.blue { background: #dbeafe; color: #1e40af; }
    .stat-icon.green { background: #dcfce7; color: #16a34a; }
    .stat-icon.yellow { background: #fef3c7; color: #d97706; }
    .stat-icon.purple { background: #f3e8ff; color: #7c3aed; }
    .stat-icon.red { background: #fee2e2; color: #dc2626; }

    .stat-content h4 {
        font-size: 12px;
        color: #6b7280;
        margin: 0;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .stat-content .value {
        font-size: 24px;
        font-weight: 700;
        color: #111827;
        line-height: 1.2;
    }

    .stat-content .subtext {
        font-size: 11px;
        color: #9ca3af;
    }

    /* Progress bar */
    .progress-bar {
        width: 100%;
        height: 8px;
        background: #e5e7eb;
        border-radius: 4px;
        overflow: hidden;
        margin-top: 8px;
    }

    .progress-bar .fill {
        height: 100%;
        border-radius: 4px;
        transition: width 0.3s ease;
    }

    .progress-bar .fill.green { background: linear-gradient(90deg, #22c55e, #16a34a); }
    .progress-bar .fill.yellow { background: linear-gradient(90deg, #eab308, #d97706); }
    .progress-bar .fill.red { background: linear-gradient(90deg, #ef4444, #dc2626); }

    /* Toolbar */
    .toolbar {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 12px;
        padding: 16px;
        margin-bottom: 16px;
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
        align-items: center;
        box-shadow: 0 1px 3px rgba(0,0,0,0.05);
    }

    .toolbar-section {
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .toolbar-divider {
        width: 1px;
        height: 32px;
        background: #e5e7eb;
        margin: 0 8px;
    }

    .btn-toolbar {
        padding: 10px 16px;
        border-radius: 8px;
        font-size: 13px;
        font-weight: 500;
        border: 1px solid #d1d5db;
        background: white;
        cursor: pointer;
        transition: all 0.2s;
        display: inline-flex;
        align-items: center;
        gap: 6px;
    }

    .btn-toolbar:hover {
        background: #f3f4f6;
        transform: translateY(-1px);
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }

    .btn-toolbar:active {
        transform: translateY(0);
    }

    .btn-primary {
        background: linear-gradient(135deg, #16a34a 0%, #15803d 100%);
        color: white;
        border-color: #16a34a;
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, #15803d 0%, #166534 100%);
    }

    .btn-danger {
        background: white;
        color: #dc2626;
        border-color: #fecaca;
    }

    .btn-danger:hover {
        background: #fee2e2;
    }

    .mode-toggle {
        display: flex;
        gap: 4px;
        background: #f3f4f6;
        padding: 4px;
        border-radius: 8px;
    }

    .mode-toggle button {
        padding: 8px 14px;
        border: none;
        background: transparent;
        border-radius: 6px;
        font-size: 12px;
        cursor: pointer;
        transition: all 0.2s;
        display: flex;
        align-items: center;
        gap: 6px;
    }

    .mode-toggle button.active {
        background: white;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        font-weight: 600;
        color: #1e40af;
    }

    .status-badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 8px 14px;
        border-radius: 8px;
        font-size: 12px;
        font-weight: 500;
    }

    .status-badge.editing {
        background: #fef3c7;
        color: #92400e;
    }

    .status-badge.saved {
        background: #dcfce7;
        color: #166534;
    }

    .status-badge.error {
        background: #fee2e2;
        color: #991b1b;
    }

    .status-dot {
        width: 8px;
        height: 8px;
        border-radius: 50%;
        animation: pulse 2s infinite;
    }

    .status-dot.editing { background: #eab308; }
    .status-dot.saved { background: #22c55e; }
    .status-dot.error { background: #ef4444; }

    @keyframes pulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.5; }
    }

    /* Period Selector */
    .period-selector {
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .period-selector label {
        font-size: 12px;
        color: #6b7280;
        font-weight: 500;
    }

    .period-selector select,
    .period-selector input[type="date"] {
        padding: 8px 12px;
        border: 1px solid #d1d5db;
        border-radius: 6px;
        font-size: 13px;
        background: white;
    }

    .period-selector input[type="date"] {
        width: 140px;
    }

    /* Mini Chart */
    .mini-chart-container {
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 12px;
        padding: 16px;
        margin-bottom: 16px;
    }

    .mini-chart-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 12px;
    }

    .mini-chart-header h3 {
        font-size: 14px;
        font-weight: 600;
        color: #374151;
        margin: 0;
    }

    /* Reference levels */
    .reference-levels {
        display: flex;
        gap: 16px;
        font-size: 11px;
        margin-bottom: 12px;
    }

    .ref-level {
        display: flex;
        align-items: center;
        gap: 4px;
    }

    .ref-line {
        width: 20px;
        height: 3px;
        border-radius: 2px;
    }

    .ref-line.npa { background: #22c55e; }
    .ref-line.nme { background: #ef4444; }
    .ref-line.atual { background: #3b82f6; }

    /* Help panel */
    .help-panel {
        background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%);
        border: 1px solid #bfdbfe;
        border-radius: 12px;
        padding: 16px;
    }

    .help-panel h4 {
        font-size: 14px;
        font-weight: 600;
        color: #1e40af;
        margin: 0 0 12px 0;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .help-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 12px;
    }

    .help-item {
        display: flex;
        align-items: flex-start;
        gap: 8px;
        font-size: 12px;
        color: #1e40af;
    }

    .help-item kbd {
        background: white;
        border: 1px solid #bfdbfe;
        border-radius: 4px;
        padding: 2px 6px;
        font-family: monospace;
        font-size: 11px;
        white-space: nowrap;
    }

    .help-item .color-box {
        width: 16px;
        height: 16px;
        border-radius: 3px;
        flex-shrink: 0;
    }

    /* Estilo para linha de hoje */
    .excel-grid tr.hoje-row {
        background: #fef9c3 !important;
        box-shadow: inset 0 0 0 3px #eab308;
    }

    .excel-grid tr.hoje-row td {
        background: #fef9c3 !important;
    }

    .excel-grid tr.hoje-row:hover td {
        background: #fef08a !important;
    }

    .excel-grid tr.hoje-row td:first-child {
        position: relative;
    }
</style>
@endpush

@section('content')
@php
    // Verificar se barragem usa nivel hidrometrico
    $temEscalaNpa = $barragem->escala_npa && $barragem->cota_npa;

    $inicio = request('data_inicio') ? \Carbon\Carbon::parse(request('data_inicio')) : now()->startOfMonth();
    $fim = request('data_fim') ? \Carbon\Carbon::parse(request('data_fim')) : now()->endOfMonth();
    $totalDias = $inicio->diffInDays($fim) + 1;

    // Contar leituras existentes no período
    $leiturasExistentes = $barragem->leituras()
        ->whereBetween('data_leitura', [$inicio, $fim])
        ->get();
    $diasComLeitura = $leiturasExistentes->count();
    $percentualPreenchido = $totalDias > 0 ? round(($diasComLeitura / $totalDias) * 100) : 0;

    // Última leitura
    $ultimaLeitura = $leiturasExistentes->sortByDesc('data_leitura')->first();

    // Médias do período
    $cotaMedia = $leiturasExistentes->avg('cota_actual');
    $volumeMedia = $leiturasExistentes->avg('volume_actual');
    $enchimentoMedia = $leiturasExistentes->avg('percentagem_enchimento');
@endphp

<div class="space-y-4">
    <!-- Header Compacto -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200">
        <div class="px-6 py-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-4">
                    <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-xl flex items-center justify-center">
                        <i class="fas fa-table text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-xl font-bold text-gray-900">Leituras - Modo Grid</h1>
                        <p class="text-sm text-gray-500">
                            <a href="{{ route('barragens.show', $barragem) }}" class="text-blue-600 hover:underline">{{ $barragem->nome }}</a>
                            <span class="mx-2">•</span>
                            <span class="text-gray-400">{{ $barragem->codigo }}</span>
                            <span class="mx-2">•</span>
                            <span>{{ $inicio->format('d/m/Y') }} - {{ $fim->format('d/m/Y') }}</span>
                        </p>
                    </div>
                </div>
                <div class="flex items-center gap-3">
                    <a href="{{ route('barragens.leituras', $barragem) }}" class="btn-toolbar">
                        <i class="fas fa-list"></i> Modo Lista
                    </a>
                    <a href="{{ route('barragens.show', $barragem) }}" class="btn-toolbar">
                        <i class="fas fa-arrow-left"></i> Voltar
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon blue">
                <i class="fas fa-calendar-check"></i>
            </div>
            <div class="stat-content">
                <h4>Preenchimento</h4>
                <div class="value">{{ $diasComLeitura }}/{{ $totalDias }}</div>
                <div class="progress-bar">
                    <div class="fill {{ $percentualPreenchido >= 80 ? 'green' : ($percentualPreenchido >= 50 ? 'yellow' : 'red') }}"
                         style="width: {{ $percentualPreenchido }}%"></div>
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon green">
                <i class="fas fa-water"></i>
            </div>
            <div class="stat-content">
                <h4>Cota Média</h4>
                <div class="value">{{ $cotaMedia ? number_format($cotaMedia, 2) : '--' }} <small style="font-size: 14px;">m</small></div>
                <div class="subtext">
                    NPA: {{ $barragem->cota_npa ?? '--' }}m • NME: {{ $barragem->cota_nme ?? '--' }}m
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon purple">
                <i class="fas fa-chart-pie"></i>
            </div>
            <div class="stat-content">
                <h4>Enchimento Médio</h4>
                <div class="value">{{ $enchimentoMedia ? number_format($enchimentoMedia, 1) : '--' }}<small style="font-size: 14px;">%</small></div>
                <div class="subtext">
                    @if($ultimaLeitura)
                        Último: {{ number_format($ultimaLeitura->percentagem_enchimento ?? 0, 1) }}%
                    @else
                        Sem dados
                    @endif
                </div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon yellow">
                <i class="fas fa-edit"></i>
            </div>
            <div class="stat-content">
                <h4>Modificações</h4>
                <div class="value" id="count-modificadas">0</div>
                <div class="subtext">linhas pendentes de salvar</div>
            </div>
        </div>
    </div>

    <!-- Mini Chart -->
    @if($leiturasExistentes->count() >= 2)
    <div class="mini-chart-container">
        <div class="mini-chart-header">
            <h3><i class="fas fa-chart-line mr-2"></i>Tendência do Período</h3>
            <div class="reference-levels" style="margin: 0;">
                <div class="ref-level">
                    <div class="ref-line atual"></div>
                    <span>Cota</span>
                </div>
                <div class="ref-level">
                    <div class="ref-line npa"></div>
                    <span>NPA</span>
                </div>
                <div class="ref-level">
                    <div class="ref-line nme"></div>
                    <span>NME</span>
                </div>
            </div>
        </div>
        <div style="width: 100%; height: 120px; position: relative;">
            @php
                $cotas = $leiturasExistentes->sortBy('data_leitura')->pluck('cota_actual')->filter()->values();
                $minCota = $cotas->min() ?? 0;
                $maxCota = $cotas->max() ?? 100;
                $range = max($maxCota - $minCota, 1);

                // Ajustar para incluir NPA e NME no range
                if ($barragem->cota_npa) $maxCota = max($maxCota, $barragem->cota_npa);
                if ($barragem->cota_nme) $minCota = min($minCota, $barragem->cota_nme);
                $range = max($maxCota - $minCota, 1);

                $width = 100;
                $height = 100;
                $points = [];
                $count = $cotas->count();

                foreach ($cotas as $i => $cota) {
                    $x = $count > 1 ? ($i / ($count - 1)) * $width : 50;
                    $y = $height - (($cota - $minCota) / $range) * $height;
                    $points[] = round($x, 1) . ',' . round($y, 1);
                }

                // NPA e NME lines
                $npaPct = $barragem->cota_npa ? $height - (($barragem->cota_npa - $minCota) / $range) * $height : null;
                $nmePct = $barragem->cota_nme ? $height - (($barragem->cota_nme - $minCota) / $range) * $height : null;
            @endphp

            <svg viewBox="0 0 100 100" preserveAspectRatio="none" style="width: 100%; height: 100%;">
                <!-- Grid lines -->
                <line x1="0" y1="25" x2="100" y2="25" stroke="#e5e7eb" stroke-width="0.5" />
                <line x1="0" y1="50" x2="100" y2="50" stroke="#e5e7eb" stroke-width="0.5" />
                <line x1="0" y1="75" x2="100" y2="75" stroke="#e5e7eb" stroke-width="0.5" />

                <!-- NPA Line -->
                @if($npaPct !== null)
                <line x1="0" y1="{{ $npaPct }}" x2="100" y2="{{ $npaPct }}"
                      stroke="#22c55e" stroke-width="1" stroke-dasharray="4,2" />
                @endif

                <!-- NME Line -->
                @if($nmePct !== null)
                <line x1="0" y1="{{ $nmePct }}" x2="100" y2="{{ $nmePct }}"
                      stroke="#ef4444" stroke-width="1" stroke-dasharray="4,2" />
                @endif

                <!-- Cota Line -->
                @if(count($points) > 1)
                <polyline points="{{ implode(' ', $points) }}"
                          fill="none" stroke="#3b82f6" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />

                <!-- Fill area -->
                <polygon points="0,100 {{ implode(' ', $points) }} 100,100"
                         fill="url(#gradient)" opacity="0.3" />
                @endif

                <defs>
                    <linearGradient id="gradient" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="0%" stop-color="#3b82f6" />
                        <stop offset="100%" stop-color="#3b82f6" stop-opacity="0" />
                    </linearGradient>
                </defs>
            </svg>

            <!-- Y-axis labels -->
            <div style="position: absolute; left: 0; top: 0; font-size: 10px; color: #6b7280;">
                {{ number_format($maxCota, 1) }}m
            </div>
            <div style="position: absolute; left: 0; bottom: 0; font-size: 10px; color: #6b7280;">
                {{ number_format($minCota, 1) }}m
            </div>
        </div>
    </div>
    @endif

    <!-- Reference Levels -->
    <div class="reference-levels bg-white rounded-lg px-4 py-3 border border-gray-200" style="display: flex; flex-wrap: wrap; align-items: center;">
        <span class="text-gray-600 font-medium text-sm mr-4">Parâmetros:</span>
        @if($temEscalaNpa)
        <div class="ref-level" style="background: #dcfce7; padding: 2px 8px; border-radius: 4px; margin-right: 8px;">
            <span style="color: #166534;"><i class="fas fa-ruler-vertical mr-1"></i>Escala NPA: <strong>{{ number_format($barragem->escala_npa, 2) }}m</strong></span>
        </div>
        @endif
        <div class="ref-level">
            <div class="ref-line npa"></div>
            <span>NPA: <strong>{{ $barragem->cota_npa ?? '--' }}m</strong></span>
        </div>
        <div class="ref-level">
            <div class="ref-line nme"></div>
            <span>NME: <strong>{{ $barragem->cota_nme ?? '--' }}m</strong></span>
        </div>
        <div class="ref-level">
            <div class="ref-line atual"></div>
            <span>Vol. Máx: <strong>{{ $barragem->volume_maximo ? number_format((float)$barragem->volume_maximo, 2) . ' Mm³' : '--' }}</strong></span>
        </div>
        <div class="ref-level">
            <span style="color: #6b7280;">Vol. Morto: <strong>{{ $barragem->volume_morto ? number_format((float)$barragem->volume_morto, 2) . ' Mm³' : '--' }}</strong></span>
        </div>
    </div>

    <!-- Toolbar -->
    <div class="toolbar">
        <!-- Modo de Visualização -->
        <div class="toolbar-section">
            <div class="mode-toggle">
                <button type="button" class="active" data-mode="simples" onclick="toggleMode('simples')">
                    <i class="fas fa-compress-alt"></i> Simples
                </button>
                <button type="button" data-mode="completo" onclick="toggleMode('completo')">
                    <i class="fas fa-expand-alt"></i> Completo
                </button>
            </div>
        </div>

        <div class="toolbar-divider"></div>

        <!-- Período -->
        <div class="toolbar-section period-selector">
            <label>Período:</label>
            <select id="periodo-select" onchange="carregarPeriodo()">
                <option value="mes-atual" {{ !request('data_inicio') ? 'selected' : '' }}>Mês Atual</option>
                <option value="mes-anterior">Mês Anterior</option>
                <option value="ano-hidrologico">Ano Hidrológico</option>
                <option value="personalizado">Personalizado</option>
            </select>
            <input type="date" id="data-inicio" value="{{ $inicio->format('Y-m-d') }}" onchange="carregarPeriodoPersonalizado()">
            <span style="color: #9ca3af;">a</span>
            <input type="date" id="data-fim" value="{{ $fim->format('Y-m-d') }}" onchange="carregarPeriodoPersonalizado()">
        </div>

        <div style="flex: 1;"></div>

        <!-- Ações -->
        <div class="toolbar-section">
            <button type="button" class="btn-toolbar" onclick="copiarDeExcel()" title="Colar dados copiados do Excel">
                <i class="fas fa-paste"></i> Colar Excel
            </button>

            <button type="button" class="btn-toolbar" onclick="preencherAutomatico()" title="Preencher automaticamente linhas vazias">
                <i class="fas fa-magic"></i> Auto
            </button>

            <button type="button" class="btn-toolbar" onclick="limparLinhasVazias()" title="Limpar linhas sem dados">
                <i class="fas fa-eraser"></i> Limpar
            </button>

            <button type="button" class="btn-toolbar" onclick="exportarExcel()" title="Exportar para Excel">
                <i class="fas fa-file-excel"></i> Exportar
            </button>
        </div>

        <div class="toolbar-divider"></div>

        <button type="button" class="btn-toolbar btn-primary" onclick="salvarTodasLeituras()" id="btn-salvar">
            <i class="fas fa-save"></i> Salvar Alterações
        </button>

        <!-- Status -->
        <div class="status-badge editing" id="status-badge">
            <span class="status-dot editing"></span>
            <span id="status-text">Pronto</span>
        </div>
    </div>

    <!-- Grid Excel -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div class="excel-grid" id="excel-grid-container">
            <table id="leituras-grid">
                <thead>
                    <tr>
                        <th class="col-data">Data</th>
                        <th class="col-hora">Hora</th>
                        @if($temEscalaNpa)
                        <th class="col-cota" style="background: linear-gradient(135deg, #16a34a 0%, #15803d 100%) !important;">
                            <i class="fas fa-ruler-vertical mr-1"></i>Nível (m)
                        </th>
                        <th class="col-cota modo-completo">Cota (m)</th>
                        @else
                        <th class="col-cota" style="background: linear-gradient(135deg, #16a34a 0%, #15803d 100%) !important;">
                            <i class="fas fa-water mr-1"></i>Cota (m)
                        </th>
                        @endif
                        <th class="col-volume">Volume (Mm³)</th>
                        <th class="col-percent">% Ench.</th>
                        <th class="col-percent">% Total</th>
                        <th class="col-volume modo-completo">Variação</th>
                        <th class="col-caudal modo-completo">Afluente</th>
                        <th class="col-caudal modo-completo">Efluente</th>
                        <th class="col-precip modo-completo">Precip.</th>
                        <th class="col-obs modo-completo">Obs.</th>
                    </tr>
                </thead>
                <tbody id="grid-body">
                    @for($data = $inicio->copy(); $data->lte($fim); $data->addDay())
                        @php
                            $leitura = $barragem->leituras()
                                ->whereDate('data_leitura', $data)
                                ->first();

                            // Determinar classe de status baseado na percentagem de enchimento
                            $statusClass = '';
                            $hasData = $leitura && ($leitura->cota_actual || $leitura->nivel_hidrometrico);
                            if ($hasData && $leitura->percentagem_enchimento !== null) {
                                if ($leitura->percentagem_enchimento >= 90) {
                                    $statusClass = 'nivel-critico';
                                } elseif ($leitura->percentagem_enchimento >= 70) {
                                    $statusClass = 'nivel-alerta';
                                } else {
                                    $statusClass = 'nivel-normal';
                                }
                            }
                        @endphp
                        @php
                            $isHoje = $data->isToday();
                        @endphp
                        <tr data-date="{{ $data->format('Y-m-d') }}"
                            data-id="{{ $leitura?->id ?? '' }}"
                            {{ $isHoje ? 'id=linha-hoje' : '' }}
                            class="{{ $hasData ? 'has-data ' . $statusClass : '' }} {{ $isHoje ? 'hoje-row' : '' }}"
                            style="{{ $isHoje ? 'background: #fef9c3 !important; box-shadow: inset 0 0 0 3px #eab308;' : '' }}">
                            <!-- Data -->
                            <td style="{{ $isHoje ? 'background: #fef08a; position: relative;' : '' }}">
                                <input type="text"
                                       class="date-input"
                                       value="{{ $data->format('d/m') }} {{ $data->locale('pt')->isoFormat('ddd') }}"
                                       data-date="{{ $data->format('Y-m-d') }}"
                                       readonly
                                       style="{{ $isHoje ? 'background: #fef08a; font-weight: 700; color: #854d0e;' : '' }}"
                                       title="{{ $data->format('d/m/Y') }} - {{ $data->locale('pt')->isoFormat('dddd') }}">
                                @if($isHoje)
                                <span style="position: absolute; top: 50%; right: 6px; transform: translateY(-50%); background: #eab308; color: #422006; font-size: 9px; padding: 2px 6px; border-radius: 4px; font-weight: 700; animation: pulse 2s infinite;">HOJE</span>
                                @endif
                            </td>

                            <!-- Hora -->
                            <td class="editable-cell">
                                @php
                                    $horaGrid = $leitura && $leitura->hora_leitura ? \Carbon\Carbon::parse($leitura->hora_leitura)->format('H:i') : '06:00';
                                @endphp
                                <select class="time-input" name="hora_leitura[]" onkeydown="handleKeyNavigation(event)">
                                    <option value="06:00" {{ $horaGrid == '06:00' ? 'selected' : '' }}>06:00h</option>
                                    <option value="12:00" {{ $horaGrid == '12:00' ? 'selected' : '' }}>12:00h</option>
                                    <option value="18:00" {{ $horaGrid == '18:00' ? 'selected' : '' }}>18:00h</option>
                                </select>
                            </td>

                            @if($temEscalaNpa)
                            <!-- NIVEL HIDROMETRICO (EDITÁVEL - Campo Principal) -->
                            <td class="editable-cell" style="background: #f0fdf4;">
                                <input type="number"
                                       step="0.01"
                                       name="nivel_hidrometrico[]"
                                       value="{{ $leitura?->nivel_hidrometrico ?? '' }}"
                                       placeholder="0.00"
                                       style="font-weight: 700; color: #166534;"
                                       oninput="calcularLinha(this)"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>

                            <!-- Cota Calculada - Modo Completo -->
                            <td class="calculated-cell modo-completo">
                                <input type="number"
                                       step="0.01"
                                       name="cota_actual[]"
                                       value="{{ $leitura?->cota_actual ?? '' }}"
                                       readonly>
                            </td>
                            @else
                            <!-- COTA ATUAL (EDITÁVEL - Campo Principal) -->
                            <td class="editable-cell" style="background: #f0fdf4;">
                                <input type="number"
                                       step="0.01"
                                       name="cota_actual[]"
                                       value="{{ $leitura?->cota_actual ?? '' }}"
                                       placeholder="0.00"
                                       style="font-weight: 700; color: #166534;"
                                       oninput="calcularLinha(this)"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>
                            @endif

                            <!-- Volume Atual (CALCULADO) -->
                            <td class="calculated-cell">
                                <input type="number"
                                       step="0.01"
                                       name="volume_actual[]"
                                       value="{{ $leitura?->volume_actual ?? '' }}"
                                       readonly>
                            </td>

                            <!-- % Enchimento (CALCULADO) -->
                            <td class="calculated-cell">
                                <input type="number"
                                       step="0.01"
                                       name="percentagem_enchimento[]"
                                       value="{{ $leitura?->percentagem_enchimento ?? '' }}"
                                       readonly>
                            </td>

                            <!-- % Total (CALCULADO) -->
                            <td class="calculated-cell">
                                <input type="number"
                                       step="0.01"
                                       name="percentagem_total[]"
                                       value="{{ $leitura?->percentagem_total ?? '' }}"
                                       readonly>
                            </td>

                            <!-- Variação Volume (CALCULADO) - Modo Completo -->
                            <td class="calculated-cell modo-completo">
                                <input type="number"
                                       step="0.01"
                                       name="variacao_volume[]"
                                       value="{{ $leitura?->variacao_volume ?? '' }}"
                                       readonly>
                            </td>

                            <!-- Caudal Afluente - Modo Completo -->
                            <td class="editable-cell modo-completo">
                                <input type="number"
                                       step="0.01"
                                       name="caudal_afluente[]"
                                       value="{{ $leitura?->caudal_afluente ?? '' }}"
                                       placeholder="--"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>

                            <!-- Caudal Efluente - Modo Completo -->
                            <td class="editable-cell modo-completo">
                                <input type="number"
                                       step="0.01"
                                       name="caudal_efluente[]"
                                       value="{{ $leitura?->caudal_efluente ?? '' }}"
                                       placeholder="--"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>

                            <!-- Precipitação - Modo Completo -->
                            <td class="editable-cell modo-completo">
                                <input type="number"
                                       step="0.1"
                                       name="precipitacao[]"
                                       value="{{ $leitura?->precipitacao ?? '' }}"
                                       placeholder="--"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>

                            <!-- Observações - Modo Completo -->
                            <td class="editable-cell modo-completo">
                                <input type="text"
                                       name="observacoes[]"
                                       value="{{ $leitura?->observacoes ?? '' }}"
                                       placeholder="..."
                                       style="text-align: left;"
                                       onkeydown="handleKeyNavigation(event)">
                            </td>
                        </tr>
                    @endfor
                </tbody>
            </table>
        </div>
    </div>

    <!-- Help Panel - Compact -->
    <div class="help-panel" style="padding: 12px 16px;">
        <div style="display: flex; flex-wrap: wrap; gap: 20px; align-items: center; font-size: 12px;">
            @if($temEscalaNpa)
            <span style="font-weight: 600; color: #166534; background: #dcfce7; padding: 4px 10px; border-radius: 6px;">
                <i class="fas fa-ruler-vertical mr-1"></i>
                Digite o NÍVEL na régua → Cota, Volume e % calculados automaticamente
            </span>
            @else
            <span style="font-weight: 600; color: #1e40af; background: #dbeafe; padding: 4px 10px; border-radius: 6px;">
                <i class="fas fa-water mr-1"></i>
                Digite a COTA → Volume e % calculados automaticamente
            </span>
            @endif
            <span style="color: #6b7280;"><kbd>Tab</kbd> Próx. célula</span>
            <span style="color: #6b7280;"><kbd>Enter</kbd> Próx. linha</span>
            <span style="color: #6b7280;"><kbd>Ctrl+S</kbd> Salvar</span>
            <span style="margin-left: auto; display: flex; gap: 12px; align-items: center;">
                <span style="display: flex; align-items: center; gap: 4px;">
                    <span style="width: 12px; height: 12px; background: #22c55e; border-radius: 2px;"></span> &lt;70%
                </span>
                <span style="display: flex; align-items: center; gap: 4px;">
                    <span style="width: 12px; height: 12px; background: #f59e0b; border-radius: 2px;"></span> 70-90%
                </span>
                <span style="display: flex; align-items: center; gap: 4px;">
                    <span style="width: 12px; height: 12px; background: #ef4444; border-radius: 2px;"></span> &gt;90%
                </span>
            </span>
        </div>
    </div>
</div>

<script>
// Configuração da barragem
const barragemConfig = {
    id: {{ $barragem->id }},
    volume_maximo: {{ $barragem->volume_maximo ?? 0 }},
    volume_morto: {{ $barragem->volume_morto ?? 0 }},
    cota_maxima: {{ $barragem->cota_maxima ?? 0 }},
    cota_minima: {{ $barragem->cota_minima ?? 0 }},
    cota_npa: {{ $barragem->cota_npa ?? 0 }},
    cota_nme: {{ $barragem->cota_nme ?? 0 }},
    escala_npa: {{ $barragem->escala_npa ?? 0 }},
    temEscalaNpa: {{ ($barragem->escala_npa && $barragem->cota_npa) ? 'true' : 'false' }},
};

// Estado atual
let modoAtual = 'simples';
let linhasModificadas = new Set();

// Toggle entre modo simples e completo
function toggleMode(mode) {
    modoAtual = mode;

    // Atualizar botões
    document.querySelectorAll('.mode-toggle button').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.mode === mode);
    });

    // Mostrar/Ocultar colunas
    const elementos = document.querySelectorAll('.modo-completo');
    elementos.forEach(el => {
        el.style.display = mode === 'completo' ? '' : 'none';
    });
}

// Carregar período selecionado
function carregarPeriodo() {
    const select = document.getElementById('periodo-select');
    const valor = select.value;
    const hoje = new Date();

    let dataInicio, dataFim;

    switch(valor) {
        case 'mes-atual':
            dataInicio = new Date(hoje.getFullYear(), hoje.getMonth(), 1);
            dataFim = new Date(hoje.getFullYear(), hoje.getMonth() + 1, 0);
            break;
        case 'mes-anterior':
            dataInicio = new Date(hoje.getFullYear(), hoje.getMonth() - 1, 1);
            dataFim = new Date(hoje.getFullYear(), hoje.getMonth(), 0);
            break;
        case 'ano-hidrologico':
            // Ano hidrológico: Out a Set
            if (hoje.getMonth() >= 9) { // Out-Dez do ano atual
                dataInicio = new Date(hoje.getFullYear(), 9, 1); // 1 de Outubro
                dataFim = new Date(hoje.getFullYear() + 1, 8, 30); // 30 de Setembro próximo
            } else { // Jan-Set
                dataInicio = new Date(hoje.getFullYear() - 1, 9, 1);
                dataFim = new Date(hoje.getFullYear(), 8, 30);
            }
            break;
        case 'personalizado':
            // Usar datas dos inputs
            return;
        default:
            return;
    }

    document.getElementById('data-inicio').value = formatDate(dataInicio);
    document.getElementById('data-fim').value = formatDate(dataFim);
    carregarPeriodoPersonalizado();
}

function carregarPeriodoPersonalizado() {
    const dataInicio = document.getElementById('data-inicio').value;
    const dataFim = document.getElementById('data-fim').value;

    if (dataInicio && dataFim) {
        window.location.href = `{{ route('barragens.leituras.grid', $barragem) }}?data_inicio=${dataInicio}&data_fim=${dataFim}`;
    }
}

function formatDate(date) {
    return date.toISOString().split('T')[0];
}

// Limpar todas as linhas (remove dados)
function limparLinhasVazias() {
    if (!confirm('Tem certeza que deseja limpar TODAS as linhas? Esta ação não pode ser desfeita.')) return;

    const rows = document.querySelectorAll('#grid-body tr');
    let limpos = 0;

    rows.forEach(row => {
        if (linhaTemDados(row)) {
            // Limpar campo principal
            if (barragemConfig.temEscalaNpa) {
                const nivelInput = row.querySelector('input[name="nivel_hidrometrico[]"]');
                if (nivelInput) nivelInput.value = '';
            }
            const cotaInput = row.querySelector('input[name="cota_actual[]"]');
            if (cotaInput) cotaInput.value = '';

            // Limpar campos calculados
            row.querySelector('input[name="percentagem_enchimento[]"]').value = '';
            row.querySelector('input[name="volume_actual[]"]').value = '';
            const percTotal = row.querySelector('input[name="percentagem_total[]"]');
            if (percTotal) percTotal.value = '';
            const variacao = row.querySelector('input[name="variacao_volume[]"]');
            if (variacao) variacao.value = '';

            row.classList.remove('has-data', 'nivel-normal', 'nivel-alerta', 'nivel-critico', 'modified');
            limpos++;
        }
    });

    linhasModificadas.clear();
    atualizarContadorModificadas();
    atualizarStatus('editing', limpos > 0 ? `${limpos} linhas limpas` : 'Nada a limpar');
}

// Exportar para Excel (CSV)
function exportarExcel() {
    const rows = document.querySelectorAll('#grid-body tr');
    let csv = 'Data,Hora,Cota (m),Enchimento (%),Volume (Mm³),Precipitação (mm)\n';

    rows.forEach(row => {
        const data = row.dataset.date;
        const hora = row.querySelector('select[name="hora_leitura[]"]').value;
        const cota = row.querySelector('input[name="cota_actual[]"]').value || '';
        const enchimento = row.querySelector('input[name="percentagem_enchimento[]"]').value || '';
        const volume = row.querySelector('input[name="volume_actual[]"]').value || '';
        const precip = row.querySelector('input[name="precipitacao[]"]')?.value || '';

        csv += `${data},${hora},${cota},${enchimento},${volume},${precip}\n`;
    });

    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `leituras_{{ $barragem->codigo }}_{{ $inicio->format('Y-m') }}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Atualizar contador de linhas modificadas
function atualizarContadorModificadas() {
    document.getElementById('count-modificadas').textContent = linhasModificadas.size;
}

// Atualizar status
function atualizarStatus(tipo, texto) {
    const badge = document.getElementById('status-badge');
    const dot = badge.querySelector('.status-dot');
    const textEl = document.getElementById('status-text');

    badge.className = 'status-badge ' + tipo;
    dot.className = 'status-dot ' + tipo;
    textEl.textContent = texto;
}

// Navegação com teclado (Tab, Enter, Arrows)
function handleKeyNavigation(event) {
    const input = event.target;
    const cell = input.closest('td');
    const row = cell.closest('tr');
    const cells = Array.from(row.cells);
    const currentIndex = cells.indexOf(cell);

    let nextCell = null;

    switch(event.key) {
        case 'Enter':
            // Enter: próxima linha, mesma coluna
            event.preventDefault();
            const nextRow = row.nextElementSibling;
            if (nextRow) {
                nextCell = nextRow.cells[currentIndex];
            }
            break;

        case 'Tab':
            if (!event.shiftKey) {
                // Tab: próxima célula editável
                event.preventDefault();
                for (let i = currentIndex + 1; i < cells.length; i++) {
                    const cellInput = cells[i].querySelector('input:not([readonly])');
                    if (cellInput) {
                        nextCell = cells[i];
                        break;
                    }
                }
                // Se não achou, vai para próxima linha
                if (!nextCell && row.nextElementSibling) {
                    nextCell = row.nextElementSibling.cells[2]; // Primeira célula editável (cota)
                }
            } else {
                // Shift+Tab: célula anterior
                event.preventDefault();
                for (let i = currentIndex - 1; i >= 0; i--) {
                    const cellInput = cells[i].querySelector('input:not([readonly])');
                    if (cellInput) {
                        nextCell = cells[i];
                        break;
                    }
                }
            }
            break;
    }

    if (nextCell) {
        const nextInput = nextCell.querySelector('input');
        if (nextInput) {
            nextInput.focus();
            nextInput.select();
        }
    }
}

// Calcular valores da linha quando nivel/cota é alterado
function calcularLinha(input) {
    const row = input.closest('tr');
    let cota = 0;

    // Determinar se estamos usando nivel ou cota
    if (barragemConfig.temEscalaNpa) {
        // Usando nivel hidrometrico
        const nivel = parseFloat(input.value);
        if (isNaN(nivel) || nivel <= 0) {
            limparLinha(row);
            return;
        }
        // Formula: Cota = NPA - (escala_npa - nivel)
        cota = barragemConfig.cota_npa - (barragemConfig.escala_npa - nivel);

        // Atualizar campo de cota calculada
        const cotaInput = row.querySelector('input[name="cota_actual[]"]');
        if (cotaInput) cotaInput.value = cota.toFixed(2);
    } else {
        // Usando cota direta
        cota = parseFloat(input.value);
        if (isNaN(cota) || cota <= 0) {
            limparLinha(row);
            return;
        }
    }

    // Marcar linha como modificada
    linhasModificadas.add(row.dataset.date);
    row.classList.add('modified');
    atualizarContadorModificadas();
    atualizarStatus('editing', `${linhasModificadas.size} alteração(ões)`);

    // Calcular Volume Atual (usando curva cota-volume ou interpolação)
    const volumeAtual = calcularVolumePorCota(cota);
    row.querySelector('input[name="volume_actual[]"]').value = volumeAtual.toFixed(2);

    // Calcular Percentagem de Enchimento (% útil)
    const percEnchimento = calcularPercentagemEnchimento(volumeAtual);
    row.querySelector('input[name="percentagem_enchimento[]"]').value = percEnchimento.toFixed(1);

    // Calcular Percentagem Total
    const percTotal = calcularPercentagemTotal(volumeAtual);
    const percTotalInput = row.querySelector('input[name="percentagem_total[]"]');
    if (percTotalInput) percTotalInput.value = percTotal.toFixed(1);

    // Calcular Variação de Volume (comparar com linha anterior)
    calcularVariacaoVolume(row);

    // Atualizar classe de status baseado no enchimento
    row.classList.remove('nivel-normal', 'nivel-alerta', 'nivel-critico');
    row.classList.add('has-data');
    if (percEnchimento >= 90) {
        row.classList.add('nivel-critico');
    } else if (percEnchimento >= 70) {
        row.classList.add('nivel-alerta');
    } else {
        row.classList.add('nivel-normal');
    }

    // Efeito visual de feedback
    row.style.transition = 'background 0.3s';
    row.style.background = '#dcfce7';
    setTimeout(() => {
        row.style.background = '';
    }, 300);
}

// Limpar valores calculados da linha
function limparLinha(row) {
    row.querySelector('input[name="volume_actual[]"]').value = '';
    row.querySelector('input[name="percentagem_enchimento[]"]').value = '';
    const percTotal = row.querySelector('input[name="percentagem_total[]"]');
    if (percTotal) percTotal.value = '';
    const variacao = row.querySelector('input[name="variacao_volume[]"]');
    if (variacao) variacao.value = '';
    const cotaCalc = row.querySelector('input[name="cota_actual[]"]:not([style*="font-weight"])');
    if (cotaCalc) cotaCalc.value = '';
    row.classList.remove('has-data', 'nivel-normal', 'nivel-alerta', 'nivel-critico', 'modified');
}

// Calcular volume baseado na cota (usando interpolação linear NME-NPA)
function calcularVolumePorCota(cota) {
    // Usar NPA e NME se disponíveis, senão usar cota_maxima e cota_minima
    const cotaMax = barragemConfig.cota_npa || barragemConfig.cota_maxima;
    const cotaMin = barragemConfig.cota_nme || barragemConfig.cota_minima;
    const volMax = barragemConfig.volume_maximo;
    const volMorto = barragemConfig.volume_morto || 0;

    if (!cotaMax || !cotaMin || !volMax) {
        console.warn('Configuração incompleta da barragem para cálculo de volume');
        return 0;
    }

    // Cota abaixo do NME = volume morto
    if (cota <= cotaMin) return volMorto;

    // Cota acima do NPA = volume máximo
    if (cota >= cotaMax) return volMax;

    // Interpolação linear entre NME e NPA
    const deltaCota = cotaMax - cotaMin;
    if (deltaCota <= 0) return volMorto;

    const deltaVolume = volMax - volMorto;
    const proporcao = (cota - cotaMin) / deltaCota;

    return volMorto + (deltaVolume * proporcao);
}

// Calcular percentagem de enchimento
// FÓRMULA: % Enchimento = (Volume Útil / Volume Útil Máximo) × 100
// Onde: Volume Útil = Volume Atual - Volume Morto
//       Volume Útil Máximo = Volume Máximo - Volume Morto
function calcularPercentagemEnchimento(volumeAtual) {
    const volMax = barragemConfig.volume_maximo;
    const volMorto = barragemConfig.volume_morto || 0;

    if (!volMax || volMax <= 0) {
        return 0;
    }

    // Calcular volume útil atual
    const volumeUtil = Math.max(0, volumeAtual - volMorto);

    // Calcular volume útil máximo
    const volumeUtilMaximo = volMax - volMorto;

    if (volumeUtilMaximo <= 0) {
        // Se não há volume útil (volume_morto >= volume_maximo), usar total
        return (volumeAtual / volMax) * 100;
    }

    // Percentagem de enchimento útil
    const percentagem = (volumeUtil / volumeUtilMaximo) * 100;

    // Limitar entre 0 e 100
    return Math.max(0, Math.min(100, percentagem));
}

// Calcular percentagem total (em relação ao volume máximo total)
// FÓRMULA EXCEL: % TOTAL = (Volume Atual / Volume Máximo) × 100
function calcularPercentagemTotal(volumeAtual) {
    if (!barragemConfig.volume_maximo || barragemConfig.volume_maximo <= 0) {
        return 0;
    }

    const percentagem = (volumeAtual / barragemConfig.volume_maximo) * 100;
    return Math.max(0, Math.min(100, percentagem));
}

// Calcular variação de volume
function calcularVariacaoVolume(row) {
    const volumeAtual = parseFloat(row.querySelector('input[name="volume_actual[]"]').value) || 0;
    const rowAnterior = row.previousElementSibling;

    if (rowAnterior) {
        const volumeAnterior = parseFloat(rowAnterior.querySelector('input[name="volume_actual[]"]').value) || 0;
        const variacao = volumeAtual - volumeAnterior;
        row.querySelector('input[name="variacao_volume[]"]').value = variacao.toFixed(2);

        // Calcular descarga total baseado na variação
        calcularDescargaTotal(row, variacao);
    }
}

// Calcular descarga total (BASEADO NA FÓRMULA DO EXCEL ORIGINAL)
// FÓRMULA EXCEL: R4 = N4 / 3.25
// Onde: N4 = Variação de Volume (Mm³), 3.25 = constante de conversão
// NOTA: Esta fórmula pode estar incorreta/simplificada. Verificar com hidrólogo.
function calcularDescargaTotal(row, variacao) {
    // TODO: Confirmar se esta é a fórmula correta para descarga total
    // A fórmula original do Excel divide a variação por 3.25
    const descargaTotal = variacao / 3.25;
    row.querySelector('input[name="descarga_total[]"]').value = descargaTotal.toFixed(2);
}

// Calcular caudais quando alterados
function calcularCaudais(input) {
    const row = input.closest('tr');
    linhasModificadas.add(row.dataset.date);
}

// Colar dados do Excel
async function copiarDeExcel() {
    try {
        const text = await navigator.clipboard.readText();
        const rows = text.trim().split('\n');

        const gridRows = document.querySelectorAll('#grid-body tr');
        let rowIndex = 0;
        let linhasColadas = 0;

        for (const line of rows) {
            if (rowIndex >= gridRows.length) break;

            const values = line.split('\t').map(v => v.trim().replace(',', '.'));
            const row = gridRows[rowIndex];

            // Primeira coluna = valor principal (nivel ou cota)
            if (values[0]) {
                const valor = parseFloat(values[0]);
                if (!isNaN(valor)) {
                    if (barragemConfig.temEscalaNpa) {
                        const input = row.querySelector('input[name="nivel_hidrometrico[]"]');
                        if (input) {
                            input.value = valor;
                            calcularLinha(input);
                        }
                    } else {
                        const input = row.querySelector('input[name="cota_actual[]"]');
                        if (input) {
                            input.value = valor;
                            calcularLinha(input);
                        }
                    }
                    linhasColadas++;
                }
            }

            rowIndex++;
        }

        atualizarStatus('editing', `${linhasColadas} linhas coladas`);
    } catch (err) {
        alert('Erro ao colar. Certifique-se de copiar os dados do Excel primeiro.\n\nDica: Copie apenas a coluna com os valores de nível ou cota.');
    }
}

// Verificar se linha tem dados
function linhaTemDados(row) {
    if (barragemConfig.temEscalaNpa) {
        return !!row.querySelector('input[name="nivel_hidrometrico[]"]')?.value;
    } else {
        return !!row.querySelector('input[name="cota_actual[]"]')?.value;
    }
}

// Preencher automático (copiar última leitura ou padrão)
function preencherAutomatico() {
    const tipo = prompt('Preencher linhas vazias com:\n1 - Última leitura conhecida\n2 - Copiar primeira linha com dados', '1');

    if (!tipo) return;

    const rows = document.querySelectorAll('#grid-body tr');
    let linhasPreenchidas = 0;

    if (tipo === '1') {
        // Buscar última leitura com dados
        let ultimaLinhaComDados = null;
        for (let i = rows.length - 1; i >= 0; i--) {
            if (linhaTemDados(rows[i])) {
                ultimaLinhaComDados = rows[i];
                break;
            }
        }

        if (!ultimaLinhaComDados) {
            alert('Nenhuma leitura com dados encontrada');
            return;
        }

        // Copiar para linhas vazias
        rows.forEach(row => {
            if (!linhaTemDados(row)) {
                copiarLinha(ultimaLinhaComDados, row);
                linhasPreenchidas++;
            }
        });
    } else if (tipo === '2') {
        // Buscar primeira linha com dados
        let primeiraLinhaComDados = null;
        for (let i = 0; i < rows.length; i++) {
            if (linhaTemDados(rows[i])) {
                primeiraLinhaComDados = rows[i];
                break;
            }
        }

        if (!primeiraLinhaComDados) {
            alert('Nenhuma leitura com dados encontrada');
            return;
        }

        rows.forEach(row => {
            if (!linhaTemDados(row)) {
                copiarLinha(primeiraLinhaComDados, row);
                linhasPreenchidas++;
            }
        });
    }

    if (linhasPreenchidas > 0) {
        atualizarStatus('editing', `${linhasPreenchidas} linhas preenchidas`);
    }
}

function copiarLinha(origem, destino) {
    // Determinar campo principal baseado na configuração
    if (barragemConfig.temEscalaNpa) {
        const valorOrigem = origem.querySelector('input[name="nivel_hidrometrico[]"]')?.value;
        const inputDestino = destino.querySelector('input[name="nivel_hidrometrico[]"]');
        if (inputDestino && valorOrigem) {
            inputDestino.value = valorOrigem;
            calcularLinha(inputDestino);
        }
    } else {
        const valorOrigem = origem.querySelector('input[name="cota_actual[]"]')?.value;
        const inputDestino = destino.querySelector('input[name="cota_actual[]"]');
        if (inputDestino && valorOrigem) {
            inputDestino.value = valorOrigem;
            calcularLinha(inputDestino);
        }
    }

    // Copiar campos opcionais
    const camposOpcionais = ['caudal_afluente', 'caudal_efluente', 'precipitacao'];
    camposOpcionais.forEach(campo => {
        const inputOrigem = origem.querySelector(`input[name="${campo}[]"]`);
        const inputDestino = destino.querySelector(`input[name="${campo}[]"]`);
        if (inputOrigem && inputDestino) {
            inputDestino.value = inputOrigem.value;
        }
    });
}

// Salvar todas as leituras
async function salvarTodasLeituras() {
    if (linhasModificadas.size === 0) {
        atualizarStatus('saved', 'Nada a salvar');
        setTimeout(() => atualizarStatus('editing', 'Pronto'), 2000);
        return;
    }

    if (!confirm(`Salvar ${linhasModificadas.size} leitura(s)?`)) return;

    const btnSalvar = document.getElementById('btn-salvar');
    btnSalvar.disabled = true;
    btnSalvar.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Salvando...';
    atualizarStatus('editing', 'Salvando...');

    const rows = document.querySelectorAll('#grid-body tr');
    const leituras = [];

    rows.forEach(row => {
        const data = row.dataset.date;
        if (!linhasModificadas.has(data)) return;

        // Verificar se tem dados (nivel ou cota)
        const nivel = row.querySelector('input[name="nivel_hidrometrico[]"]')?.value;
        const cota = row.querySelector('input[name="cota_actual[]"]')?.value;
        if (!nivel && !cota) return; // Pular linhas vazias

        // Obter a data original do atributo data-date do input
        const dataInput = row.querySelector('input.date-input');
        const dataReal = dataInput ? dataInput.dataset.date : data;

        leituras.push({
            id: row.dataset.id || null,
            data_leitura: dataReal,
            hora_leitura: row.querySelector('select[name="hora_leitura[]"]').value,
            nivel_hidrometrico: row.querySelector('input[name="nivel_hidrometrico[]"]')?.value || null,
            cota_actual: row.querySelector('input[name="cota_actual[]"]')?.value || cota,
            volume_actual: row.querySelector('input[name="volume_actual[]"]').value,
            percentagem_enchimento: row.querySelector('input[name="percentagem_enchimento[]"]').value,
            percentagem_total: row.querySelector('input[name="percentagem_total[]"]')?.value || null,
            variacao_volume: row.querySelector('input[name="variacao_volume[]"]')?.value || null,
            caudal_afluente: row.querySelector('input[name="caudal_afluente[]"]')?.value || null,
            caudal_efluente: row.querySelector('input[name="caudal_efluente[]"]')?.value || null,
            precipitacao: row.querySelector('input[name="precipitacao[]"]')?.value || null,
            observacoes: row.querySelector('input[name="observacoes[]"]')?.value || null,
        });
    });

    // Enviar para backend
    try {
        const response = await fetch('{{ route("barragens.storeLeiturasBulk", $barragem) }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ leituras })
        });

        const result = await response.json();

        if (response.ok) {
            // Limpar marcação de modificadas
            linhasModificadas.clear();
            document.querySelectorAll('#grid-body tr.modified').forEach(row => {
                row.classList.remove('modified');
            });

            atualizarContadorModificadas();
            atualizarStatus('saved', `${result.saved + result.updated} salvas!`);

            setTimeout(() => {
                atualizarStatus('editing', 'Pronto');
            }, 3000);
        } else {
            atualizarStatus('error', 'Erro ao salvar');
            alert('Erro ao salvar: ' + (result.message || 'Erro desconhecido'));
        }
    } catch (error) {
        console.error('Erro:', error);
        atualizarStatus('error', 'Erro de conexão');
        alert('Erro ao salvar leituras. Verifique a conexão.');
    } finally {
        btnSalvar.disabled = false;
        btnSalvar.innerHTML = '<i class="fas fa-save"></i> Salvar Alterações';
    }
}

// Inicializar
document.addEventListener('DOMContentLoaded', function() {
    // Iniciar em modo simples
    toggleMode('simples');

    // Inicializar contador
    atualizarContadorModificadas();

    // Auto-scroll e auto-focus na linha de hoje
    const linhaHoje = document.getElementById('linha-hoje');
    if (linhaHoje) {
        // Scroll para a linha de hoje com margem superior
        setTimeout(() => {
            linhaHoje.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }, 100);

        // Auto-focus no primeiro input editável vazio da linha de hoje
        setTimeout(() => {
            const inputs = linhaHoje.querySelectorAll('input:not([readonly]):not(.date-input), select');
            for (let input of inputs) {
                if (!input.value || input.value === '' || input.value === '--') {
                    input.focus();
                    if (input.select) input.select();
                    break;
                }
            }
            // Se todos têm valor, focar no campo principal (nivel ou cota)
            if (inputs.length > 0 && !document.activeElement.closest('#linha-hoje')) {
                const campoMain = linhaHoje.querySelector('input[name="nivel_hidrometrico[]"], input[name="cota_actual[]"]:not([readonly])');
                if (campoMain) {
                    campoMain.focus();
                    campoMain.select();
                }
            }
        }, 200);
    }

    // Atalhos de teclado
    document.addEventListener('keydown', function(e) {
        // Ctrl+S para salvar
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            salvarTodasLeituras();
        }

        // Ctrl+V para colar (se não estiver em input)
        if (e.ctrlKey && e.key === 'v' && !document.activeElement.matches('input')) {
            e.preventDefault();
            copiarDeExcel();
        }
    });

    // Avisar antes de sair se houver alterações não salvas
    window.addEventListener('beforeunload', function(e) {
        if (linhasModificadas.size > 0) {
            e.preventDefault();
            e.returnValue = 'Existem alterações não salvas. Deseja sair?';
            return e.returnValue;
        }
    });

    // Mostrar data completa no tooltip ao passar mouse
    document.querySelectorAll('.date-input').forEach(input => {
        input.title = input.dataset.date ? new Date(input.dataset.date + 'T00:00:00').toLocaleDateString('pt-BR', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        }) : '';
    });
});
</script>
@endsection
